#Requires -Version 5.1
<#
.SYNOPSIS
    SRM Update Script for Windows
.DESCRIPTION
    Automated update script for Smart Report Maker (SRM) on Windows systems.
    Handles version migration, backup creation, dependency updates, and rollback.
.PARAMETER ResumeAfterComposer
    Resume all steps After Composer Dependency Installation
.PARAMETER RetryMigrations
    Retry update migrations
.PARAMETER Rollback
    Roll back to previous version
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\update.ps1
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\update.ps1 -ResumeAfterComposer
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\update.ps1 -RetryMigrations
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\update.ps1 -Rollback
#>

param(
    [switch]$ResumeAfterComposer = $false,
    [switch]$RetryMigrations = $false,
    [switch]$Rollback = $false
)


# ============================================================================
# GLOBAL VARIABLES AND CONSTANTS
# ============================================================================

$ErrorActionPreference = "Stop"
$global:ResumeAfterComposer = $ResumeAfterComposer
$global:RetryMigrations = $RetryMigrations
$global:Rollback = $Rollback

$global:ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$global:SrmNewDir = Join-Path $global:ScriptDir "srm"
$global:SrmOldDir = Join-Path $global:ScriptDir "srm_old"
$global:BackupDir = Join-Path $global:ScriptDir "existed_resources"
$global:ProgressFile = Join-Path $global:ScriptDir ".progress"
$global:LogFile = Join-Path $global:ScriptDir "update.log"
$global:RequiredExtensions = @("zip", "curl")

$global:PhpPath = ""
$global:CurrentStep = ""
$global:Warnings = @()

# ============================================================================
# LOGGING AND USER FEEDBACK FUNCTIONS
# ============================================================================

function Write-LogInfo {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [INFO] $Message"
    Write-Host $logMessage -ForegroundColor Green
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
}

function Write-LogWarning {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [WARNING] $Message"
    Write-Host $logMessage -ForegroundColor Yellow
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
    $global:Warnings += $Message
}

function Write-LogError {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [ERROR] $Message"
    Write-Host $logMessage -ForegroundColor Red
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
}

function Show-Progress {
    param(
        [int]$ProcessId,
        [int]$IntervalMs = 500
    )
    
    while (Get-Process -Id $ProcessId -ErrorAction SilentlyContinue) {
        Write-Host -NoNewline "."
        Start-Sleep -Milliseconds $IntervalMs
    }
    Write-Host ""
}

function Write-StepHeader {
    param([string]$StepName)
    $global:CurrentStep = $StepName
    Write-Host ""
    Write-Host "===================" -ForegroundColor Cyan
    Write-Host " $StepName" -ForegroundColor Cyan
    Write-Host "===================" -ForegroundColor Cyan
    Write-LogInfo "Starting step: $StepName"
}

function Update-ProgressState {
    param([string]$State)
    try {
        # Read existing content and filter out migration_ lines
        $existingLines = @()
        if (Test-Path $global:ProgressFile) {
            $existingLines = Get-Content -Path $global:ProgressFile -ErrorAction SilentlyContinue | Where-Object { $_ -notmatch '^migration_' }
        }
        
        # Combine existing lines with new state, ensuring proper line breaks
        $newContent = @()
        if ($existingLines) {
            $newContent += $existingLines
        }
        $newContent += $State
        
        # Write content with explicit line breaks
        $newContent | Set-Content -Path $global:ProgressFile -Encoding UTF8
        Write-LogInfo "Progress state updated: $State"
    }
    catch {
        Write-LogWarning "Failed to update progress state: $($_.Exception.Message)"
    }
}

# ============================================================================
# ERROR HANDLING FUNCTIONS
# ============================================================================

function Invoke-ErrorHandler {
    param([string]$ErrorMessage)
    Write-LogError "Update failed at step: $global:CurrentStep"
    Write-LogError "Error: $ErrorMessage"
    
    Write-Host ""
    Write-Host "Update Failed!" -ForegroundColor Red
    Write-Host "Step: $global:CurrentStep" -ForegroundColor Red
    Write-Host "Error: $ErrorMessage" -ForegroundColor Red
    Write-Host ""
    Write-Host "Check the log file at: $global:LogFile" -ForegroundColor Yellow
    
    Stop-Transcript -ErrorAction SilentlyContinue
    exit 1
}

function Get-PathFromUser {
    param(
        [string]$PromptMessage,
        [string]$PathDescription
    )
    
    while ($true) {
        Write-Host ""
        Write-Host $PromptMessage -ForegroundColor Yellow
        $pathValue = Read-Host "Path"
        
        if (-not $pathValue) {
            Write-Host "Path cannot be empty. Please try again." -ForegroundColor Red
            continue
        }
        
        if (Test-Path -Path $pathValue -PathType Container) {
            Write-LogInfo "$PathDescription path verified: $pathValue"
            return $pathValue
        }
        else {
            Write-Host "Invalid directory path. Please try again." -ForegroundColor Red
        }
    }
}

# ============================================================================
# VALIDATION FUNCTIONS
# ============================================================================

function Test-Prerequisites {
    Write-StepHeader "Prerequisite Validation"
    
    # Check directory structure
    $missingDirs = @()
    if (-not (Test-Path $global:SrmOldDir -PathType Container)) { $missingDirs += "srm_old" }
    if (-not (Test-Path $global:SrmNewDir -PathType Container)) { $missingDirs += "srm" }
    
    if ($missingDirs.Count -eq 2) {
        throw "Neither 'srm' nor 'srm_old' directories found. Please rename your old SRM folder to 'srm_old' and place the new 'srm' folder alongside this script."
    }
    elseif ($missingDirs -contains "srm_old") {
        throw "The 'srm_old' directory was not found. Please rename your existing 'srm' directory to 'srm_old' and rerun this script."
    }
    elseif ($missingDirs -contains "srm") {
        throw "The 'srm' directory for the new version was not found. Ensure both 'srm_old' and 'srm' directories exist alongside this script."
    }
    
    Write-LogInfo "Directory structure validation passed"
}

function Get-SrmVersion {
    param([string]$ConfigPath)
    
    if (-not (Test-Path $ConfigPath -PathType Leaf)) {
        throw "Version configuration file not found: $ConfigPath"
    }
    
    try {
        $content = Get-Content -Path $ConfigPath -Raw
        if ($content -match '"version_to_install"\s*=>\s*"([^"]+)"') {
            return $matches[1]
        }
        else {
            throw "Version information not found in configuration file"
        }
    }
    catch {
        throw "Failed to read version from $ConfigPath : $($_.Exception.Message)"
    }
}

function Test-VersionDifference {
    Write-StepHeader "Version Comparison"
    
    $oldConfigPath = Join-Path $global:SrmOldDir "dashboard\config\srm_config.php"
    $newConfigPath = Join-Path $global:SrmNewDir "dashboard\config\srm_config.php"
    
    $oldVersion = Get-SrmVersion -ConfigPath $oldConfigPath
    $newVersion = Get-SrmVersion -ConfigPath $newConfigPath
    
    Write-LogInfo "Old version: $oldVersion"
    Write-LogInfo "New version: $newVersion"
    
    if ($oldVersion -eq $newVersion -and -not $Force) {
        Write-Host "No version difference detected between 'srm_old' and 'srm'; both are from the same release." -ForegroundColor Yellow
        return $false
    }
    
    # Initialize progress file
    "$oldVersion -> $newVersion" | Out-File -FilePath $global:ProgressFile -Encoding utf8
    return $true
}

# ============================================================================
# PHP DETECTION FUNCTIONS (Reused from install.ps1)
# ============================================================================

function Find-PhpInDirectories {
    $searchDirs = @(
        "C:\xampp\php",
        "D:\xampp\php",
        "C:\wamp64\bin\php",
        "D:\wamp64\bin\php", 
        "C:\php",
        "D:\php",
        "C:\Program Files\PHP",
        "C:\Program Files (x86)\PHP",
        "D:\Program Files\PHP",
        "D:\Program Files (x86)\PHP"
    )

    Write-LogInfo "Searching for PHP in common directories..."

    foreach ($dir in $searchDirs) {
        Write-LogInfo "Checking directory: $dir"
        
        if (-not (Test-Path $dir)) { 
            Write-LogInfo "Directory does not exist: $dir"
            continue 
        }
        
        Write-LogInfo "Directory exists, searching for php.exe..."
        $phpExe = Get-ChildItem -Path $dir -Filter "php.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
        
        if ($phpExe) {
            $phpPath = Join-Path $dir $phpExe.Name
            Write-LogInfo "Found php.exe, testing: $phpPath"
            
            Write-LogInfo "PHP found at: $phpPath"
            return $phpPath
        }
        else {
            Write-LogInfo "No php.exe found in: $dir"
            # Also check for common subdirectories
            $subDirs = @("", "bin", "cli")
            foreach ($subDir in $subDirs) {
                $subPath = if ($subDir) { Join-Path $dir $subDir } else { $dir }
                if (Test-Path $subPath) {
                    $phpInSub = Get-ChildItem -Path $subPath -Filter "php.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
                    if ($phpInSub) {
                        $phpPath = Join-Path $subPath $phpInSub.Name
                        Write-LogInfo "Found php.exe in subdirectory, testing: $phpPath"
                        Write-LogInfo "PHP found at: $phpPath"
                        return $phpPath
                    }
                }
            }
        }
    }
    
    Write-LogInfo "PHP not found in any common directories"
    return $null
}


function Get-PhpFromUser {
    while ($true) {
        Write-Host ""
        Write-Host "Please provide the full path to your PHP executable (php.exe):" -ForegroundColor Yellow
        Write-Host "Example: C:\xampp\php\php.exe" -ForegroundColor Gray
        
        $phpPath = Read-Host "PHP Path"
        
        if (-not $phpPath) {
            Write-Host "Path cannot be empty. Please try again." -ForegroundColor Red
            continue
        }
        
        if (-not (Test-Path $phpPath)) {
            Write-Host "File not found: $phpPath" -ForegroundColor Red
            continue
        }
        
        Write-LogInfo "PHP verified at user-provided path: $phpPath"
        return $phpPath
    }
}
function Initialize-PhpEnvironment {
    Write-StepHeader "PHP Environment Setup"
    
    # Check if PHP is in PATH
    try {
        $output = & "php" -v 2>$null
        if ($output -and $output.ToString().Contains("PHP")) {
            Write-LogInfo "PHP is accessible via PATH"
            $global:PhpPath = "php"
            return
        }
    }
    catch {
        Write-LogInfo "PHP not found in PATH, searching directories..."
    }
    
    # Search in common directories
    $phpPath = Find-PhpInDirectories
    
    if (-not $phpPath) {
        $phpPath = Get-PhpFromUser
    }
    
    if (-not $phpPath) {
        throw "Unable to locate PHP executable"
    }
    
    $global:PhpPath = $phpPath
    Write-LogInfo "PHP environment initialized with: $global:PhpPath"
}

function Check-PHPExtensions {
    param (
        [string[]]$RequiredExtensions
    )

    Write-StepHeader "Check PHP Extensions Step"

    Write-LogInfo "Checking required PHP extensions..." -ForegroundColor Cyan

    # Get active php.ini file path
    $phpIniOutput = & $global:PhpPath --ini 2>&1
    $phpIniPath = ($phpIniOutput | Select-String 'Loaded Configuration File') -replace '.*:\s*', ''

    if (-not $phpIniPath) {
        Write-Error "Unable to detect php.ini path. Is PHP installed and in PATH?" -ForegroundColor Red
        exit 1
    }

    Write-Host "`nActive php.ini file: $phpIniPath`n"

    while ($true) {
        $loadedExtensions = & $global:PhpPath -m 2>&1
        $missingExtensions = @()

        foreach ($extension in $RequiredExtensions) {
            if (-not ($loadedExtensions -match "^$extension$")) {
                $missingExtensions += $extension
            }
        }

        if ($missingExtensions.Count -eq 0) {
            Write-LogInfo "All required PHP extensions are available.`n" -ForegroundColor Green
            break
        }

        Write-LogError "Missing PHP extensions: $($missingExtensions -join ', ')" -ForegroundColor Red
        Write-Host ""
        Write-Host "Please do the following steps:"
        Write-Host "1. Open the php.ini file:"
        Write-Host "   $phpIniPath"
        Write-Host "2. Enable the following missing extensions by adding or uncommenting lines like:"
        foreach ($ext in $missingExtensions) {
            Write-Host "   extension=$ext"
        }
        Write-Host "3. Save the file and restart your web server (e.g., Apache, Nginx, or PHP-FPM)."
        Write-Host ""
        Read-Host "After doing that, press Enter to recheck..."
    }
}


# ============================================================================
# BACKUP FUNCTIONS
# ============================================================================

function New-BackupDirectory {
    if (Test-Path $global:BackupDir) {
        Write-LogInfo "Backup directory already exists: $global:BackupDir"
        return
    }
    
    try {
        New-Item -ItemType Directory -Path $global:BackupDir | Out-Null
        Write-LogInfo "Created backup directory: $global:BackupDir"
    }
    catch {
        throw "Failed to create backup directory: $($_.Exception.Message)"
    }
}

function Backup-Configuration {
    Write-LogInfo "Backing up configuration files..."
    
    $configSource = Join-Path $global:SrmOldDir "dashboard\config"
    $configBackup = Join-Path $global:BackupDir "config"
    
    if (-not (Test-Path $configSource -PathType Container)) {
        $configSource = Get-PathFromUser -PromptMessage "Configuration directory not found at expected location. Please enter the correct path:" -PathDescription "Configuration"
    }
    
    if (Test-Path $configBackup) {
        Write-LogInfo "Configuration already backed up, skipping..."
        return $configSource
    }
    
    try {
        Copy-Item -Path $configSource -Destination $configBackup -Recurse -Force
        Write-LogInfo "Configuration backed up from $configSource to $configBackup"
        return $configSource
    }
    catch {
        throw "Failed to backup configuration: $($_.Exception.Message)"
    }
}

function Backup-Reports {
    Write-LogInfo "Backing up reports..."
    
    $reportsSource = Join-Path $global:SrmOldDir "modules\SRM9\SRM\Reports9"
    $reportsBackup = Join-Path $global:BackupDir "Reports9"
    
    if (-not (Test-Path $reportsSource -PathType Container)) {
        $reportsSource = Get-PathFromUser -PromptMessage "Reports directory not found at expected location. Please enter the correct path:" -PathDescription "Reports"
    }
    
    if (Test-Path $reportsBackup) {
        Write-LogInfo "Reports already backed up, skipping..."
        return $reportsSource
    }
    
    try {
        Copy-Item -Path $reportsSource -Destination $reportsBackup -Recurse -Force
        Write-LogInfo "Reports backed up from $reportsSource to $reportsBackup"
        
        # Remove shared folder from backup
        $sharedFolder = Join-Path $reportsBackup "shared"
        if (Test-Path $sharedFolder) {
            Remove-Item -Path $sharedFolder -Recurse -Force
            Write-LogInfo "Removed 'shared' folder from Reports9 backup"
        }
        
        return $reportsSource
    }
    catch {
        throw "Failed to backup reports: $($_.Exception.Message)"
    }
}

function Backup-AppStorage {
    Write-LogInfo "Backing up App storage..."
    
    $AppStorageSource = Join-Path $global:SrmOldDir "dashboard\storage\app\public"
    $AppStorageBackup = Join-Path $global:BackupDir "public"
    
    if (-not (Test-Path $AppStorageSource -PathType Container)) {
        $AppStorageSource = Get-PathFromUser -PromptMessage "App Storage directory not found at expected location. Please enter the correct path:" -PathDescription "Reports"
    }
    
    if (Test-Path $AppStorageBackup) {
        Write-LogInfo "App storage already backed up, skipping..."
        return $AppStorageSource
    }
    
    try {
        Copy-Item -Path $AppStorageSource -Destination $AppStorageBackup -Recurse -Force
        Write-LogInfo "App storage backed up from $AppStorageSource to $AppStorageBackup"
        return $AppStorageSource
    }
    catch {
        throw "Failed to backup app storage: $($_.Exception.Message)"
    }
}

function Invoke-BackupProcess {
    if ($SkipBackup) {
        Write-LogWarning "Skipping backup process as requested"
        return @{
            ConfigSource = Join-Path $global:SrmOldDir "dashboard\config"
            ReportsSource = Join-Path $global:SrmOldDir "modules\SRM9\SRM\Reports9"
            AppStorageSource = Join-Path $global:SrmOldDir "dashboard\storage\app"
        }
    }
    
    Write-StepHeader "Backup Creation"
    
    New-BackupDirectory
    $configSource = Backup-Configuration
    $reportsSource = Backup-Reports
    $appStorageSource = Backup-AppStorage
    
    return @{
        ConfigSource = $configSource
        ReportsSource = $reportsSource
        AppStorageSource = $appStorageSource

    }
}

# ============================================================================
# MIGRATION FUNCTIONS
# ============================================================================

function Copy-ConfigurationToNew {
    param([string]$ConfigSource)
    
    Write-LogInfo "Copying configuration to new version..."
    
    $newConfigDir = Join-Path $global:SrmNewDir "dashboard\config"
    
    try {
        Copy-Item -Path (Join-Path $ConfigSource "srm_db_config.php") -Destination $newConfigDir -Force
        Copy-Item -Path (Join-Path $ConfigSource "srm_install_config.php") -Destination $newConfigDir -Force
        Write-LogInfo "Configuration files copied successfully"
    }
    catch {
        throw "Failed to copy configuration files: $($_.Exception.Message)"
    }
}

function Copy-ReportsToNew {
    param([string]$ReportsSource)
    
    Write-LogInfo "Copying reports to new version..."
    
    $newReportsDir = Join-Path $global:SrmNewDir "modules\SRM9\SRM\Reports9"
    
    try {
        # Ensure destination directory exists
        if (-not (Test-Path $newReportsDir)) {
            New-Item -ItemType Directory -Path $newReportsDir -Force | Out-Null
        }
        
        # Copy only directories inside Reports9, excluding 'shared' folder
        $sourceItems = Get-ChildItem -Path $ReportsSource -Directory
        foreach ($item in $sourceItems) {
            if ($item.Name -ne "shared") {
                $destinationPath = Join-Path $newReportsDir $item.Name
                Copy-Item -Path $item.FullName -Destination $destinationPath -Recurse -Force
                Write-LogInfo "Copied report directory: $($item.Name)"
            }
        }
        
        Write-LogInfo "Reports directories copied successfully (excluding 'shared')"
    }
    catch {
        throw "Failed to copy reports: $($_.Exception.Message)"
    }
}

function Copy-AppStorageToNew {
    param([string]$AppStorageSource)
    
    Write-LogInfo "Copying app storage to new version..."
    
    $newAppStorageDir = Join-Path $global:SrmNewDir "dashboard\storage\app"
    
    try {
        Copy-Item -Path $AppStorageSource -Destination $newAppStorageDir -Recurse -Force
        Write-LogInfo "App storage copied successfully"
    }
    catch {
        throw "Failed to copy reports: $($_.Exception.Message)"
    }
}

# ============================================================================
# COMPOSER FUNCTIONS
# ============================================================================

function Test-GitAvailability {
    try {
        $null = & "git" --version 2>$null
        return $true
    }
    catch {
        return $false
    }
}

function Test-ZipExtension {
    try {
        $modules = & $global:PhpPath -m 2>$null
        return $modules -contains "zip"
    }
    catch {
        return $false
    }
}

function Test-Dependencies {
    Write-StepHeader "Dependency Verification"
    
    if (Test-GitAvailability) {
        Write-LogInfo "Git is available"
        return
    }
    
    if (Test-ZipExtension) {
        Write-LogInfo "PHP zip extension is available (Git not required)"
        return
    }
    
    Write-LogError "Neither Git nor PHP zip extension are available"
    Write-Host ""
    Write-Host "DEPENDENCY REQUIREMENT NOT MET" -ForegroundColor Red
    Write-Host ""
    Write-Host "Please install Git from: https://git-scm.com/downloads/win" -ForegroundColor Yellow
    Write-Host "OR enable the zip extension in your PHP configuration." -ForegroundColor Yellow
    throw "Required dependencies not available"
}

function Invoke-ComposerCommand {
    param(
        [string]$Command,
        [string]$Description,
        [string[]]$Arguments = @()
    )
    
    Write-LogInfo $Description
    
    $composerPath = Join-Path $global:SrmNewDir "composer"
    
    # Define command-specific arguments
    $baseArgs = @($composerPath, $Command) + $Arguments + @("--no-interaction")
    
    switch ($Command) {
        "install" {
            $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
        }
        "update" {
            $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
        }
        "dump-autoload" {
            # dump-autoload doesn't support --prefer-dist, --no-dev, etc.
            $allArgs = $baseArgs + @("--optimize")
        }
        default {
            $allArgs = $baseArgs
        }
    }
    
    try {
        $process = Start-Process -FilePath $global:PhpPath -ArgumentList $allArgs -NoNewWindow -PassThru -RedirectStandardError "stderr.log" -RedirectStandardOutput "stdout.log"
        Show-Progress -ProcessId $process.Id
        $process | Wait-Process
        
        $stdout = Get-Content -Path "stdout.log" -Raw -ErrorAction SilentlyContinue
        $stderr = Get-Content -Path "stderr.log" -Raw -ErrorAction SilentlyContinue
        
        # Check for actual errors vs warnings/info
        if ($stderr -and $stderr.Trim()) {
            Write-LogInfo "Composer output: $($stderr.Trim())"
        }
        if ($stdout -and $stdout.Trim()) {
            Write-LogInfo "Composer output: $($stdout.Trim())"
        }
    }
    finally {
        Remove-Item -Path "stderr.log", "stdout.log" -ErrorAction SilentlyContinue
    }
}

function Update-ComposerDependencies {
    Write-StepHeader "Composer Dependency Installation"
    
    $originalLocation = Get-Location
    try {
        Set-Location $global:SrmNewDir
        
        # Check if composer.json exists
        if (-not (Test-Path "composer.json")) {
            Write-LogWarning "composer.json not found in $global:SrmNewDir - skipping Composer operations"
            return
        }
        
        # Function to execute composer command and check for errors
        function Invoke-ComposerCommandSafe {
            param(
                [string]$Command,
                [string]$Description,
                [string[]]$Arguments = @()
            )
            
            Write-LogInfo $Description
            
            $composerPath = Join-Path $global:SrmNewDir "composer"
            
            # Define command-specific arguments
            $baseArgs = @($composerPath, $Command) + $Arguments + @("--no-interaction")
            
            switch ($Command) {
                "install" {
                    $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
                }
                "update" {
                    $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
                }
                "dump-autoload" {
                    # dump-autoload doesn't support --prefer-dist, --no-dev, etc.
                    $allArgs = $baseArgs + @("--optimize")
                }
                default {
                    $allArgs = $baseArgs
                }
            }
            
            try {
                $process = Start-Process -FilePath $global:PhpPath -ArgumentList $allArgs -NoNewWindow -PassThru -RedirectStandardError "stderr.log" -RedirectStandardOutput "stdout.log"
                Show-Progress -ProcessId $process.Id
                $process | Wait-Process
                
                $stdout = Get-Content -Path "stdout.log" -Raw -ErrorAction SilentlyContinue
                $stderr = Get-Content -Path "stderr.log" -Raw -ErrorAction SilentlyContinue
                
                # Combine stdout and stderr for error checking
                $combinedOutput = "$stdout`n$stderr"

                # Check for common error indicators in the output
                $errorKeywords = @(
                    "(?i)error\s",
                    "(?i)failed\s",
                    "(?i)exception\s",
                    "(?i)fatal\s",
                    "(?i)could not\s",
                    "(?i)cannot\s",
                    "(?i)unable to\s",
                    "(?i)connection timed out\s",
                    "(?i)network error\s",
                    "(?i)curl error\s",
                    "(?i)ssl error\s",
                    "(?i)certificate\s",
                    "(?i)permission denied\s",
                    "(?i)access denied\s",
                    "(?i)file not found\s",
                    "(?i)directory not found\s",
                    "(?i)out of memory\s",
                    "(?i)killed\s",
                    "(?i)aborted\s",
                    "(?i)interrupted\s"
                )
                
                $hasError = $false
                foreach ($keyword in $errorKeywords) {
                    if ($combinedOutput -match $keyword) {
                        $hasError = $true
                        break
                    }
                }

                
                if ($hasError) {
                    Write-LogError "Composer command failed: $Command"
                    if ($stdout -and $stdout.Trim()) {
                        Write-LogError "Stdout: $($stdout.Trim())"
                    }
                    if ($stderr -and $stderr.Trim()) {
                        Write-LogError "Stderr: $($stderr.Trim())"
                    }
                    throw "Composer has stopped unexpectedly. This may be due to your antivirus or firewall interfering, or an unstable internet connection. Please try temporarily disabling your antivirus/firewall or checking your internet stability. You can also run the updater script again by repeating the previous command"
                }
                
                # Log output for successful operations
                if ($stderr -and $stderr.Trim()) {
                    Write-LogInfo "Composer output: $($stderr.Trim())"
                }
                if ($stdout -and $stdout.Trim()) {
                    Write-LogInfo "Composer output: $($stdout.Trim())"
                }
                
                Write-LogInfo "Command completed successfully"
                return $combinedOutput
            }
            finally {
                Remove-Item -Path "stderr.log", "stdout.log" -ErrorAction SilentlyContinue
            }
        }
        
        # Install dependencies first
        Invoke-ComposerCommandSafe -Command "install" -Description "Installing Composer dependencies..."
        
        # Only update if install was successful and update is needed
        try {
            Invoke-ComposerCommandSafe -Command "update" -Description "Updating Composer dependencies..."
        }
        catch {
            Write-LogWarning "Composer update failed, but install was successful. Continuing..."
            # Check if this is the specific error we want to handle
            if ($_.Exception.Message -match "Composer has stopped unexpectedly") {
                throw $_.Exception.Message
            }
        }
        
        # Generate optimized autoloader
        Invoke-ComposerCommandSafe -Command "dump-autoload" -Description "Generating optimized autoloader..."
        
        Write-LogInfo "Composer operations completed successfully"
    }
    catch {
        if ($_.Exception.Message -match "Composer has stopped unexpectedly") {
            Write-LogError $_.Exception.Message
            exit 1
        } else {
            Write-LogError "Composer operations failed: $($_.Exception.Message)"
            # Don't throw here - allow the update to continue even if Composer fails
            Write-LogWarning "Continuing update process despite Composer issues..."
        }
    }
    finally {
        Set-Location $originalLocation
    }
}

# ============================================================================
# Junctions FUNCTIONS
# ============================================================================

function New-SafeJunction {
    param(
        [string]$Source,
        [string]$Destination
    )
    
    if (-not (Test-Path $Source)) {
        Write-LogWarning "Source path does not exist: $Source"
        return $false
    }

    if (-not (Get-Item $Source).PSIsContainer) {
        Write-LogWarning "Junctions can only be created for directories. Provided source is not a directory: $Source"
        return $false
    }

    # Remove existing link/directory if it exists
    if (Test-Path $Destination) {
        $item = Get-Item $Destination -Force
        if ($item.Attributes -band [System.IO.FileAttributes]::ReparsePoint) {
            Write-LogInfo "Removing existing junction/symlink: $Destination"
            Remove-Item -Path $Destination -Force
        }
        else {
            Write-LogInfo "Removing existing directory/file: $Destination"
            Remove-Item -Path $Destination -Recurse -Force
        }
    }

    try {
        New-Item -ItemType Junction -Path $Destination -Target $Source | Out-Null
        Write-LogInfo "Created NTFS junction: $Destination -> $Source"
        return $true
    }
    catch {
        Write-LogWarning "Failed to create junction: $Destination -> $Source. Error: $($_.Exception.Message)"
        return $false
    }
}

function Initialize-Junctions {
    Write-StepHeader "Junction Setup"
    
    $links = @(
        @{
            Source = Join-Path $global:SrmNewDir "dashboard\storage\app\public"
            Destination = Join-Path $global:SrmNewDir "dashboard\public\srm_storage"
        },
        @{
            Source = Join-Path $global:SrmNewDir "srm_install"
            Destination = Join-Path $global:SrmNewDir "dashboard\public\srm_install"
        },
        @{
            Source = Join-Path $global:SrmNewDir "modules"
            Destination = Join-Path $global:SrmNewDir "dashboard\public\srm_modules"
        }
    )
    
    foreach ($link in $links) {
        New-SafeJunction -Source $link.Source -Destination $link.Destination
    }

    $linkPaths = $links | ForEach-Object { $_.Destination }
    if (-not (Test-Junctions -Links $linkPaths)) {
        throw "Junction verification failed"
    }
}

function Test-Junctions {
    param([string[]]$Links)
    
    Write-LogInfo "Verifying junctions..."
    $brokenLinks = @()
    
    foreach ($link in $Links) {
        if (-not (Test-Path $link)) {
            Write-LogError "Missing junction: $link"
            $brokenLinks += $link
        }
        else {
            $item = Get-Item $link -Force
            if (-not ($item.Attributes -band [System.IO.FileAttributes]::ReparsePoint)) {
                Write-LogError "Path is not a junction: $link"
                $brokenLinks += $link
            }
            else {
                # Additional check to ensure it's specifically a junction (not a symbolic link)
                try {
                    $reparsePoint = $item.LinkType
                    if ($reparsePoint -ne "Junction") {
                        Write-LogError "Path is a reparse point but not a junction: $link (Type: $reparsePoint)"
                        $brokenLinks += $link
                    }
                }
                catch {
                    # If LinkType property is not available, assume it's valid if it's a ReparsePoint
                    Write-LogInfo "Could not determine exact reparse point type for: $link (assuming valid junction)"
                }
            }
        }
    }
    
    if ($brokenLinks.Count -gt 0) {
        Write-LogError "One or more junctions are broken"
        return $false
    }
    
    Write-LogInfo "All junctions are valid"
    return $true
}

# ============================================================================
# MIGRATION DATABASE FUNCTIONS
# ============================================================================

function Invoke-UpgradeCommand {
    param(
        [string]$Action,
        [string]$SuccessState,
        [string]$ErrorState
    )
    
    Write-LogInfo "Running upgrade.php $Action..."
    
    # Clean up old logs
    Remove-Item -Path "stdout.log", "stderr.log" -ErrorAction SilentlyContinue
    
    $arguments = @("srm\srm_install\upgrade.php", $Action)
    
    try {
        $process = Start-Process -FilePath $global:PhpPath -ArgumentList $arguments -NoNewWindow -PassThru -RedirectStandardOutput "stdout.log" -RedirectStandardError "stderr.log"
        Show-Progress -ProcessId $process.Id
        $process | Wait-Process
        
        $stdout = Get-Content -Path "stdout.log" -Raw -ErrorAction SilentlyContinue
        $stderr = Get-Content -Path "stderr.log" -Raw -ErrorAction SilentlyContinue

        if ($stderr -and $stderr.Trim()) {
            Write-LogInfo "Composer output: $($stderr.Trim())"
        }
        else {
            Update-ProgressState $SuccessState
            Write-LogInfo "$Action completed successfully"
            return $true
        }
    }
    catch {
        Update-ProgressState $ErrorState
        Write-LogError "$Action failed: $($_.Exception.Message)"
        return $false
    }
    finally {
        Remove-Item -Path "stdout.log", "stderr.log" -ErrorAction SilentlyContinue
    }
}

function Start-DatabaseMigration {
    Write-StepHeader "Database Migration"
    
    Update-ProgressState "migration_started"
    
    # Attempt initial migration
    $migrationResult = Invoke-UpgradeCommand -Action "migrate" -SuccessState "migration_success" -ErrorState "migration_error"
    
    if (-not $migrationResult) {

        $errorMessage = $global:LASTERROR  # Assuming LASTERROR stores the most recent error message
        Write-Host "`nA migration error has occurred: $errorMessage" -ForegroundColor Red

        # Migration failed, offer retry/rollback options
        while ($true) {
            Write-Host ""
            Write-Host "To retry the migration, type 1. To roll back to the previous version of smart report maker, type 2:" -ForegroundColor Yellow
            Write-Host "1) Retry the migration" -ForegroundColor White
            Write-Host "2) Roll back to the previous version of Smart Report Maker" -ForegroundColor White
            
            $choice = Read-Host "Enter 1 or 2"
            
            switch ($choice) {
                '1' {
                    Write-LogInfo "Retrying migration..."
                    $migrationResult = Invoke-UpgradeCommand -Action "migrate" -SuccessState "migration_success" -ErrorState "migration_error"
                    if ($migrationResult) {
                        Write-LogInfo "Migration succeeded after retry."
                        return
                    }
                    $errorMessage = $global:LASTERROR
                    Write-Host "`nA migration error has occurred: $errorMessage" -ForegroundColor Red
                }
                '2' {
                    Write-LogInfo "User chose to rollback"
                    Invoke-UpgradeCommand -Action "rollback" -SuccessState "rollback_success" -ErrorState "rollback_error"
                    throw "Migration aborted by user"
                }
                default {
                    Write-Host "Invalid option. Please enter 1 or 2." -ForegroundColor Red
                }
            }
        }
    }
    
    Write-LogInfo "Database migration completed successfully"
}

# ============================================================================
# MAIN UPDATE FUNCTIONS
# ============================================================================

function Initialize-Update {
    Write-StepHeader "Update Initialization"
    
    # Initialize logging with safe file handling
    Initialize-Logging
    
    Write-LogInfo "SRM Update Started"
    Write-LogInfo "Script directory: $global:ScriptDir"
    Write-LogInfo "Old SRM directory: $global:SrmOldDir"
    Write-LogInfo "New SRM directory: $global:SrmNewDir"
    Write-LogInfo "Backup directory: $global:BackupDir"
}

function Initialize-Logging {
    # Stop any existing transcript
    try {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
    catch {
        # Ignore errors from stopping transcript
    }
    
    # Find an available log file name
    $logCounter = 0
    $baseLogFile = $global:LogFile
    $transcriptEnabled = $false
    
    while ($logCounter -le 10) {
        try {
            # Test if we can write to this log file
            $testContent = "# Log file test - $(Get-Date)"
            Add-Content -Path $global:LogFile -Value $testContent -ErrorAction Stop
            
            # If we can write, try to start transcript
            try {
                Start-Transcript -Path $global:LogFile -Append | Out-Null
                $transcriptEnabled = $true
                Write-Host "Logging initialized with transcript: $global:LogFile" -ForegroundColor Green
                break
            }
            catch {
                # Transcript failed but we can write to file, continue with basic logging
                Write-Host "Logging initialized (basic mode): $global:LogFile" -ForegroundColor Yellow
                break
            }
        }
        catch {
            # Cannot write to this file, try a different name
            $logCounter++
            if ($logCounter -le 10) {
                $fileInfo = [System.IO.FileInfo]$baseLogFile
                $global:LogFile = Join-Path $fileInfo.Directory.FullName "$($fileInfo.BaseName)_$logCounter$($fileInfo.Extension)"
            }
        }
    }
    
    if ($logCounter -gt 10) {
        # Generate a unique name with timestamp
        $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
        $fileInfo = [System.IO.FileInfo]$baseLogFile
        $global:LogFile = Join-Path $fileInfo.Directory.FullName "$($fileInfo.BaseName)_$timestamp$($fileInfo.Extension)"
        
        try {
            $testContent = "# Log file created - $(Get-Date)"
            Add-Content -Path $global:LogFile -Value $testContent -ErrorAction Stop
            Write-Host "Logging initialized with timestamp: $global:LogFile" -ForegroundColor Yellow
        }
        catch {
            Write-Warning "Could not initialize any log file. Proceeding without file logging."
            $global:LogFile = $null
        }
    }
}

function Show-UpdateSummary {
    Write-Host ""
    Write-Host "==============" -ForegroundColor Green
    Write-Host " UPDATE COMPLETE!" -ForegroundColor Green
    Write-Host "==============" -ForegroundColor Green
    Write-Host ""
    Write-Host "Next Steps:" -ForegroundColor Cyan
    Write-Host "1. Log in to your dashboard using your existing credentials" -ForegroundColor White
    Write-Host "2. Verify that everything is working correctly, including your reports" -ForegroundColor White
    Write-Host "3. Once confirmed, you may delete 'srm_old' and 'existed_resources' directories" -ForegroundColor White
    Write-Host "   (Keep backups for emergency recovery)" -ForegroundColor Gray
    Write-Host ""
    
    if ($global:Warnings.Count -gt 0) {
        Write-Host "Update completed with $($global:Warnings.Count) warning(s)" -ForegroundColor Yellow
        Write-Host "Check the log file for details: $global:LogFile" -ForegroundColor Yellow
    }
    else {
        Write-Host "Update completed successfully with no warnings!" -ForegroundColor Green
    }
    Write-Host ""
}

# ============================================================================
# PERMISSIONS FUNCTIONS
# ============================================================================
function Set-DirectoryPermissions {
    param([string[]]$Directories)
    
    Write-StepHeader "Permission Configuration"
    
    if (-not (Test-AdminRights)) {
        Write-LogWarning "Not running as administrator - some permission operations may fail"
    }
    
    foreach ($dir in $Directories) {
        if (-not (Test-Path $dir)) {
            Write-LogWarning "Directory does not exist: $dir"
            continue
        }
        
        Write-LogInfo "Setting permissions for: $dir"
        try {
            # Grant full control to Everyone (modify permissions)
            & icacls $dir /grant "Everyone:(OI)(CI)M" /T /C | Out-Null
            Write-LogInfo "Successfully set permissions for: $dir"
        }
        catch {
            Write-LogWarning "Failed to set permissions for: $dir. Error: $($_.Exception.Message)"
        }
    }
}

function Retry-Migrations {
    try {
        Initialize-Update
        Test-Prerequisites
        
        # if (-not (Test-VersionDifference)) {
        #     Write-LogInfo "No update required"
        #     return
        # }

        # Update-ProgressState "pre_migration"

        # $backupInfo = Invoke-BackupProcess
        
        # Copy-ConfigurationToNew -ConfigSource $backupInfo.ConfigSource
        # Copy-ReportsToNew -ReportsSource $backupInfo.ReportsSource
        # Copy-AppStorageToNew -AppStorageSource $backupInfo.AppStorageSource
        
        # Initialize-PhpEnvironment
        # Check-PHPExtensions -RequiredExtensions $global:RequiredExtensions
        # Test-Dependencies
        # Update-ComposerDependencies
        # Initialize-Junctions
        Start-DatabaseMigration
        
        Show-UpdateSummary
    }
    catch {
        Invoke-ErrorHandler -ErrorMessage $_.Exception.Message
    }
    finally {
        Stop-Transcript -ErrorAction SilentlyContinue
    }

}

function Start-UpdateProcess {
    try {
        Initialize-Update
        Test-Prerequisites
        
        if (-not (Test-VersionDifference)) {
            Write-LogInfo "No update required"
            return
        }

        Update-ProgressState "pre_migration"

        $backupInfo = Invoke-BackupProcess
        
        Copy-ConfigurationToNew -ConfigSource $backupInfo.ConfigSource
        Copy-ReportsToNew -ReportsSource $backupInfo.ReportsSource
        Copy-AppStorageToNew -AppStorageSource $backupInfo.AppStorageSource
        
        Initialize-PhpEnvironment
        Check-PHPExtensions -RequiredExtensions $global:RequiredExtensions
        Test-Dependencies
        Update-ComposerDependencies
        Initialize-Junctions
        Start-DatabaseMigration
        
        Show-UpdateSummary
    }
    catch {
        Invoke-ErrorHandler -ErrorMessage $_.Exception.Message
    }
    finally {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
}

function Resume-AfterComposer {
    try {
        Initialize-Update

        $backupInfo = Invoke-BackupProcess
        
        Copy-ConfigurationToNew -ConfigSource $backupInfo.ConfigSource
        Copy-ReportsToNew -ReportsSource $backupInfo.ReportsSource
        Copy-AppStorageToNew -AppStorageSource $backupInfo.AppStorageSource

        Initialize-PhpEnvironment
        Write-LogInfo "Resume all steps After Composer Dependency Installation"
        
        # Skip directly to symbolic links
        Initialize-Junctions
        
        $directories = @(
            $global:SrmNewDir,
            (Join-Path $global:SrmNewDir "dashboard"),
            (Join-Path $global:SrmNewDir "dashboard\public"),
            (Join-Path $global:SrmNewDir "dashboard\config"),
            (Join-Path $global:SrmNewDir "dashboard\storage\logs"),
            (Join-Path $global:SrmNewDir "modules\SRM9\SRM\Reports9")
        )
        
        Set-DirectoryPermissions -Directories $directories
        Start-DatabaseMigration
        Show-UpdateSummary
    }
    catch {
        Invoke-ErrorHandler -ErrorMessage $_.Exception.Message
    }
    finally {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
}

function Test-AdminRights {
    $currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($currentUser)
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# ============================================================================
# ROLLBACK FUNCTIONS
# ============================================================================
function Initialize-Rollback {
    Write-StepHeader "Rollback Initialization"
    Initialize-PhpEnvironment
    if (-not (Test-Path $global:ProgressFile)) {
        throw "No progress file found. Cannot determine rollback state."
    }

    $progressState = Get-Content $global:ProgressFile | Where-Object {
        $_ -match "^(pre_migration|migration_started|migration_error|migration_success)$"
    } | Select-Object -Last 1

    if (-not $progressState) {
        throw "Invalid progress state in $global:ProgressFile"
    }

    switch ($progressState) {
        "pre_migration" {
            Write-LogInfo "Rollback detected in pre-migration state"
            Invoke-RollbackPreMigration
        }
        {"migration_started", "migration_error"} {
            Write-LogInfo "Rollback detected during/after failed migration"
            Invoke-RollbackWithMigration
        }
        "migration_success" {
            Write-LogInfo "Rollback detected after successful migration"
            Confirm-RollbackAfterSuccess
        }
        default {
            throw "Unknown progress state: $progressState"
        }
    }
}

function Invoke-RollbackPreMigration {
    Write-LogInfo "Performing pre-migration rollback..."
    
    if (-not (Test-Path $global:SrmOldDir)) {
        throw "Cannot find $global:SrmOldDir for rollback"
    }

    if (Test-Path $global:SrmNewDir) {
        Rename-Item -Path $global:SrmNewDir -NewName "srm_new" -ErrorAction Stop
    }

    Rename-Item -Path $global:SrmOldDir -NewName "srm" -ErrorAction Stop

    Write-LogInfo "Pre-migration rollback completed successfully"
    Show-RollbackSuccessMessage
    exit 0
}

function Invoke-RollbackWithMigration {
    Write-LogInfo "Performing rollback with database migration..."

    # Check if srm_install directory exists
    $srmInstallPath = Join-Path $global:SrmNewDir "srm_install"
    if (-not (Test-Path $srmInstallPath)) {

        throw "The 'srm_install' directory is missing, possibly due to automatic deletion for security purposes. Please copy it from the downloaded full version and place it under the '$global:SrmNewDir/' directory, so the path becomes '$global:SrmNewDir/srm_install/'."
    }

    # Perform database rollback
    Write-LogInfo "Attempting database rollback..."
    $phpArgs = @(
        (Join-Path $global:SrmNewDir "srm_install\upgrade.php"),
        "rollback"
    )
    
    try {
        $process = Start-Process -FilePath $global:PhpPath -ArgumentList $phpArgs -NoNewWindow -PassThru -Wait
        if ($process.ExitCode -ne 0) {
            throw "Database rollback failed with exit code $($process.ExitCode)"
        }
    }
    catch {
        throw "Database migration rollback failed. Error: $($_.Exception.Message)"
    }

    # Restore directories
    if (-not (Test-Path $global:SrmOldDir)) {
        throw "Cannot find $global:SrmOldDir for rollback"
    }

    if (Test-Path $global:SrmNewDir) {
        Rename-Item -Path $global:SrmNewDir -NewName "srm_new" -ErrorAction Stop
    }

    Rename-Item -Path $global:SrmOldDir -NewName "srm" -ErrorAction Stop

    Write-LogInfo "Rollback with migration completed successfully"
    Show-RollbackSuccessMessage
    exit 0
}

function Confirm-RollbackAfterSuccess {
    Write-LogWarning "The migration was previously completed successfully."
    Write-Host ""
    Write-Host "============================================================================"
    Write-Host "WARNING: Migration Was Previously Successful" -ForegroundColor Yellow
    Write-Host "============================================================================"
    Write-Host "According to the upgrade history, the migration was completed successfully."
    Write-Host "Are you sure you want to roll back?"
    Write-Host ""
    $userInput = Read-Host "Type 'yes' to confirm rollback, or anything else to cancel"
    
    if ($userInput -ne "yes") {
        Write-LogInfo "Rollback has been canceled by user"
        Write-Host ""
        Write-Host "Rollback has been canceled" -ForegroundColor Green
        exit 0
    }

    Write-LogInfo "User confirmed rollback after successful migration"
    Invoke-RollbackWithMigration
}

function Show-RollbackSuccessMessage {
    Write-Host ""
    Write-Host "============================================================================"
    Write-Host "ROLLBACK COMPLETED SUCCESSFULLY" -ForegroundColor Green
    Write-Host "============================================================================"
    Write-Host "The 'srm_old' directory has been successfully restored. Please log in to your"
    Write-Host "dashboard to verify that everything is functioning correctly. Once confirmed,"
    Write-Host "you may safely delete the 'srm_new' and 'existed_resources' directories."
    Write-Host "============================================================================"
}



# ============================================================================
# SCRIPT ENTRY POINT
# ============================================================================

# Validate PowerShell version
if ($PSVersionTable.PSVersion.Major -lt 5) {
    Write-Error "This script requires PowerShell 5.1 or later. Current version: $($PSVersionTable.PSVersion)"
    exit 1
}

# Display header
Clear-Host
Write-Host "Smart Report Maker (SRM) - Windows Update Script" -ForegroundColor Cyan
Write-Host "Version: 1.0" -ForegroundColor Gray


# Start installation based on flag
if ($Rollback) {
    Write-Host "Mode: Rollback" -ForegroundColor Yellow

    Write-Host ""
    Initialize-Rollback

}
elseif ($ResumeAfterComposer) {
    Write-Host "Mode: Resume After Composer" -ForegroundColor Yellow
    Write-Host ""

    Resume-AfterComposer

}
elseif ($RetryMigrations) {
    Write-Host "Mode: Retry Migrations" -ForegroundColor Yellow
    Write-Host ""

    Retry-Migrations
    
}
else {
    Start-UpdateProcess
}
