$(function () {

    window.handleNumericSlider = function handleNumericSlider(formID) {
        let sliders = document.querySelectorAll(`#${formID} [id$="-numeric-slider"]`);
        sliders.forEach(slider => {
            let min = Number(slider.nextElementSibling.querySelector(`#${formID} [id^="min"]`).textContent);
            let max = Number(slider.nextElementSibling.querySelector(`#${formID} [id^="max"]`).textContent);
            noUiSlider.create(slider, {
                start: [min, max],
                connect: true,
                range: {
                    'min': min,
                    'max': max
                },
                step: 1,
                tooltips: true,
                format: {

                    to: function (value) {
                        return Math.round(value);
                    },
                    from: function (value) {
                        return Number(value);
                    }
                }
            });

            var $tooltips = $(slider).find('.noUi-tooltip');

            $tooltips.hide();

            slider.noUiSlider.on('start', function () {
                $tooltips.show();
            });

            slider.addEventListener('mouseover', function () {
                $tooltips.show();
            });

            slider.addEventListener('mouseout', function () {
                $tooltips.hide();
            });

            slider.noUiSlider.on('change', function (values, handle) {
                let min = Math.round(values[0]);
                let max = Math.round(values[1]);
                let inputs = slider.parentElement.querySelector('input');
                inputs.value = `min:${min},max:${max}`;

                setTimeout(function () {
                    $tooltips.hide();
                }, 1000);
            });
        });
    }

    window.handleTimeSlider = function handleTimeSlider(formID) {
        let sliders = document.querySelectorAll(`#${formID} [id$="-time-slider"]`);

        sliders.forEach(slider => {
            noUiSlider.create(slider, {
                start: [0, 1439], // Midnight to Noon
                connect: true,
                range: {
                    'min': 0,
                    'max': 1439 // 24 hours * 60 minutes = 1439
                },
                step: 1, // Adjust to 5-minute increments
                tooltips: true,
                format: {
                    to: function (value) {
                        let totalMinutes = Math.round(value);
                        let hours = Math.floor(totalMinutes / 60);
                        let minutes = totalMinutes % 60;
                        let period = hours < 12 ? "AM" : "PM";
                        let displayHour = hours % 12 || 12;
                        let displayMinutes = String(minutes).padStart(2, "0");

                        return `${displayHour}:${displayMinutes} ${period}`;
                    },
                    from: function (value) {
                        let match = value.match(/^(\d+):(\d+) (AM|PM)$/);
                        if (!match) return Math.round(value); // Prevent NaN

                        let hour = parseInt(match[1], 10);
                        let minute = parseInt(match[2], 10);
                        let period = match[3];

                        if (period === "PM" && hour !== 12) hour += 12;
                        if (period === "AM" && hour === 12) hour = 0;

                        return hour * 60 + minute;
                    }
                }
            });

            var $tooltips = $(slider).find('.noUi-tooltip');
            $tooltips.hide();

            slider.noUiSlider.on('start', function () {
                $tooltips.show();
            });

            slider.addEventListener('mouseover', function () {
                $tooltips.show();
            });

            slider.addEventListener('mouseout', function () {
                $tooltips.hide();
            });


            slider.noUiSlider.on('end', function () {
                setTimeout(function () {
                    $tooltips.hide();
                }, 1000);

            });

            slider.noUiSlider.on('change', function (values, handle) {

                function formatTo24Hours(value) {
                    let match = value.match(/^(\d+):(\d+) (AM|PM)$/);
                    if (!match) return "00:00"; // Default fallback

                    let hour = parseInt(match[1], 10);
                    let minute = parseInt(match[2], 10);
                    let period = match[3];

                    if (period === "PM" && hour !== 12) hour += 12;
                    if (period === "AM" && hour === 12) hour = 0;

                    return `${hour.toString().padStart(2, '0')}:${minute.toString().padStart(2, '0')}`;

                }

                let min = formatTo24Hours(values[0]);
                let max = formatTo24Hours(values[1]);

                let inputs = slider.parentElement.querySelector('input');
                inputs.value = `min=${min},max=${max}`;
            });
        });
    }


    window.countKeys = function countKeys(obj) {
        let keys = 0;

        function traverse(o) {
            Object.keys(o).forEach(key => {
                keys++;
                if (typeof o[key] === 'object' && o[key] !== null) {
                    traverse(o[key]);
                }
            });
        }

        traverse(obj);
        return keys;
    }


    window.countLastLevelKeys = function countLastLevelKeys(obj) {
        let keys = new Set();

        function traverse(o) {
            Object.keys(o).forEach(key => {
                if (typeof o[key] === 'object' && o[key] !== null) {
                    traverse(o[key]);
                } else {
                    keys.add(key);
                }
            });
        }

        traverse(obj);
        return keys.size;
    }


    window.getHiddenColumnIndexes = function getHiddenColumnIndexes(formID) {
        let hiddenIndexes = new Set();


        document.querySelectorAll(`#${formID} tbody tr`).forEach((row) => {
            row.querySelectorAll("td").forEach((el, index) => {
                if (el.style.display === "none") {
                    hiddenIndexes.add(index);
                }
            });
        });


        document.querySelectorAll(`#${formID} .grand-total td`).forEach((el, index) => {
            if (el.style.display === "none") {
                hiddenIndexes.add(index);
            }
        });

        return hiddenIndexes;
    }

    window.restoreHiddenColumns = function restoreHiddenColumns(hiddenIndexes, formID) {

        document.querySelectorAll(`#${formID} tbody tr`).forEach((row) => {
            row.querySelectorAll("td").forEach((el, index) => {
                if (hiddenIndexes.has(index)) {
                    el.style.display = "none";
                }
            });
        });


        document.querySelectorAll(`#${formID} .grand-total td`).forEach((el, index) => {
            if (hiddenIndexes.has(index)) {
                el.style.display = "none";
            }
        });
    }

    window.toggleAllYears = function toggleAllYears(formID) {

        $(`#${formID} .toggle-years`).each(function () {
            var th = $(this).parent("th");
            var year = th.data("year");
            var originalColspan = th.data("original-colspan") || th.attr("colspan");

            if (!th.data("original-colspan")) {
                th.data("original-colspan", originalColspan);
            }

            if ($(`#${formID}-toggle-columns`).attr('data-toggle') === "true") {
                // Hide year columns
                $(`#${formID} .sub-header th`).filter(`[data-year='${year}']`).hide();
                var subHeaders = $(`#${formID} .sub-sub-header th`).filter(`[data-year='${year}']`);
                subHeaders.hide();
                $(`#${formID} tbody tr, #${formID} tfoot tr`).each(function () {
                    $(this).find(`td[data-year='${year}']`).hide();
                });

                // Hide null columns too
                $(`#${formID} .sub-header th`).filter(`[data-year='Null']`).hide();
                $(`#${formID} .sub-sub-header th`).filter(`[data-year='Null']`).hide();
                $(`#${formID} tbody tr, #${formID} tfoot tr`).each(function () {
                    $(this).find(`td[data-year='Null']`).hide();
                });

                th.attr("colspan", 1);
            } else {
                // Show year columns
                $(`#${formID} .sub-header th`).filter(`[data-year='${year}']`).show();
                var subHeaders = $(`#${formID} .sub-sub-header th`).filter(`[data-year='${year}']`);
                subHeaders.show();
                $(`#${formID} tbody tr, #${formID} tfoot tr`).each(function () {
                    $(this).find(`td[data-year='${year}']`).show();
                });

                // Show null columns too
                $(`#${formID} .sub-header th`).filter(`[data-year='Null']`).show();
                $(`#${formID} .sub-sub-header th`).filter(`[data-year='Null']`).show();
                $(`#${formID} tbody tr, #${formID} tfoot tr`).each(function () {
                    $(this).find(`td[data-year='Null']`).show();
                });

                th.attr("colspan", originalColspan);
            }

            if ($(`#${formID}-toggle-columns`).attr('data-attribute') === "true") {
                $(this).removeClass("bi-dash-square").addClass("bi-plus-square");
            } else {
                $(this).removeClass("bi-plus-square").addClass("bi-dash-square");
            }
        });
    }

    window.sortHeaders = function sortHeaders(arr) {
        if (!Array.isArray(arr) || arr.length < 3) return arr;

        let first = arr[0];
        let last = arr[arr.length - 1];


        let years = arr.slice(1, -1)
            .map(s => s.trim())
            .sort((a, b) => a - b)
            .map(year => ' ' + year);

        return [first, ...years, last];
    }

    window.renderComplex3DTable = function renderComplex3DTable(data, formID = "pivot-table-form", page) {

        const subHeader = document.querySelector(`#${formID} .sub-header`);
        const mainHeader = document.querySelector(`#${formID} .main-header`);
        const tableBody = document.querySelector(`#${formID} .table-body`);


        let headers = Object.keys(data.data[0]);

        // Move "Null" column before the last column
        const nullIndex = headers.indexOf("Null");
        if (nullIndex !== -1 && headers.length > 1) {
            const [nullHeader] = headers.splice(nullIndex, 1);
            headers.splice(data.options?.total ? headers.length - 1 : headers.length, 0, nullHeader);
        }

        const totalRow = data.data[data.data.length - 1];
        const fullLength = countKeys(data.data[0]);

        let subSubHeader = document.querySelector(`#${formID} .sub-sub-header`);

        mainHeader.querySelectorAll(`#${formID} tr:not(.sub-header):not(.sub-sub-header)`).forEach(tr => tr.remove());

        let sortState = {};

        const totalRows = data.data.length;
        const totalPages = Math.ceil(totalRows / rowsPerPage);
        currentPage = Math.max(1, Math.min(page, totalPages));

        const startIdx = (currentPage - 1) * rowsPerPage;
        const endIdx = startIdx + rowsPerPage;
        const paginatedData = data.data.slice(startIdx, endIdx);

        const firstRowHeader = document.createElement("tr");

        firstRowHeader.innerHTML = `<th></th>
        <th colspan="${fullLength}">
            ${data.options?.total ? `<i class="bi bi-dash-square no-print" data-toggle="false" style="cursor:pointer" id="${formID}-toggle-columns"></i>` : ''}
            <h6 class="d-inline">${data.options?.value}</h6>
        </th>`;
        firstRowHeader.classList.add("table-light");
        mainHeader.prepend(firstRowHeader);

        subHeader.classList.add("sub-header");

        if (!subHeader.hasChildNodes()) {

            headers.forEach((header, index) => {
                sortState[header] = "asc";

                const firstValue = data.data[0][header];
                const headerLength = typeof firstValue === "object" && firstValue !== null
                    ? Object.keys(firstValue).length
                    : 1;

                const th = document.createElement("th");

                // "Null" column gets rowspan="2", others behave as before
                if (header === "Null" || (index === headers.length - 1 && data.options?.total)) {
                    th.setAttribute("rowspan", 2);
                } else if (index === 0) {
                    th.setAttribute("rowspan", 2);
                } else {
                    th.setAttribute("colspan", headerLength);
                }

                // th.setAttribute("rowspan", (index === headers.length - 1 || index === 0 || header == "Null") ? 2 : 1);
                th.setAttribute("data-year", header.trim());

                // First column with toggle
                if (index === 0) {
                    th.innerHTML = `
                    ${data.options?.total ? `<i class="bi bi-dash-square no-print" style="cursor:pointer" id="${formID}-toggle-rows"></i>` : ''}
                    <h6 class="d-inline">${header}</h6>
                    <span class="sort-arrow" data-column="${header}" style="cursor: pointer;">
                        <i class="bi bi-sort-down-alt no-print"></i>
                    </span>`;
                }
                // Null column
                else if (header === "Null" || (index === headers.length - 1 && data.options?.total)) {
                    th.innerHTML = `
                        <h6 class="d-inline">${header}</h6>
                        <span class="sort-arrow" data-column="${header}" style="cursor: pointer;">
                            <i class="bi bi-sort-down-alt no-print"></i>
                        </span>`;
                }
                // Other columns (conditionally show toggle icon)
                else {
                    th.innerHTML = `
                    ${data.options?.total ? '<i class="bi bi-dash-square no-print toggle-years" style="cursor:pointer"></i>' : ''}
                    <h6 class="d-inline">${header}</h6>`;
                }

                // Attach sort event
                th.querySelector(`.sort-arrow`)?.addEventListener('click', function () {

                    let hiddenColumns = getHiddenColumnIndexes(formID);
                    sortState[header] = sortState[header] === "asc" ? "desc" : "asc";
                    sort3DTable(data.data, header, sortState[header]);

                    renderComplex3DTable(data, formID, currentPage, false);
                    updateSortIcons(header, sortState[header]);

                    restoreHiddenColumns(hiddenColumns, formID);
                });

                subHeader.appendChild(th);

            });


            $(document).off('click', `#${formID}-toggle-rows`);
            $(document).off('click', `#${formID}-toggle-columns`);
            $(document).off('click', `#${formID} .toggle-years`);

            $(document).on("click", `#${formID}-toggle-rows`, function () {
                $(`#${formID} .table-body`).toggle();
                var icon = $(this).find("i");
                $(this).toggleClass("bi-plus-square bi-dash-square");
            });

            $(document).on("click", `#${formID}-toggle-columns`, function () {
                $(this).attr("data-toggle", $(this).attr('data-toggle') === 'true' ? 'false' : 'true');
                toggleAllYears(formID);

                var icon = $(this);
                icon.toggleClass("bi-plus-square bi-dash-square");
            });

            $(document).on("click", `#${formID} .toggle-years`, function () {
                var th = $(this).parent("th");

                var year = th.data("year");
                var colspan = th.attr("colspan") ? parseInt(th.attr("colspan")) : 1;


                if (!th.data("original-colspan")) {
                    th.data("original-colspan", colspan);
                }

                var originalColspan = th.data("original-colspan");
                var isCollapsed = th.attr("colspan") == "1";


                var subHeaders = $(`#${formID} .sub-sub-header th`).filter(`[data-year='${year}']`);

                if (isCollapsed) {

                    subHeaders.show();
                } else {

                    subHeaders.hide().last().show();
                }


                $("tbody tr, tfoot tr").each(function () {
                    var tds = $(this).find(`td[data-year='${year}']`);

                    if (isCollapsed) {

                        tds.show();
                    } else {

                        tds.hide().last().show();
                    }
                });


                th.attr("colspan", isCollapsed ? originalColspan : "1");

                $(this).toggleClass("bi-plus-square bi-dash-square");

            });


        }

        if (!subSubHeader.hasChildNodes()) {

            headers.forEach((header, index) => {

                const subSubHeaders = data.data[0][header] || "";

                if (subSubHeaders !== "" && typeof subSubHeaders === 'object') {

                    Object.entries(subSubHeaders).forEach(([key, value], index) => {
                        let column = header.trim() + "-" + key;
                        sortState[column] = "asc";

                        const subTh = document.createElement("th");
                        subTh.setAttribute("data-year", header.trim());

                        subTh.classList.add("sortable");

                        subTh.innerHTML = `
                        <h6 class="d-inline">${key}
                            <span class="sort-arrow" data-column="${column}" style="cursor: pointer;">
                            <i class="bi bi-sort-down-alt  no-print"></i>
                            </span>
                        </h6>
                        `;

                        subTh.querySelector(`.sort-arrow`).addEventListener('click', function () {
                            let hiddenColumns = getHiddenColumnIndexes();
                            sortState[key] = sortState[key] === "desc" ? "asc" : "desc";
                            console.log(header, key);

                            sort3DComplexTable(data.data, header, key, sortState[key], data.options?.total);
                            renderComplex3DTable(data, formID, currentPage, false);
                            updateSortIcons(`${header}-${key}`, sortState[key]);

                            restoreHiddenColumns(hiddenColumns, formID);
                        });
                        subSubHeader.appendChild(subTh);
                    })
                }

            });
        }


        tableBody.innerHTML = '';

        renderComplex3DTableBody(paginatedData, formID, headers, data.options?.total);
        const footerRow = document.querySelector(`#${formID} .grand-total`);
        footerRow.innerHTML = '';

        if (data.options?.total) {
            if (footerRow) {
                footerRow.innerHTML = `<td class="table-light"><h6>${Object.values(data.options.total)[0]}</h6></td>`;
                headers.slice(1).forEach(header => {
                    if (typeof totalRow[header] === 'object') {
                        Object.entries(totalRow[header]).forEach(([key, value]) => {
                            const td = document.createElement("td");
                            td.classList.add("table-default", "text-primary");
                            td.innerHTML = `<h6>${value ?? 0}</h6>`;
                            td.setAttribute("data-year", header.trim());
                            footerRow.appendChild(td);
                        });
                    } else {
                        const td = document.createElement("td");
                        td.classList.add("table-default", "text-primary");
                        td.innerHTML = `<span>${totalRow[header] ?? 0}</span>`;
                        td.setAttribute("data-year", header.trim());

                        footerRow.appendChild(td);
                    }
                });
            } else {
                const footerRow = document.querySelector(`#${formID} .grand-total`);
                if (footerRow) {
                    while (footerRow.firstChild) {
                        footerRow.removeChild(footerRow.firstChild);
                    }
                }
            }

        }
        changeAlignment(formID);
        createPagination(totalPages, data, formID);
    }



    window.render3DTable = function render3DTable(data, formID = "pivot-table-form", page) {

        const subHeader = document.querySelector(`#${formID} .sub-header`);
        const mainHeader = document.querySelector(`#${formID} .main-header`);
        const tableBody = document.querySelector(`#${formID} .table-body`);


        const totalRow = data.data[data.data.length - 1];
        let = data.data.length;


        document.querySelector(`#${formID} .show-hide-columns`)?.removeAttribute("disabled");

        document.querySelector(`#${formID} .main-header`).querySelectorAll("tr:not(.sub-header)").forEach(tr => tr.remove());

        const headers = Object.keys(data.data[0]);
        let sortState = {};


        const totalRows = data.data.length;
        const totalPages = Math.ceil(totalRows / rowsPerPage);
        currentPage = Math.max(1, Math.min(page, totalPages));

        const startIdx = (currentPage - 1) * rowsPerPage;
        const endIdx = startIdx + rowsPerPage;
        const paginatedData = data.data.slice(startIdx, endIdx);

        const firstRowHeader = document.createElement("tr");

        firstRowHeader.innerHTML = `<th></th>
        <th colspan="${headers.length - 1}">
            ${data.options?.total ? `<i class="bi bi-dash-square no-print" style="cursor:pointer" id="${formID}-toggle-columns"></i>` : ''}
            <h6 class="d-inline">${data.options?.value}</h6>
        </th>`;

        firstRowHeader.classList.add("table-light");

        mainHeader.prepend(firstRowHeader);


        subHeader.classList.add("sub-header");

        if (!subHeader.hasChildNodes()) {

            headers.forEach((header, index) => {
                sortState[header] = "asc";

                const th = document.createElement("th");
                th.classList.add("sortable");

                if (index === 0) {
                    th.innerHTML = `
                        ${data.options?.total ? `<i class="bi bi-dash-square no-print" style="cursor: pointer;" id="${formID}-toggle-rows"></i>` : ''}
                        <h6 class="d-inline">${header}
                            <span class="sort-arrow" data-column="${header}">
                                <i class="bi bi-sort-down-alt no-print"></i>
                            </span>
                        </h6>
                    `;
                } else {
                    th.innerHTML = `
                        <h6>${header}
                            <span class="sort-arrow" data-column="${header}" style="cursor: pointer;">
                                <i class="bi bi-sort-down-alt no-print"></i>
                            </span>
                        </h6>
                    `;
                }

                th.querySelector('.sort-arrow').addEventListener('click', function () {

                    sortState[header] = sortState[header] === "asc" ? "desc" : "asc";

                    sort3DTable(data.data, header, sortState[header]);
                    render3DTable(data, formID, currentPage, false);
                    updateSortIcons(header, sortState[header]);
                });

                subHeader.appendChild(th);
            });

            $(document).off('click', `#${formID}-toggle-rows`);
            $(document).off('click', `#${formID}-toggle-columns`);


            $(document).on("click", `#${formID}-toggle-rows`, function () {
                $(`#${formID} .table-body`).toggle();
                var icon = $(this).find("i");
                $(this).toggleClass("bi-plus-square bi-dash-square");
            });

            $(document).on("click", `#${formID}-toggle-columns`, function () {
                var lastColIndex = $(`#${formID} .sub-header th`).length;

                // Hide/show all columns except the first and last
                $(`#${formID} .sub-header th:nth-child(n+2):nth-child(-n+${lastColIndex - 1}),
                    #${formID} tbody tr td:nth-child(n+2):nth-child(-n+${lastColIndex - 1}),
                    #${formID} .grand-total td:nth-child(n+2):nth-child(-n+${lastColIndex - 1})`).toggle();

                var icon = $(this);
                icon.toggleClass("bi-plus-square bi-dash-square");
            });
        }


        tableBody.innerHTML = '';


        render3DTableBody(paginatedData, formID, headers, data.options?.total);
        if (data.options?.total) {
            const footerRow = document.querySelector(`#${formID} .grand-total`);
            if (footerRow) {
                footerRow.innerHTML = `<td class="table-light"><h6>${Object.values(data.options.total)[0]}</h6></td>`;
                headers.slice(1).forEach(header => {
                    const td = document.createElement("td");
                    td.classList.add("table-default", "text-primary");
                    td.innerHTML = `<span>${totalRow[header] ?? 0}</span>`;
                    footerRow.appendChild(td);
                });
            }
        } else {

            const footerRow = document.querySelector(`#${formID} .grand-total`);

            if (footerRow) {
                while (footerRow.firstChild) {
                    footerRow.removeChild(footerRow.firstChild);
                }
            }
        }

        changeAlignment(formID);

        createPagination(totalPages, data, formID);
    }

    window.sort3DTable = function sort3DTable(dataArray, column, order) {
        if (dataArray.length < 2) return;

        const totalRow = dataArray[dataArray.length - 1];
        const sortableData = dataArray.slice(0, -1);

        sortableData.sort((a, b) => {
            let valueA = a[column] || 0;
            let valueB = b[column] || 0;

            if (!isNaN(valueA) && !isNaN(valueB)) {
                return order === "asc" ? valueA - valueB : valueB - valueA;
            } else {
                return order === "asc"
                    ? valueA.toString().localeCompare(valueB.toString())
                    : valueB.toString().localeCompare(valueA.toString());
            }
        });


        dataArray.length = 0;
        dataArray.push(...sortableData, totalRow);
    }


    window.sort3DComplexTable = function sort3DComplexTable(dataArray, column, subColumn, order, removeLastRow) {
        if (dataArray.length < 2) return;


        const totalRow = dataArray[dataArray.length - 1];
        const sortableData = removeLastRow ? dataArray.slice(0, -1) : dataArray.slice();

        sortableData.sort((a, b) => {
            let valueA = a[column][subColumn] || 0;
            let valueB = b[column][subColumn] || 0;

            if (!isNaN(valueA) && !isNaN(valueB)) {
                return order === "asc" ? valueA - valueB : valueB - valueA;
            } else {
                return order === "asc"
                    ? valueA.toString().localeCompare(valueB.toString())
                    : valueB.toString().localeCompare(valueA.toString());
            }
        });

        dataArray.length = 0;

        if (removeLastRow)
            dataArray.push(...sortableData, totalRow);
        else
            dataArray.push(...sortableData);

    }

    window.renderTableAsRows = function renderTableAsRows(data, formID, page) {
        const subHeader = document.querySelector(`#${formID} .sub-header`);
        const tableBody = document.querySelector(`#${formID} .table-body`);
        document.querySelector(`#${formID} .main-header`).querySelectorAll(`tr:not(.sub-header)`).forEach(tr => tr.remove());


        document.getElementById("show-hide-columns")?.setAttribute("disabled", true);
        const headers = Object.keys(data.data[0]);
        let sortState = {};


        const totalRows = data.data.length;
        const totalPages = Math.ceil(totalRows / rowsPerPage);
        currentPage = Math.max(1, Math.min(page, totalPages));

        const startIdx = (currentPage - 1) * rowsPerPage;
        const endIdx = startIdx + rowsPerPage;
        const paginatedData = data.data.slice(startIdx, endIdx);


        if (!subHeader.hasChildNodes()) {
            headers.forEach((header, index) => {
                sortState[header] = "desc";

                const th = document.createElement("th");
                th.classList.add("sortable");

                if (index === 0) {
                    th.innerHTML = `
                        ${data.options?.total
                            ? `<i class="bi bi-dash-square no-print" id="${formID}-toggle-rows" style="cursor: pointer;"></i>`
                            : ''}
                        <h6 class="d-inline">${header}
                            <span class="sort-arrow" data-column="${header}" style="cursor: pointer;">
                                <i class="bi bi-sort-up-alt no-print"></i>
                            </span>
                        </h6>
                    `;
                } else {
                    th.innerHTML = `
                        <h6>${header}
                            <span class="sort-arrow" data-column="${header}" style="cursor: pointer;">
                                <i class="bi bi-sort-up-alt no-print"></i>
                            </span>
                        </h6>
                    `;
                }


                th.querySelector('.sort-arrow').addEventListener('click', function () {

                    sortState[header] = sortState[header] === "desc" ? "asc" : "desc";
                    sortTable(data.data, header, sortState[header]);
                    renderTableAsRows(data, formID, currentPage);
                    updateSortIcons(header, sortState[header]);
                });

                subHeader.appendChild(th);
            });

        }

        $(document).off('click', `#${formID}-toggle-rows`);


        $(document).on("click", `#${formID}-toggle-rows`, function () {
            $(`#${formID} .table-body`).toggle();
            var icon = $(this).find("i");
            $(this).toggleClass("bi-plus-square bi-dash-square");
        });



        tableBody.innerHTML = '';


        render2DTableBody(paginatedData, formID, headers);


        if (data.options?.total) {
            const footerRow = document.querySelector(`#${formID} .grand-total`);
            if (footerRow) {
                footerRow.innerHTML = `<td class="table-light"><h6>${Object.values(data.options.total)[0]}</h6></td>`;
                headers.slice(1).forEach((header, index) => {
                    const td = document.createElement("td");

                    td.innerHTML = `<span class="text-primary">${data.options.total[header] ?? 0}</span>`;
                    footerRow.appendChild(td);
                });
            }
        } else {
            const footerRow = document.querySelector(`#${formID} .grand-total`);
            if (footerRow) {
                while (footerRow.firstChild) {
                    footerRow.removeChild(footerRow.firstChild);
                }
            }
        }

        changeAlignment(formID);

        createPagination(totalPages, data, formID);
    }



    window.sortTable = function sortTable(dataArray, column, order) {
        dataArray.sort((a, b) => {
            let valueA = a[column] || 0;
            let valueB = b[column] || 0;

            if (!isNaN(valueA) && !isNaN(valueB)) {
                return order === "asc" ? valueA - valueB : valueB - valueA;
            } else {
                return order === "asc"
                    ? valueA.toString().localeCompare(valueB.toString())
                    : valueB.toString().localeCompare(valueA.toString());
            }
        });
    }


    window.renderComplex3DTableBody = function renderComplex3DTableBody(paginatedData, formID, headers, removeLastRow = false) {

        const tableBody = document.querySelector(`#${formID} .table-body`);

        if (removeLastRow) {
            paginatedData = paginatedData.slice(0, -1)
        }

        paginatedData.forEach((row) => {

            const tr = document.createElement("tr");

            headers.forEach((header, index) => {

                const td = document.createElement("td");

                td.classList.toggle("table-light", index === 0);

                td.classList.toggle("text-primary", (index === headers.length - 1 && removeLastRow));

                if (typeof row[header] === 'object') {
                    Object.entries(row[header]).forEach(([key, value], index) => {

                        const td2 = document.createElement("td");
                        td2.setAttribute("data-year", header.trim());
                        td2.textContent = value ?? '';

                        td2.classList.toggle("text-default");


                        tr.appendChild(td2);
                    });
                } else {
                    td.textContent = row[header] ?? '';
                    td.setAttribute("data-year", header.trim());
                    tr.appendChild(td);
                }

                tableBody.appendChild(tr);
            });
        });


    }

    window.render3DTableBody = function render3DTableBody(paginatedData, formID, headers, removeLastRow = false) {
        const tableBody = document.querySelector(`#${formID} .table-body`);

        if (removeLastRow) {
            paginatedData = paginatedData.slice(0, -1)
        }
        paginatedData.forEach((row) => {
            const tr = document.createElement("tr");
            headers.forEach((header, index) => {
                const td = document.createElement("td");

                if (index === 0) {
                    td.classList.add("table-light")
                    const h6 = document.createElement("h6");
                    h6.innerHTML = row[header] ?? ''
                    td.appendChild(h6);
                } else {
                    td.classList.add("text-default");
                    td.innerHTML = row[header] ?? ''
                }
                if (index === headers.length - 1 && removeLastRow) {
                    td.classList.add("text-primary");
                }
                tr.appendChild(td);
            });

            tableBody.appendChild(tr);
        });

    }

    window.render2DTableBody = function render2DTableBody(paginatedData, formID, headers, removeLastRow = false) {
        const tableBody = document.querySelector(`#${formID} .table-body`);

        if (removeLastRow) {
            paginatedData = paginatedData.slice(0, -1)
        }
        paginatedData.forEach((row) => {
            const tr = document.createElement("tr");
            headers.forEach((header, index) => {
                const td = document.createElement("td");

                if (index === 0) {
                    td.classList.add("table-light")

                    const h6 = document.createElement("h6");
                    h6.innerHTML = row[header] ?? ''
                    td.appendChild(h6);

                } else {

                    td.classList.add("text-default");
                    td.innerHTML = row[header] ?? ''

                }

                tr.appendChild(td);
            });

            tableBody.appendChild(tr);
        });

    }



    window.sortTable = function sortTable(dataArray, column, order) {
        dataArray.sort((a, b) => {
            let valueA = a[column] || 0;
            let valueB = b[column] || 0;

            if (!isNaN(valueA) && !isNaN(valueB)) {
                return order === "asc" ? valueA - valueB : valueB - valueA;
            } else {
                return order === "asc"
                    ? valueA.toString().localeCompare(valueB.toString())
                    : valueB.toString().localeCompare(valueA.toString());
            }
        });
    }


    window.updateSortIcons = function updateSortIcons(activeColumn, order) {
        const $activeArrow = $(`.sort-arrow[data-column="${!isNaN(activeColumn) ? activeColumn : activeColumn.trim()}"]`);
        if ($activeArrow.length) {
            $activeArrow.html(order === "asc"
                ? `<i class="bi bi-sort-down-alt no-print"></i>`
                : `<i class="bi bi-sort-up-alt no-print"></i>`);
        }
    }




    window.renderTableAsColumns = function renderTableAsColumns(data, formID = "pivot-table-form") {
        const subHeader = document.querySelector(`#${formID} .sub-header`);
        const mainHeader = document.querySelector(`#${formID} .main-header`);
        const tableBody = document.querySelector(`#${formID} .table-body`);
        const fullDataLength = data.options?.total ? data.data.length + 1 : data.data.length;

        mainHeader.querySelectorAll(`#${formID} tr:not(.sub-header)`).forEach(tr => tr.remove());
        document.querySelector(`#${formID} .show-hide-columns`)?.removeAttribute("disabled");

        subHeader.innerHTML = "";
        tableBody.innerHTML = "";

        const fields = Object.keys(data.data[0])[0];
        const values = Object.keys(data.data[0]).slice(1);



        const firstRowHeader = document.createElement("tr");

        firstRowHeader.innerHTML = `<th></th>
        <th colspan="${fullDataLength}">
            ${data.options?.total ? `<i class="bi bi-dash-square no-print" style="cursor:pointer" id="${formID}-toggle-columns"></i>` : ''}
            <h6 class="d-inline">${fields}</h6>
        </th>`;

        firstRowHeader.classList.add("table-light");
        mainHeader.prepend(firstRowHeader);

        const firstHeader = document.createElement("th");

        subHeader.append(firstHeader);


        subHeader.classList.add("sub-header")


        data.data.forEach((item, index) => {
            const th = document.createElement("th");

            th.innerHTML = `<h6>${item[fields]}</h6>`;
            subHeader.appendChild(th);
        });


        if (data.options?.total) {
            const totalHeader = document.createElement("th");

            totalHeader.innerHTML = `<h6>${Object.values(data.options.total)[0]}</h6>`;
            subHeader.appendChild(totalHeader);
        }


        values.forEach(value => {
            const row = document.createElement("tr");


            const rowLabel = document.createElement("td");
            rowLabel.classList.add("table-light");
            rowLabel.innerHTML = `<h6>${value}</h6>`;
            row.appendChild(rowLabel);


            data.data.forEach((item, index) => {
                const td = document.createElement("td");

                td.textContent = item[value] ?? "-";
                row.appendChild(td);
            });


            if (data.options?.total) {
                const totalValue = data.options.total[value] ?? 0;
                const totalTd = document.createElement("td");
                totalTd.innerHTML = `<span class="text-primary">${totalValue}</span>`;
                row.appendChild(totalTd);
            } else {
                const totalTd = document.querySelector(`#${formID} .grand-total td:nth-child(` + (fullDataLength + 1) + ")");
                if (totalTd) totalTd.remove();
            }

            tableBody.appendChild(row);
        });

        $(`#${formID}-toggle-columns`).click(function () {
            $(`#${formID} .sub-header, #${formID} tbody tr td:nth-child(n+2):nth-child(-n+${fullDataLength}), #${formID} .grand-total td:nth-child(n+2):nth-child(-n+${fullDataLength})`).toggle();
            var icon = $(this);
            icon.toggleClass("bi-plus-square bi-dash-square");
        });

        document.querySelectorAll(`#${formID} .grand-total td`).forEach(td => td.remove());

        changeAlignment(formID);

    }

    window.createPagination = function createPagination(totalPages, data, formID) {
        if (totalPages <= 1) return;

        const paginationContainer = document.querySelector(`#${formID} .pagination`);
        paginationContainer.innerHTML = "";
        const tableBody = document.querySelector(`#${formID} .table-body`); // Select your table body
        const rows = tableBody.querySelectorAll(`#${formID} tr`); // Get all rows
        const rowsPerPage = Math.ceil(rows.length / totalPages);

        // Create Previous button
        const prevPage = document.createElement("li");
        prevPage.classList.add("page-item");
        if (currentPage === 1) prevPage.classList.add("disabled");

        const prevLink = document.createElement("a");
        prevLink.classList.add("page-link");
        prevLink.href = "#";
        prevLink.innerHTML = "Previous";
        prevLink.addEventListener("click", function (event) {
            event.preventDefault();
            if (currentPage > 1) {
                currentPage--;
                updateVisibleRows();
                updatePaginationUI(formID);
            }
        });

        prevPage.appendChild(prevLink);
        paginationContainer.appendChild(prevPage);

        // Create page number buttons
        for (let i = 1; i <= totalPages; i++) {
            const pageItem = document.createElement("li");
            pageItem.classList.add("page-item");
            if (currentPage === i) pageItem.classList.add("active");

            const pageLink = document.createElement("a");
            pageLink.classList.add("page-link");
            pageLink.href = "#";
            pageLink.textContent = i;

            pageLink.addEventListener("click", function (event) {
                event.preventDefault();
                currentPage = i;
                updateVisibleRows();
                updatePaginationUI(formID);
            });

            pageItem.appendChild(pageLink);
            paginationContainer.appendChild(pageItem);
        }

        // Create Next button
        const nextPage = document.createElement("li");
        nextPage.classList.add("page-item");
        if (currentPage === totalPages) nextPage.classList.add("disabled");

        const nextLink = document.createElement("a");
        nextLink.classList.add("page-link");
        nextLink.href = "#";
        nextLink.innerHTML = "Next";
        nextLink.addEventListener("click", function (event) {
            event.preventDefault();
            if (currentPage < totalPages) {
                currentPage++;
                updateVisibleRows();
                updatePaginationUI(formID);
            }
        });

        nextPage.appendChild(nextLink);
        paginationContainer.appendChild(nextPage);

        // Function to update which rows are visible
        function updateVisibleRows() {
            const startIndex = (currentPage - 1) * rowsPerPage;
            const endIndex = Math.min(startIndex + rowsPerPage, rows.length);

            rows.forEach((row, index) => {
                if (index >= startIndex && index < endIndex) {
                    row.style.display = "";
                } else {
                    row.style.display = "none";
                }
            });
        }

        // Function to update pagination UI state
        function updatePaginationUI(formID) {
            const pageItems = paginationContainer.querySelectorAll(`#${formID} .page-item`);

            // Update Previous button
            if (currentPage === 1) {
                paginationContainer.querySelector(`#${formID} .page-item:first-child`).classList.add("disabled");
            } else {
                paginationContainer.querySelector(`#${formID} .page-item:first-child`).classList.remove("disabled");
            }

            // Update Next button
            if (currentPage === totalPages) {
                paginationContainer.querySelector(`#${formID} .page-item:last-child`).classList.add("disabled");
            } else {
                paginationContainer.querySelector(`#${formID} .page-item:last-child`).classList.remove("disabled");
            }

            // Update active page number
            pageItems.forEach((item, index) => {
                if (index === currentPage) {
                    item.classList.add("active");
                } else {
                    item.classList.remove("active");
                }
            });
        }

        // Initialize visible rows
        updateVisibleRows();
    }


    window.updateVisibleRows = function updateVisibleRows() {
        const startIndex = (currentPage - 1) * rowsPerPage;
        const endIndex = Math.min(startIndex + rowsPerPage, rows.length);

        rows.forEach((row, index) => {
            if (index >= startIndex && index < endIndex) {
                row.style.display = "";
            } else {
                row.style.display = "none";
            }
        });
    }

    // Function to update pagination UI state
    window.updatePaginationUI = function updatePaginationUI(formID) {
        const pageItems = paginationContainer.querySelectorAll(`#${formID} .page-item`);

        // Update Previous button
        if (currentPage === 1) {
            paginationContainer.querySelector(`#${formID} .page-item:first-child`).classList.add("disabled");
        } else {
            paginationContainer.querySelector(`#${formID} .page-item:first-child`).classList.remove("disabled");
        }

        // Update Next button
        if (currentPage === totalPages) {
            paginationContainer.querySelector(`#${formID} .page-item:last-child`).classList.add("disabled");
        } else {
            paginationContainer.querySelector(`#${formID} .page-item:last-child`).classList.remove("disabled");
        }

        // Update active page number
        pageItems.forEach((item, index) => {
            if (index === currentPage) {
                item.classList.add("active");
            } else {
                item.classList.remove("active");
            }
        });
    }


    window.updateSelectBox = function updateSelectBox(id, data, placeholder, placeholderSentence = null) {
        $(`select[id="${id}"]`).empty();

        if (placeholder)
            $(`select[id="${id}"]`).append(
                `<option value="">${placeholderSentence}</option>`
            );

        $.each(data, function (key, value) {
            $(`select[id="${id}"]`).append(
                '<option value="' + value + '">' + value[0].toUpperCase() + value.substring(1) + "</option>"
            );
        });

    }

    window.changeTheme = function changeTheme() {
        var themeLocation = $("#theme-selector").attr("data-theme-location");

        var theme = $("#theme-selector").val();
        if (theme == "bootstrap") {
            $("#theme-stylesheet").attr("href", "");
        }
        $("#theme-stylesheet").attr("href", `${themeLocation}/${theme}/bootstrap.css`);
    }


    window.changePivotTitle = function changePivotTitle() {

        $("#pivot-title").text($("#pivot-table-title").val());
    }

    window.createJoinListItem = function createJoinListItem(primaryTable, primaryKey, foreignTable, foreignKey, joinType) {

        primaryTable = $(`#${primaryTable.attr("id")} :selected`);
        primaryKey = $(`#${primaryKey.attr("id")} :selected`);
        foreignTable = $(`#${foreignTable.attr("id")} :selected`);
        foreignKey = $(`#${foreignKey.attr("id")} :selected`);
        joinType = $('input[name=join_type]:checked');

        text = prepareMessage(primaryTable.text() + " - " + foreignTable.text(), 20).fullMsg;

        return `
        <li class="px-2 list-item d-inline w-100"
            data-primaryTable="${primaryTable.val()}" data-primaryKey="${primaryKey.val()}"
            data-foreignTable="${foreignTable.val()}" data-foreignKey="${foreignKey.val()}"
            data-jointype="${joinType.val()}"
            data-name="${primaryTable.val()}.${primaryKey.val()}=${foreignTable.val()}.${foreignKey.val()}=${joinType.val()}"
            >
            <span role="button" class="modify-join label text-sm join-tables" data-toggle="tooltip"
            data-title="${primaryTable.text()} - ${foreignTable.text()}">
            ${text}
            </span>
            <div class="icon-container">
                <a role="button" data-toggle="tooltip" data-title="Remove this field from the pivot table." class="delete-join btn btn-default btn-sm" style="float: right; margin-left: 5px"><i class="fa fa-trash mr-0"></i></a>
                <a role="button" data-toggle="tooltip" data-title="Assign a label to this field." class="modify-join btn btn-default btn-sm" style="float: right;"><i class="fa fa-edit mr-0"></i></a>
            </div>
            <hr style="border: 0; border-top: 1px solid #ccc; margin: 5px 0;">
        </li>
        `;

    }

    window.applyValidations = function applyValidations(primaryTable, primaryKey, foreignTable, foreignKey) {

        var errorMessage = '';


        if (!primaryTable.val() || !foreignTable.val()) {
            errorMessage = "Ensure that both the Primary Table and the Foreign Table are selected before adding the Join.";
        } else if (!primaryKey.val() || !foreignKey.val()) {
            errorMessage = "Ensure that both the Primary Key and the Foreign Key are selected before adding the Join.";
        } else if (primaryTable.val() === foreignTable.val() && primaryKey.val() === foreignKey.val()) {
            errorMessage = "Please ensure that the specified Join is valid.";
        }

        return errorMessage;
    }




    window.toggleColumnVisibility3D = function toggleColumnVisibility3D(columnIndex, isVisible, formID) {
        const rows = document.querySelectorAll(`#${formID} table tr`);

        rows.forEach(row => {
            const cells = row.querySelectorAll(`#${formID} td, #${formID} .sub-header th`);
            const cell = cells[columnIndex];
            if (cell) {
                if (isVisible) {
                    cell.classList.remove('d-none');
                } else {
                    cell.classList.add('d-none');
                }
            }
        });

    }

    window.changeAlignment = function changeAlignment(formID) {
        let align = $(`#${formID} #alignment`).val();

        $(`#${formID} .table-preview table th, #${formID} .table-preview table td`).css("text-align", align);
    }

    window.updateSelectAllState = function updateSelectAllState(formID) {
        const dropdownMenu = document.querySelector(`#${formID} .dropdown-menu.data-filter-columns`);
        const checkboxes = dropdownMenu.querySelectorAll(`#${formID} .column-toggle`);
        const selectAllCheckbox = dropdownMenu.querySelector(`#${formID} .selectall`);

        if (selectAllCheckbox) {
            const allChecked = [...checkboxes].every(cb => cb.checked);
            selectAllCheckbox.checked = allChecked;
        }
    }


    window.showAllTableRows = function showAllTableRows(formID) {
        $(`#${formID} table tbody, #${formID} table tr, #${formID} table tr td`).each(function () {
            $(this).css("display", "");
        });
    }

    window.renderColumnVisibilityDropdown = function renderColumnVisibilityDropdown(formID, response) {

        const tableHeaders = document.querySelectorAll(`#${formID} .sub-header th`);
        const dropdownMenu = document.querySelector(`#${formID} .dropdown-menu.data-filter-columns`);


        dropdownMenu?.querySelectorAll("li:not(:first-child):not(.divider)")?.forEach(li => li.remove());

        tableHeaders?.forEach((header, index) => {

            if (index === 0 || (index === tableHeaders.length - 1 && response.options?.total)) {
                return;
            }

            const label = document.createElement("label");
            label.classList.add("form-check-label");
            label.innerHTML = header.querySelector("h6")?.firstChild?.textContent.trim().split("\n")[0] ||
                header.textContent || header.innerText;

            const checkbox = document.createElement("input");
            checkbox.type = "checkbox";
            checkbox.classList.add("column-toggle", "mr-1", "justone");
            checkbox.dataset.index = index;


            let isHidden = false;
            const rows = document.querySelectorAll("table tr");
            for (const row of rows) {
                if (row.children[index] && row.children[index].classList.contains('d-none')) {
                    isHidden = true;
                    break;
                }
            }

            checkbox.checked = !isHidden;


            const listItem = document.createElement("li");
            const link = document.createElement("a");
            link.href = "#";
            link.classList.add("option-link");


            label.prepend(checkbox);
            link.appendChild(label);
            listItem.appendChild(link);
            dropdownMenu?.appendChild(listItem);


            checkbox.addEventListener('change', function () {
                toggleColumnVisibility(index, this.checked, formID);
                updateSelectAllState(formID);
            });
        });


        const selectAllCheckbox = dropdownMenu?.querySelector(`#${formID} .selectall`);
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function () {
                const checkboxes = dropdownMenu.querySelectorAll(`#${formID} .column-toggle`);
                checkboxes.forEach(cb => {
                    cb.checked = this.checked;
                    toggleColumnVisibility(cb.dataset.index, this.checked, formID);
                });
            });

            updateSelectAllState(formID);
        }



        $(document).on("click", `#${formID} .selectall`, function () {

            if ($(this).is(':checked')) {
                $('.option').prop('checked', true);
                var total = $('input[name="options[]"]:checked').length;
                $(".select-text").html(' Deselect');
            } else {
                $('.option').prop('checked', false);
                $(".select-text").html(' Select');
            }
        });

        $(document).on("click", `#${formID} input[type='checkbox'].justone`, function () {
            var a = $(`#${formID} input[type='checkbox'].justone`);
            if (a.length == a.filter(":checked").length) {
                $('.selectall').prop('checked', true);
                $(".select-text").html(' Deselect');
            }
            else {
                $('.selectall').prop('checked', false);
                $(".select-text").html(' Select');
            }
        });


        $(document).on("shown.bs.dropdown", `#${formID} .dropdown`, function () {
            $(this).find(".dropdown-menu").css("transform", "translate3d(10px, 0px, 0px)");
        });

    }



    window.toggleColumnVisibility = function toggleColumnVisibility(columnIndex, isVisible, formID) {
        const table = document.querySelector("table");


        if (table.querySelectorAll(`th[data-year]`).length > 0) {
            window.toggleColumnVisibilityComplex3D(columnIndex, isVisible, formID)
        } else {
            window.toggleColumnVisibility3D(columnIndex, isVisible, formID)
        }
    }

    window.toggleColumnVisibilityComplex3D = function toggleColumnVisibilityComplex3D(columnIndex, isVisible, formID) {
        const table = document.querySelector(`#${formID} table`);
        if (!table) return;

        const yearToToggle = table.querySelectorAll(`#${formID} th[data-year],#${formID} td[data-year]`)[columnIndex]?.getAttribute("data-year");
        if (!yearToToggle) return;



        document.querySelectorAll(`#${formID} th[data-year='${yearToToggle}'], #${formID} td[data-year='${yearToToggle}']`).forEach(cell => {
            if (isVisible) {
                cell.classList.remove('d-none');
            } else {
                cell.classList.add('d-none');
            }
        });


        const yearHeader = document.querySelector(`#${formID} th.main-year[data-year='${yearToToggle}']`);
        if (yearHeader) {
            if (isVisible) {
                yearHeader.setAttribute("colspan", yearHeader.dataset.originalColspan || "1");
            } else {
                yearHeader.setAttribute("colspan", "1");
            }
        }
    }


    window.buildParametersBlocks = function buildParametersBlocks(data, formID, page = "create") {

        const parametersContainer = $(`#${formID} .parameters-preview-container`);

        parametersContainer.empty();

        if (data.options?.parameters && typeof data.options.parameters === 'object' && Object.keys(data.options.parameters).length > 0) {

            renderMultiSelectParameters(data.options.parameters, parametersContainer, formID);

            renderOtherParameters(data.options.parameters, parametersContainer, formID);
            renderButtons(data.options.parameters, parametersContainer, formID);

            parametersEvents(page, formID);

        }

    }


    const rowsPerPage = 100;
    let currentPage = 1;

    window.generateTable = function generateTable(data, formID = "pivot-table-form", page = 1) {

        const subHeader = document.querySelector(`#${formID} .sub-header`);
        const paginationContainer = document.querySelector(`#${formID} .pagination`);
        const mainHeader = document.querySelector(`#${formID} .main-header`);

        let subSubHeader = document.querySelector(`#${formID} .sub-sub-header`);

        if (!subSubHeader) {
            subSubHeader = document.createElement("tr");
            subSubHeader.classList.add("sub-sub-header");
            subSubHeader.classList.add("table-light");
            mainHeader.appendChild(subSubHeader);
        }

        subSubHeader.innerHTML = '';
        subHeader.innerHTML = "";
        paginationContainer.innerHTML = "";


        if (Object.keys(data.data).length === 0)
            $(`#${formID} #pivot-table`);

        const isComplexLayout = data.options?.complex === true;
        const isRowLayout = data.options?.row === true;
        const isColumnLayout = data.options?.column === true;


        showAllTableRows(formID);

        if (isComplexLayout) {
            window.renderComplex3DTable(data, formID, page);
        } else if (isColumnLayout && isRowLayout) {
            render3DTable(data, formID, page);
        } else if (isColumnLayout) {

            renderTableAsColumns(data, formID);
        } else if (isRowLayout) {
            renderTableAsRows(data, formID, page);
        }

        changeAlignment(formID);

    }



    window.renderMultiSelectParameters = function renderMultiSelectParameters(data, container, formID) {
        let parametersRow = $("<div>");
        let parameterItems = [];

        Object.entries(data)
            .filter(([key, value]) => value.type === 'multi-select')
            .map(([key, value]) => [key, value.data])
            .forEach(([key, value], index) => {

                if (index % 3 === 0 && index > 0) {
                    container.append(parametersRow);
                    parametersRow = $("<div>").addClass('mt-2');
                }

                let parameterItem = `
            <div class="btn-group-dropdown">
                <button style="margin-right:10px; " class="btn btn-light dropdown-toggle" type="button"
                    data-toggle="dropdown" aria-expanded="false">
                    ${key} - All
                </button>
                <ul class="dropdown-menu" style="width: 300px;">
                    <div class="parameters-info row pt-3 pl-3">
                        <div class="col" style="white-space: nowrap;">
                            <a class="selectall-parameters">Select All ${value.length}</a> - <a class="clearAll-parameters">Clear</a>
                        </div>
                        <div class="col">
                            <label for="" class="pl-3 pr-1 fw-normal display-numbers" style="color:gray;white-space: nowrap;">Displaying ${value.length}</label>
                        </div>
                    </div>
                    <div class="d-flex form-inputs mx-3 mb-2">
                        <input class="form-control search-parameters" type="text" placeholder="Search">
                        <i class="fas fa-search"></i>
                    </div>
                    <div class="parameters-list overflow-auto" data-name="${key}" style="height:100px">
                        ${Object.values(value).map((val, index) => `
                            <li>
                                <a class="option-link">
                                    <label class="form-check-label">
                                        <input type="checkbox" class="mr-1 justone" name="parametersValues[${key}][]" value="${val}" data-index="${index}" checked>${val}
                                    </label>
                                </a>
                            </li>
                        `).join('')}
                    </div>
                    <div class="text-center mt-3">
                        <button class="btn btn-primary px-5">Ok</button>
                    </div>
                </ul>
            </div>
        `;

                parameterItems.push(parameterItem);
                if (parameterItems.length === 3) {
                    parametersRow.append(parameterItems);
                    parameterItems = [];
                }

            });

        if (parameterItems.length > 0) {
            parametersRow.append(parameterItems);
        }

        container.append(parametersRow);

    }


    window.renderOtherParameters = function renderOtherParameters(data, container, formID) {
        let parametersRow = $("<div>");
        parametersRow.addClass("col-9 row")
        let parameterItems = [];

        Object.entries(data)
            .filter(([key, value]) => value.type !== 'multi-select')
            // .map(([key, value]) => [key, value.data])
            .forEach(([key, value], index) => {

                let type = value.type;
                let data = value.data;

                let parameterItem = getOtherParameterItem(type, key, data);

                parameterItems.push(parameterItem);


            });

        if (parameterItems.length > 0) {
            parametersRow.append(parameterItems);
        }

        container.append(parametersRow);


        let maxWidth = 0;

        // First, find the widest label
        $(`#${formID} .parameter-label`).each(function () {
            const width = $(this).outerWidth();
            if (width > maxWidth) {
                maxWidth = width;
            }
        });

        // Then apply that width to all labels
        $(`#${formID} .parameter-label`).css('width', (maxWidth + 5) + 'px');

    }


    window.renderButtons = function renderButtons(data, container) {
        let parametersRow = $("<div>");
        parametersRow.addClass("col-6 row")
        let parameterItems = [];

        let buttonItem = null;
        Object.entries(data).some(([key, value]) => {
            if (value.type !== 'multi-select') {
                buttonItem = `
            <div class="mt-2 row">
                <div class="col-9">
                    <button class="btn btn-primary filter-btn">
                        <i class="bi bi-funnel-fill"></i> Filter
                    </button>
                    <button class="btn btn-secondary show-all">
                        Show All
                    </button>
                </div>

            </div>
            `;
                return true;
            }
            return false;
        });

        if (buttonItem) {
            parameterItems.push(buttonItem);
        }

        if (parameterItems.length > 0) {
            parametersRow.append(parameterItems);
        }

        container.append(parametersRow);

    }


    window.getOtherParameterItem = function getOtherParameterItem(type, key, value) {

        switch (type) {
            case 'search-box':
                return `
                <div class="btn-group align-items-center mt-3">
                    <label class="fw-normal mr-3 parameter-label"> ${key}:</label>
                    <input class="form-control w-50" type="text" name="parametersValues[${key}]">
                </div>
            `;
            case 'numeric-search-box':
                return `
                    <div class="btn-group align-items-center mt-3">
                        <label class="fw-normal mr-3 parameter-label"> ${key}:</label>
                        <input class="form-control w-50" type="number" name="parametersValues[${key}]">
                    </div>
                `;
            case 'date-calendar':
                return `
                    <div class="btn-group align-items-center mt-3 date-calendar-parameter">
                        <label class="fw-normal mr-3 d-inline parameter-label"> ${key}:</label>

                        <input class="form-control d-inline from mr-2" type="date" name="parametersValues[${key}][from]">
                        <input class="form-control d-inline to" type="date" name="parametersValues[${key}][to]">
                    </div>
                `;
            case 'numeric-slider':
                return `
                        <div class="btn-group align-items-top mt-3">
                            <div>
                                <label class="fw-normal mr-3 parameter-label"> ${key}:</label>
                            </div>
                            <div class="w-100">
                                <div id="${key}-numeric-slider"></div>
                                <div class="slider-values">
                                    <span>Min: <span id="min-${key}-value">${value.min}</span></span>
                                    <span>Max: <span id="max-${key}-value">${value.max}</span></span>
                                    <input type="hidden" name="parametersValues[${key}]" value="min:${value.min},max:${value.max}">
                                </div>
                            </div>
                        </div>
                    `;
            case 'time-slider':
                return `
                <div class="btn-group align-items-top mt-3">
                    <div>
                        <label class="fw-normal mr-3 parameter-label"> ${key}:</label>
                    </div>
                    <div class="w-100">
                        <div id="${key}-time-slider"></div>
                        <div class="slider-values">
                            <input type="hidden" name="parametersValues[${key}]" value="min=00:00,max=23:59">
                        </div>
                    </div>
                </div>
            `;

            default:
                break;
        }

    }

    window.applyFilter = function applyFilter(formID, page = "create") {
        return new Promise((resolve, reject) => {
            let applyUrl = $(`#${formID}`).attr('data-apply');
            let form = document.getElementById(formID);
            let connection = document.getElementById("connection");
            let formObject = new FormData(form)
            if (page == "create")
                formObject.append("connection", connection.value);
            formObject.append("operation", "filter");

            let checkedValues = {};

            $(`#${formID} .dropdown-menu`).each(function () {
                let parameterName = $(this).closest('.btn-group').find('.dropdown-menu .parameters-list').attr('data-name');
                let values = $(this).find('.justone:checked').map(function () {
                    return this.value;
                }).get();
                checkedValues[parameterName] = values;
            });

            $.ajax({
                type: "POST",
                url: applyUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {
                    $('html, body').animate({
                        scrollTop: $("#pivot-table-preview").offset().top
                    }, 0);
                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (Object.keys(response.data).length === 0) {
                        $(`#${formID} #table-container`).hide();
                        $(`#${formID} .pagination`).hide();
                        $(`#${formID} #info-box`).show();
                    } else {
                        $(`#${formID} .apply-errors`).hide();
                        generateTable(response, formID);
                        $(`#${formID} #table-container`).show();
                        $(`#${formID} .table-preview`).show();
                        $(`#${formID} .pagination`).show();

                        $(`#${formID} #info-box`).hide();

                    }
                    $(`#${formID} .apply-success`).hide();
                    $(`#${formID} .apply-errors`).hide();

                    $('html, body').animate({
                        scrollTop: $(`#${formID} .table-preview`).offset().top
                    }, 0);
                    if (page == "create")
                        changeTheme();
                    changePivotTitle();

                    renderColumnVisibilityDropdown(formID, response);

                    resolve(true);

                },
                error: function (response) {
                    $("#info-box").hide();
                    var fields = ['row_headers', 'values_fields', 'joins'];
                    response = response.responseJSON;
                    if (response.errors) {
                        $("#table-container").hide();
                        $(".pagination").hide();
                        fields.push(...Object.keys(response.errors).filter(key => key.match(/^parametersValues\.[^.]+\.from$/)));
                        fields.push(...Object.keys(response.errors).filter(key => key.match(/^parametersValues\.[^.]+\.to$/)));

                        fields.every(error => {
                            if (response.errors[error] || error.match(/parametersValues\.\*\./)) {
                                $(`.apply-errors`).text(response.errors[error][0]).show();
                                $(`.apply-errors`).attr("title", response.errors[error][0]).show();

                                return false;
                            }
                            return true;

                        });
                        $(".apply-success").hide();
                        $(".apply-errors").show();
                    }

                    if (response.sqlErrors) {
                        $("#table-container").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                    <ul>
                        <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                        <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                    </ul>

                    If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();

                    }
                    reject(false);

                }, complete: function () {
                    $('#preview-loader').hide();

                }

            });
        })
    }


    window.applyFilterBeforeSave = function applyFilterBeforeSave() {
        return new Promise((resolve, reject) => {


            let applyUrl = $('#pivot-table-form').attr('data-apply');
            let form = document.getElementById("pivot-table-form");
            let connection = document.getElementById("connection");
            let formObject = new FormData(form)
            formObject.append("connection", connection.value);
            formObject.append("operation", "filter");

            let checkedValues = {};

            $('.dropdown-menu').each(function () {
                let parameterName = $(this).closest('.btn-group').find('.dropdown-menu .parameters-list').attr('data-name');
                let values = $(this).find('.justone:checked').map(function () {
                    return this.value;
                }).get();
                checkedValues[parameterName] = values;
            });

            $.ajax({
                type: "POST",
                url: applyUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {
                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (Object.keys(response.data).length === 0) {
                        $("#table-container").hide();
                        $(".pagination").hide();
                        $("#info-box").show();
                    } else {
                        $(`.apply-errors`).hide();
                        generateTable(response);
                        $("#table-container").show();
                        $(".table-preview").show();
                        $(".pagination").show();

                        $("#info-box").hide();

                    }
                    $(".apply-success").hide();
                    $(".apply-errors").hide();

                    changeTheme();
                    changePivotTitle();

                    resolve(true);

                },
                error: function (response) {
                    $('html, body').animate({
                        scrollTop: $("#pivot-table-preview").offset().top
                    }, 0);

                    $("#info-box").hide();
                    var fields = ['row_headers', 'values_fields', 'joins'];
                    response = response.responseJSON;
                    if (response.errors) {
                        $("#table-container").hide();
                        $(".pagination").hide();
                        fields.push(...Object.keys(response.errors).filter(key => key.match(/^parametersValues\.[^.]+\.from$/)));
                        fields.push(...Object.keys(response.errors).filter(key => key.match(/^parametersValues\.[^.]+\.to$/)));

                        fields.every(error => {
                            if (response.errors[error] || error.match(/parametersValues\.\*\./)) {
                                $(`.apply-errors`).text(response.errors[error][0]).show();
                                $(`.apply-errors`).attr("title", response.errors[error][0]).show();

                                return false;
                            }
                            return true;

                        });
                        $(".apply-success").hide();
                        $(".apply-errors").show();
                    }

                    if (response.sqlErrors) {
                        $("#table-container").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                    <ul>
                        <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                        <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                    </ul>

                    If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();

                    }
                    reject(false);

                }, complete: function () {
                    $('#preview-loader').hide();

                }

            });
        })
    }



    window.updatePivotTable = function updatePivotTable() {
        return new Promise((resolve, reject) => {
            let updateUrl;
            if ($("#pivot-table-form").attr("data-status") == "update")
                updateUrl = $('#pivot-table-form').attr('data-update');

            let form = document.getElementById("pivot-table-form");
            let connection = document.getElementById("connection");
            let formObject = new FormData(form);
            formObject.append("connection", connection.value);
            formObject.append("operation", "update");

            let checkedValues = {};

            $('.dropdown-menu').each(function () {
                let parameterName = $(this).closest('.btn-group').find('.dropdown-menu .parameters-list').attr('data-name');
                let values = $(this).find('.justone:checked').map(function () {
                    return this.value;
                }).get();
                checkedValues[parameterName] = values;
            });

            $.ajax({
                type: "POST",
                url: updateUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {

                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (response.success) {
                        $(".save-success").text(response.message).show();

                        $(".save-errors").hide();
                    }
                    if (response.demo) {
                        $(".save-errors").text(response.demo).show();
                        $(".save-success").hide();
                    }

                    resolve(true);

                },
                error: function (response) {
                    $("#info-box").hide();
                    response = response.responseJSON;

                    let saveFields = ["pivot_table_name"];

                    if (response.errors) {
                        saveFields.every(error => {
                            if (response.errors[error] || error.match(/parametersValues\.\*\./)) {
                                $(`.save-errors`).text(response.errors[error][0]).show();
                                $(`.save-errors`).attr("title", response.errors[error][0]).show();

                                return false;
                            }
                            return true;

                        });
                        $(".save-success").hide();
                        $(".save-errors").show();
                    }

                    if (response.sqlErrors) {
                        $("#table-container").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                    <ul>
                        <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                        <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                    </ul>

                    If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();

                    }
                    $('#preview-loader').hide();
                    reject(false);

                }, complete: function () {
                    $('#preview-loader').hide();

                }

            });
        })
    }


    window.setPivotUpdateUrl = function setPivotUpdateUrl(resourceId) {

        let fullUpdateUrl = $('#pivot-table-form').attr('data-update') + "/" + resourceId;
        updateUrl = fullUpdateUrl;
        $('#pivot-table-form').attr('data-update', fullUpdateUrl);
        $('#pivot-table-form').attr('data-status', "update")
    }


    window.parametersEvents = function parametersEvents(page, formID) {

        $('.selectall-parameters').on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).closest('.dropdown-menu').find('.justone').prop('checked', true);
        });

        $('.search-parameters').on('input', function (e) {
            e.preventDefault();
            e.stopPropagation();
            let searchValue = $(this).val().toLowerCase();

            let count = 0;

            $(".parameters-list li").each(function () {
                let text = $(this).text().trim().toLowerCase();
                $(this).toggle(text.includes(searchValue));
                if ($(this).css('display') !== 'none') {
                    count++;
                }
            });

            $(this).closest('.dropdown-menu').find('.display-numbers').text(`Displaying ${count}`);

        });

        $('.clearAll-parameters').on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();

            $(this).closest('.dropdown-menu').find('.justone').prop('checked', false);
        });


        $('.btn-group-dropdown .dropdown-toggle').on('click', function () {
            let dropdownMenu = $(this).next('.dropdown-menu');
            let checkboxes = dropdownMenu.find('.justone');

            dropdownMenu.data('initialState', checkboxes.map(function () {
                return this.checked;
            }).get());

            dropdownMenu.removeData('confirmed');
        });

        $('.option-link, .justone').on('click', function (e) {
            e.stopPropagation();
        });

        $('.btn-group-dropdown').on('hide.bs.dropdown', function (e) {
            let dropdownMenu = $(this).find('.dropdown-menu');
            let checkboxes = dropdownMenu.find('.justone');

            if (!dropdownMenu.data('confirmed')) {
                let initialState = dropdownMenu.data('initialState') || [];
                checkboxes.each(function (index) {
                    this.checked = initialState[index] || false;
                });
            }
        });

        $(`#${formID} .show-all`).on('click', function (e) {
            e.preventDefault();
            showAllInHomePage(formID);
            if (page != "show")
                window.applyPivotTable(formID, function () {
                    window.changeTheme();
                    window.changePivotTitle();
                });
        });

        $(`#${formID} .dropdown-menu .btn-primary, #${formID} .filter-btn`).on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();

            let dropdownMenu = $(this).closest('.dropdown-menu');
            let button = $(this).closest('.btn-group-dropdown').find('button.dropdown-toggle');

            dropdownMenu.data('confirmed', true);

            if (button.length)
                button.dropdown('toggle');


            let buttonText = button.text().split(" - ")[0];

            let checked = dropdownMenu.find('input:checkbox:checked');

            if (checked.length === dropdownMenu.find('input:checkbox').length) {
                button.text(`${buttonText} - All`);
            } else if (checked.length === 1) {
                button.text(`${buttonText} - ${checked.closest('label').text()}`);
            } else if (checked.length > 1) {
                button.text(`${buttonText} - Multiple`);
            } else if (checked.length === 0) {
                button.text(`${buttonText} - Null`);

            }

            applyFilter(formID, page);
        });

        $(document).on('keypress', `#${formID}input[type="number"]`, function (event) {
            if (event.key.toLowerCase() === 'e') {
                event.preventDefault();
            }
        });

        $(document).on('input', `#${formID} input[type="number"]`, function () {
            $(this).val($(this).val().replace(/e/gi, '')); // Removes 'e' or 'E'
        });

        handleNumericSlider(formID);
        handleTimeSlider(formID);

    }



    window.resetParameterContainer = function resetParameterContainer() {
        const parametersContainer = $(".parameters-preview-container");
        parametersContainer.empty();

    }

    window.applyPivotTable = function applyPivotTable(formID, callback = null) {
        return new Promise((resolve, reject) => {
            let applyUrl = $('#pivot-table-form').attr('data-apply');
            let form = document.getElementById("pivot-table-form");
            let connection = document.getElementById("connection");
            window.resetParameterContainer();

            let formObject = new FormData(form)
            formObject.append("connection", connection.value);
            formObject.append("operation", "apply");

            $.ajax({
                type: "POST",
                url: applyUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {
                    $('html, body').animate({
                        scrollTop: $("#pivot-table-preview").offset().top
                    }, 0);
                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (Object.keys(response.data).length === 0) {
                        $(".table-preview").hide();
                        $("#table-container").hide();
                        $("#info-box").show();
                    } else {
                        $(`.apply-errors`).hide();
                        generateTable(response, "pivot-table-form", 1);
                        $(".table-preview").show();
                        $("#table-container").show();
                        $("#info-box").hide();

                        renderColumnVisibilityDropdown("pivot-table-form", response);
                        buildParametersBlocks(response, "pivot-table-form");
                    }

                    $('html, body').animate({
                        scrollTop: $(".table-preview").offset().top
                    }, 0);

                    if (callback) {
                        callback();
                    }

                    // Resolve with true for success
                    resolve(true);
                },
                error: function (response) {
                    var fields = ['row_headers', 'values_fields', 'joins'];
                    response = response.responseJSON;
                    if (response.errors) {
                        $(".table-preview").hide();

                        fields.every(error => {
                            if (response.errors[error]) {
                                $(`.apply-errors`).text(response.errors[error][0]).show();
                                $(`.apply-errors`).attr("title", response.errors[error][0]).show();
                                return false;
                            }
                            return true;
                        });
                        $(".apply-success").hide();
                        $(".apply-errors").show();
                    }

                    if (response.sqlErrors) {
                        $(".table-preview").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                        <ul>
                            <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                            <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                        </ul>

                        If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                    `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();
                    }

                    // Reject with false for error
                    reject(false);
                },
                complete: function () {
                    $('#preview-loader').hide();
                }
            });
        });
    }

    window.savePivotTable = function savePivotTable() {
        return new Promise((resolve, reject) => {

            let storeUrl = $('#pivot-table-form').attr('data-store');
            let form = document.getElementById("pivot-table-form");
            let connection = document.getElementById("connection");
            let formObject = new FormData(form)
            formObject.append("connection", connection.value);
            formObject.append("operation", "save");

            let checkedValues = {};

            $('.dropdown-menu').each(function () {
                let parameterName = $(this).closest('.btn-group').find('.dropdown-menu .parameters-list').attr('data-name');
                let values = $(this).find('.justone:checked').map(function () {
                    return this.value;
                }).get();
                checkedValues[parameterName] = values;
            });

            $.ajax({
                type: "POST",
                url: storeUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {

                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (response.success) {
                        $(".save-success").text(response.message).show();

                        $(".save-errors").hide();
                    }

                    if (response.demo) {
                        $(".save-errors").text(response.demo).show();
                        $(".save-success").hide();
                    }


                    resolve({
                        status: true,
                        resource: response.resource
                    });

                },
                error: function (response) {
                    $("#info-box").hide();
                    response = response.responseJSON;

                    let saveFields = ["pivot_table_name"];

                    if (response.errors) {
                        saveFields.every(error => {
                            if (response.errors[error] || error.match(/parametersValues\.\*\./)) {
                                $(`.save-errors`).text(response.errors[error][0]).show();
                                $(`.save-errors`).attr("title", response.errors[error][0]).show();

                                return false;
                            }
                            return true;

                        });
                        $(".save-success").hide();
                        $(".save-errors").show();
                    }

                    if (response.sqlErrors) {
                        $("#table-container").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                    <ul>
                        <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                        <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                    </ul>

                    If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();

                    }
                    $('#preview-loader').hide();
                    reject(false);

                }, complete: function () {
                    $('#preview-loader').hide();

                }

            });
        })
    }

    window.resetParameterContainer = function resetParameterContainer() {
        const parametersContainer = $("#parameters-preview-container");
        parametersContainer.empty();

    }

    window.showPivotTable = function showPivotTable(callback = null) {
        return new Promise((resolve, reject) => {

            let applyUrl = $('#pivot-table-form').attr('data-apply');
            let form = document.getElementById("pivot-table-form");
            window.resetParameterContainer();

            let formObject = new FormData(form)
            formObject.append("operation", "apply");

            $.ajax({
                type: "POST",
                url: applyUrl,
                data: formObject,
                dataType: 'json',
                contentType: false,
                processData: false,
                beforeSend: function () {
                    $('html, body').animate({
                        scrollTop: $("#pivot-table-preview").offset().top
                    }, 0);
                    $('#preview-loader').show();
                },
                success: function (response) {
                    if (response.data && Object.keys(response.data).length === 0) {
                        $(".table-preview").hide();
                        $("#table-container").hide();
                        $("#info-box").show();
                    } else {
                        $(`.apply-errors`).hide();
                        generateTable(response);
                        $(".table-preview").show();
                        $("#table-container").show();
                        $("#info-box").hide();

                        renderColumnVisibilityDropdown("pivot-table-form", response)
                        buildParametersBlocks(response, "pivot-table-form", 'show');
                    }

                    $('html, body').animate({
                        scrollTop: $(".table-preview").offset().top
                    }, 0);

                    if (callback) {
                        callback();
                    }

                    // Resolve with true for success
                    resolve(true);
                },
                error: function (response) {
                    var fields = ['row_headers', 'values_fields', 'joins'];
                    response = response.responseJSON;
                    if (response.errors) {
                        $(".table-preview").hide();

                        fields.every(error => {
                            if (response.errors[error]) {
                                $(`.apply-errors`).text(response.errors[error][0]).show();
                                $(`.apply-errors`).attr("title", response.errors[error][0]).show();
                                return false;
                            }
                            return true;
                        });
                        $(".apply-success").hide();
                        $(".apply-errors").show();
                    }

                    if (response.message) {
                        $(`.table-preview`).hide();
                        $(`.apply-errors`).html(response.message).show();
                        $(`.apply-errors`).attr("title", response.message).show();
                        $(`.apply-errors`).attr("data-html", true).show();
                    }


                    if (response.sqlErrors) {
                        $(".table-preview").hide();
                        let sqlErrors = response.sqlErrors;

                        let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                        let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                        let fullHTMLMsg = `
                            <ul>
                                <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                                <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                            </ul>

                            If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                        `;

                        let fullErrorMsg =
                            `${fullDetailsMsg}

    ${fullSqlMsg}`;
                        $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                        $(`.apply-errors`).attr("title", fullErrorMsg).show();
                        $(`.apply-errors`).attr("data-html", true).show();
                    }

                    // Reject with false for error
                    reject(false);
                },
                complete: function () {
                    $('#preview-loader').hide();
                }
            });
        });
    }

    window.showAllInHomePage = function showAllInHomePage(formID, reloadedPart, url = window.location.href) {

        let formEl = document.getElementById(formID);
        let applyUrl = formEl.getAttribute('data-apply');
        window.resetParameterContainer();

        let formObject = new FormData(formEl);
        formObject.append("operation", "apply");
        $.ajax({
            type: "POST",
            url: applyUrl,
            data: formObject,
            dataType: 'json',
            contentType: false,
            processData: false,
            success: function (response) {
                $(`#${formID} .apply-errors`).hide();
                generateTable(response, formID, 1);
                $(`#${formID} .table-preview`).show();
                $(`#${formID} #table-container`).show();
                $(`#${formID} #info-box`).hide();

                renderColumnVisibilityDropdown(formID, response);
                buildParametersBlocks(response, formID, 'show');


                $('html, body').animate({
                    scrollTop: $(`#${formID} .table-preview`).offset().top
                }, 0);
            },

        });

    }



    window.showInHomePivotTable = function showInHomePivotTable(callback = null) {
        const forms = $('.pivot-table-form').toArray();
        const promises = forms.map(formEl => {
            return new Promise((resolve, reject) => {
                let applyUrl = formEl.getAttribute('data-apply');
                let actualForm = document.getElementById(formEl.getAttribute('id'));
                window.resetParameterContainer();

                let formObject = new FormData(actualForm);
                formObject.append("operation", "apply");
                $.ajax({
                    type: "POST",
                    url: applyUrl,
                    data: formObject,
                    dataType: 'json',
                    contentType: false,
                    processData: false,
                    success: function (response) {
                        if (Object.keys(response.data).length === 0) {
                            $(`#${formEl.getAttribute('id')} .table-preview`).hide();
                            $("#table-container").hide();
                            $("#info-box").show();
                        } else {
                            $(`.apply-errors`).hide();
                            generateTable(response, formEl.getAttribute('id'), 1);
                            $(`#${formEl.getAttribute('id')} .table-preview`).show();
                            $("#table-container").show();
                            $("#info-box").hide();

                            renderColumnVisibilityDropdown(formEl.getAttribute('id'), response);
                            buildParametersBlocks(response, formEl.getAttribute('id'), 'show');
                        }

                        if (callback) {
                            callback();
                        }

                        resolve(true);
                    },
                    error: function (response) {
                        const fields = ['row_headers', 'values_fields', 'joins'];
                        response = response.responseJSON;
                        // if (response.errors) {
                        //     $(`#${formEl.getAttribute('id')} .table-preview`).hide();

                        //     fields.every(field => {
                        //         if (response.errors[field]) {
                        //             $(`.apply-errors`).text(response.errors[field][0]).show();
                        //             $(`.apply-errors`).attr("title", response.errors[field][0]).show();
                        //             return false;
                        //         }
                        //         return true;
                        //     });

                        //     $(".apply-success").hide();
                        //     $(".apply-errors").show();
                        // }

                        if (response.message || response.has('exception')) {
                            $(`#${formEl.getAttribute('id')} .table-preview`).hide();
                            $(`#${formEl.getAttribute('id')} .show-errors`).html(response.message).show();
                            $(`#${formEl.getAttribute('id')} .show-errors`).attr("title", response.message).show();
                            $(`#${formEl.getAttribute('id')} .show-errors`).attr("data-html", true).show();
                            $(`#${formEl.getAttribute('id')} .export`).hide();
                        }

                        if (response.sqlErrors) {

                            $(`#${formEl.getAttribute('id')} .table-container`).hide();
                            let sqlErrors = response.sqlErrors;

                            let fullDetailsMsg = `Error Details: ${sqlErrors.details}`;
                            let fullSqlMsg = `SQL Statement: ${sqlErrors.sql}`;

                            let fullHTMLMsg = `
                                <ul>
                                    <li data-toggle="tooltip" data-title="${fullDetailsMsg}">${fullDetailsMsg}</li>
                                    <li data-toggle="tooltip" data-title="${fullSqlMsg}">${fullSqlMsg}</li>
                                </ul>
                                If the issue persists, please verify your query structure and ensure all selected fields and joins are correctly defined.
                            `;

                            let fullErrorMsg = `${fullDetailsMsg}\n\n${fullSqlMsg}`;
                            $(`.apply-errors`).removeClass('text-center').html(fullHTMLMsg).show();
                            $(`.apply-errors`).attr("title", fullErrorMsg).show();
                            $(`.apply-errors`).attr("data-html", true).show();
                        }
                        $(`#${formEl.getAttribute('id')} .table-preview`).hide();

                        reject(false);
                    },
                    complete: function () {
                        $('#preview-loader').hide();
                    }
                });
            });
        });

        return Promise.all(promises);
    }





    window.destructParameterBlock = function destructParameterBlock(field) {
        const parts = field.split('.');
        const type = parts[0];
        const table = parts[1];
        const column = parts[2];

        const [label, func] = parts[3].split(':');

        return {
            type: type,
            table: table,
            column: column,
            label: label,
            function: func
        };
    }

    window.destructParametersBlocks = function destructParametersBlocks(fields) {
        if (!fields || typeof fields !== 'string' || fields.trim() === '') {
            return null;
        }

        const blocks = fields.split(', ');
        const final = blocks.map(block => destructParameterBlock(block));

        return final;
    }


    window.getSelectedFunctionLabel = function getSelectedFunctionLabel(value) {

        switch (value) {
            case 'multi-select':
                return "Multi-Select Filter"
            case 'search-box':
                return "Search Box"
            case 'numeric-search-box':
                return "Numeric Search Box"
            case 'numeric-slider':
                return "Numeric Slider"
            case 'date-calendar':
                return "Date Calendar"
            case 'time-slider':
                return "Time Slider"
            default:
                break;
        }

    }

    window.buildParametersDropdownMenuInEdit = function buildParametersDropdownMenuInEdit(closestList, datatype) {
        datatype = datatype.toLowerCase();

        let booleanAndEnum = ["bit", "boolean", "tinyint(1)", "enum", "set"];
        let numerical = ["integer", "int", "tinyint", "smallint", "mediumint", "bigint", "decimal", "numeric", "float", "double", "year"];
        let text = ["varchar", "text", "char", "tinytext", "mediumtext", "longtext", "string"];
        let date = ["date", "datetime", "timestamp"];
        let time = ["time"];

        if (closestList === 'parameters') {
            let options = "";

            if (booleanAndEnum.includes(datatype)) {
                options = `<li><a class="parameters_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>`;
            } else if (datatype === 'json') {
                options = `<li><a class="parameters_function" data-action="search-box" href="#">Search Box</a></li>`;
            } else if (text.includes(datatype)) {
                options = `
                    <li><a class="parameters_function" data-action="search-box" href="#">Search Box</a></li>
                    <li><a class="parameters_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>
                `;
            } else if (numerical.includes(datatype)) {
                options = `
                    <li><a class="parameters_function" data-action="numeric-search-box" href="#">Numeric Search Box</a></li>
                    <li><a class="parameters_function" data-action="numeric-slider" href="#">Numeric Slider</a></li>
                    <li><a class="parameters_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>
                `;
            } else if (date.includes(datatype)) {
                options = `
                    <li><a class="parameters_function" data-action="date-calendar" href="#">Date Calendar</a></li>
                    <li><a class="parameters_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>
                `;
            } else if (time.includes(datatype)) {
                defaultSelectedDataType = "Time Slider";
                options = `
                    <li><a class="parameters_function" data-action="time-slider" href="#">Time Slider</a></li>
                    <li><a class="parameters_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>
                `;
            }

            return `
                <ul class="dropdown-menu dropdown-menu-parameters text-sm">
                    ${options}
                </ul>
            `;
        }

        return `
            <div class="position-relative z-3">
                <div class="position-fixed">
                    <ul class="dropdown-menu text-sm">
                        <li><a class="column_function" data-action="multi-select" href="#">Multi-Select Filter</a></li>
                    </ul>
                </div>
            </div>
        `;
    }


    window.createParameterListItemInEdit = function createParameterListItemInEdit(id, text, table, datatype, label, closestList, _function) {

        let deleteTooltip = "Remove this field from the pivot table.";
        let editTooltip = "Assign a label to this field.";

        let selectedFunction = getSelectedFunctionLabel(_function);

        if (preventColumnDropWhenTableNotExists(table, id))
            return;

        if (preventDuplicateDrop(closestList, id)) {
            return;
        }

        return `
        <li class="px-2 list-item d-inline w-100"
            data-id="${id}" data-name="${text.original}" data-label="${text.original}"
            data-type="${datatype}" data-parent="${parent}" data-table="${table}"
            data-function="${_function}">
            <span role="button" class="label text-sm column-name" data-tooltip="tooltip" data-title="${text.original}">
            ${text.truncated}
            </span>
            <span role="button" class="dropdown-toggle label text-sm function" data-toggle="dropdown">
                [${selectedFunction}]
            </span>

            ${buildParametersDropdownMenuInEdit(closestList, datatype)}
            <div class="icon-container">
                <a role="button" data-toggle="tooltip" data-title="${deleteTooltip}" class="delete-icon btn btn-default btn-sm" style="float: right; margin-left: 5px"><i class="fa fa-trash mr-0"></i></a>
                <a role="button" data-toggle="tooltip" data-title="${editTooltip}" class="modify-field btn btn-default btn-sm" style="float: right;"><i class="fa fa-edit mr-0"></i></a>
            </div>
            <hr style="border: 0; border-top: 1px solid #ccc; margin: 5px 0;">
        </li>
        `;

    }


    window.rebuildParameterListItem = function rebuildParameterListItem(textbox, closestList) {

        if ($(`#${textbox}`).val() == "") {
            return
        }

        let listItems = destructParametersBlocks($(`#${textbox}`).val());

        listItems.forEach(listItem => {

            let text = {
                original: listItem.label,
                truncated: prepareMessage(listItem.label, 6).fullMsg
            };

            let Item = createParameterListItemInEdit(
                `${listItem.table}_${listItem.label}`,
                text,
                listItem.table,
                listItem.type,
                listItem.label,
                closestList,
                listItem.function
            );

            $(`#${closestList}`).append(Item);

            bindModifyFieldEvent();

            bindDeleteEvent(closestList);

            toggleFieldLabels(closestList);


        });

    }



    window.getSelectedLabel = function getSelectedLabel(value) {
        switch (value) {
            case 'raw':
                return "Raw value"
            case 'count':
                return "Count"
            case 'sum':
                return "Sum"
            case 'count-distinct':
                return "Count Distinct"
            case 'average':
                return "Average"
            case 'min':
                return "Min"
            case 'max':
                return "Max"
            case 'year':
                return "Year"
            case 'quarter':
                return "Quarter"
            case 'month':
                return "Month"
            case 'week':
                return "Week"
            case 'weekday':
                return "Weekday"
            case 'day':
                return "Day"
            case 'hour':
                return "Hour"

            default:
                break;
        }
    }



    window.destructFieldBlocks = function destructFieldBlocks(fields) {
        if (fields === null || fields.trim() === '') {
            return null;
        }

        const blocks = fields.split(', ');
        const final = blocks.map(block => destructFieldBlock(block));

        return final;
    }

    window.destructFieldBlock = function destructFieldBlock(field) {
        const parts = field.split('.');
        const type = parts[0];
        const table = parts[1];
        const column = parts[2];

        const [label, func] = parts[3].split(':');

        return {
            type: type,
            table: table,
            column: column,
            label: label,
            function: func
        };
    }



    window.createFieldListItemInEdit = function createFieldListItemInEdit(id, text, table, datatype, label, closestList, enumValues, _function) {

        deleteToolTip = 'Remove this field from the pivot table.';
        editTooltip = 'Assign a label to this field.';

        isValuesDropMenu = closestList == "values-fields" ? true : false;
        SelectedDataType = getSelectedLabel(_function);

        return `
        <li class="px-2 list-item d-inline w-100"
            data-id="${id}" data-name="${text.original}" data-label="${label}"
            data-type="${datatype}" data-parent="${table}" data-table="${table}"
            data-function="${_function}" data-enum="${enumValues}">
            <span role="button" class="label text-sm column-name" data-tooltip="tooltip" data-title="${text.original}">
            ${text.truncated}
            </span>
            <span role="button" class="dropdown-toggle label text-sm function" data-toggle="dropdown">
            ${' [' + SelectedDataType + ']'}
            </span>
            ${isValuesDropMenu ? buildValuesDropdownMenu(closestList, datatype) : buildDropdownMenu(closestList, datatype)}
            <div class="icon-container">
                <a role="button" data-toggle="tooltip" data-title="${deleteToolTip}" class="delete-icon btn btn-default btn-sm" style="float: right; margin-left: 5px"><i class="fa fa-trash mr-0"></i></a>
                <a role="button" data-toggle="tooltip" data-title="${editTooltip}" class="modify-field btn btn-default btn-sm" style="float: right;"><i class="fa fa-edit mr-0"></i></a>
            </div>
            <hr style="border: 0; border-top: 1px solid #ccc; margin: 5px 0;">
        </li>
        `;

    }

    window.rebuildFieldListItem = function rebuildFieldListItem(textbox, closestList) {

        if ($(`#${textbox}`).val() == "") {
            return
        }

        let listItems = destructFieldBlocks($(`#${textbox}`).val());

        let text = {
            original: listItems[0].label,
            truncated: prepareMessage(listItems[0].label, 6).fullMsg
        };

        let listItem = createFieldListItemInEdit(
            `${listItems[0].table}_${listItems[0].label}`,
            text,
            listItems[0].table,
            listItems[0].type,
            listItems[0].label,
            closestList,
            "enumValues",
            listItems[0].function
        );

        $(`#${closestList}`).append(listItem);

        bindModifyFieldEvent();

        bindDeleteEvent(closestList);

        toggleFieldLabels(closestList);

    }

    window.destructFilter = function destructFilter(filter) {
        const final = {};
        const parts = filter.split('&');

        for (const part of parts) {
            const [key, value] = part.split('=');
            final[key] = value;
        }

        return final;
    }

    window.destructFilters = function destructFilters(filters) {
        if (!filters || typeof filters !== 'string' || filters.trim() === '') {
            return null;
        }

        const blocks = filters.split('; ');
        return blocks.map(f => destructFilter(f));
    }


    window.createStaticFilterItemInEdit = function createStaticFilterItemInEdit(id, text, table, datatype, values) {

        let exists = $("#static-filters li[data-name='" + text.original + "']").length > 0;
        let index = exists ? $("#static-filters li[data-name='" + text.original + "']").length + 1 : 1;

        let attributes = '';
        if (values.min !== null) attributes += ` data-min="${values.min}"`;
        if (values.max !== null) attributes += ` data-max="${values.max}"`;
        if (values.value !== null) attributes += ` data-value="${values.value}"`;
        if (values.criteria !== null) attributes += ` data-criteria="${values.criteria}"`;
        if (values.typeOfRange !== null) attributes += ` data-type-of-range="${values.typeOfRange}"`;
        if (values.relative !== null) attributes += ` data-relative="${values.relative}"`;
        if (values.fixed !== null) attributes += ` data-fixed="${values.fixed}"`;
        if (values.enumValues !== null) attributes += ` data-enum="${values.enumValues}"`;
        if (values.selectType !== null) attributes += ` data-select-type="${values.selectType}"`;

        return `
        <li class="px-2 list-item d-inline w-100"
            data-id="${id}_Filter_${index}" data-name="${text.original}" data-label="${text.original}"
            data-type="${datatype}" data-table="${table}" ${attributes}>
            <span role="button" class="label text-sm column-name modify-static-filter" data-tooltip="tooltip" data-title="${text.original}_Filter_${index}"
            data-index="${index}">

            ${text.truncated}_Filter_${index}

            </span>
            <div class="icon-container">
                <a role="button" data-toggle="tooltip" data-title="Remove this field from the pivot table." class="delete-icon btn btn-default btn-sm" style="float: right; margin-left: 5px"><i class="fa fa-trash mr-0"></i></a>
                <a role="button" data-toggle="tooltip" data-title="edit static filters." class="modify-static-filter btn btn-default btn-sm" style="float: right;"><i class="fa fa-edit mr-0"></i></a>
            </div>
            <hr style="border: 0; border-top: 1px solid #ccc; margin: 5px 0;">
        </li>
        `;

    }

    window.rebuildStaticFilterItems = function rebuildStaticFilterItems() {

        let listItems = destructFilters($(`#static-filters-textbox`).val());


        if ($(`#static-filters-textbox`).val() == "") {
            return
        }


        listItems.forEach(listItem => {

            let text = {
                original: listItem.column,
                truncated: prepareMessage(listItem.column, 6).fullMsg
            };

            let values = {};
            if (listItem.value !== undefined) values.value = listItem.value;
            if (listItem.max !== undefined) values.max = listItem.max;
            if (listItem.min !== undefined) values.min = listItem.min;
            if (listItem.criteria !== undefined) values.criteria = listItem.criteria;
            if (listItem.typeOfRange !== undefined) values.typeOfRange = listItem.typeOfRange;
            if (listItem.relative !== undefined) values.relative = listItem.relative;
            if (listItem.fixed !== undefined) values.fixed = listItem.fixed;
            if (listItem.enumValues !== undefined) values.enumValues = listItem.enumValues;
            if (listItem.selectType !== undefined) values.selectType = listItem.selectType;

            let item = createStaticFilterItemInEdit(
                `${listItem.table}_${listItem.label}`,
                text,
                listItem.table,
                listItem.type,
                values
            );

            $("#static-filters").append(item);

            toggleFieldLabels("static-filters");
            initializeTooltip();
            bindModifyFieldEvent();

            bindDeleteEvent("static-filters");

            toggleFieldLabels("static-filters");

            updateStaticFilterTextBox("static-filters", `static-filters-textbox`);


        });
    }

    window.destructJoin = function destructJoin(join) {
        const parts = join.split('=');
        const left = parts[0].split('.');
        const right = parts[1].split('.');
        const type = parts[2];

        return {
            left: left[0],
            columnLeft: left[1],
            right: right[0],
            columnRight: right[1],
            type: type
        };
    }

    window.destructJoins = function destructJoins(joins) {
        if (!joins || typeof joins !== 'string' || joins.trim() === '') {
            return null;
        }

        const blocks = joins.split(', ');
        return blocks.map(j => destructJoin(j));
    }


    window.createJoinListItemInEdit = function createJoinListItemInEdit(primaryTable, primaryKey, foreignTable, foreignKey, joinType) {

        text = prepareMessage(primaryTable + " - " + foreignTable, 20).fullMsg;

        return `
        <li class="px-2 list-item d-inline w-100"
            data-primaryTable="${primaryTable}" data-primaryKey="${primaryKey}"
            data-foreignTable="${foreignTable}" data-foreignKey="${foreignKey}"
            data-jointype="${joinType}"
            data-name="${primaryTable}.${primaryKey}=${foreignTable}.${foreignKey}=${joinType}">
            <span role="button" class="modify-join label text-sm join-tables" data-toggle="tooltip"
            data-title="${primaryTable} - ${foreignTable}">
            ${text}
            </span>
            <div class="icon-container">
                <a role="button" data-toggle="tooltip" data-title="Remove this field from the pivot table." class="delete-join btn btn-default btn-sm" style="float: right; margin-left: 5px"><i class="fa fa-trash mr-0"></i></a>
                <a role="button" data-toggle="tooltip" data-title="edit join." class="modify-join btn btn-default btn-sm" style="float: right;"><i class="fa fa-edit mr-0"></i></a>
            </div>
            <hr style="border: 0; border-top: 1px solid #ccc; margin: 5px 0;">
        </li>
        `;
    }

    window.rebuildJoinsItems = function rebuildJoinsItems() {
        let listItems = destructJoins($(`#joins-textbox`).val());

        if ($(`#joins-textbox`).val() == "") {
            return
        }


        listItems.forEach(listItem => {

            let item = createJoinListItemInEdit(
                listItem.left,
                listItem.columnLeft,
                listItem.right,
                listItem.columnRight,
                listItem.type
            );

            $(`#joins`).append(item);
            updateJoinTextBox("joins", `joins-textbox`);

            initializeTooltip()
        });

    }

    window.clearPivotTable = function clearPivotTable() {
        $("#row-headers").empty();
        $("#column-headings").empty();
        $("#values-fields").empty();
        $("#joins").empty();
        $("#static-filters").empty();
        $("#parameters").empty();
        $(".drop-field-label").show();
        $(".drop-field-label-first-row").show();
        $("input[readonly]").val("");
        $(".apply-errors").hide();
        $(".table-preview").css("display", "none");
        $('#show-grand-totals').prop('checked', true);

        $("#category").val($("#category option:first").val());
        $("#grand-totals-function").val($("#grand-totals-function option:first").val());
        $("#combine-filters").val($("#combine-filters option:first").val());
        $("#access-level").val($("#access-level option:first").val());
        $("#theme-selector").val($("#theme-selector option:first").val());
        $("#alignment").val($("#alignment option:first").val());
        $("#values-format").val($("#values-format option:first").val());
        $(".currency").hide();
        $(".custom-unit").hide();

        existingJoins.clear();
        
        $("#pivot-table-title").val("Pivot table title");
        $("#grand-totals-label").val("Total");

    }


    window.clearPivotTableAndAddNew = function clearPivotTableAndAddNew() {
        $("#row-headers").empty();
        $("#column-headings").empty();
        $("#values-fields").empty();
        $("#joins").empty();
        $("#static-filters").empty();
        $("#parameters").empty();
        $(".drop-field-label").show();
        $(".drop-field-label-first-row").show();
        $("input[readonly]").val("");
        $(".apply-errors").hide();
        $(".table-preview").css("display", "none");
        $('#show-grand-totals').prop('checked', true);


        $("#category").val($("#category option:first").val());
        $("#grand-totals-function").val($("#grand-totals-function option:first").val());
        $("#combine-filters").val($("#combine-filters option:first").val());
        $("#access-level").val($("#access-level option:first").val());
        $("#theme-selector").val($("#theme-selector option:first").val());
        $("#alignment").val($("#alignment option:first").val());
        $("#values-format").val($("#values-format option:first").val());
        $(".currency").hide();
        $(".custom-unit").hide();


        $("#pivot-table-title").val("Pivot table title");
        $("#grand-totals-label").val("Total");

        existingJoins.clear();

        $(".pivot-table-name").text("Pivot_Table1");

        let url = $("#pivot-table-form").attr("data-update");
        if (url) {
            let urlParts = url.split("/");
            urlParts.pop(); // Remove the last segment
            $("#pivot-table-form").attr("data-update", urlParts.join("/"));
        }

        $("#pivot-table-form").removeAttr('data-status');
        $(".save-success").hide();
        $(".save-errors").hide();

    }


});
