<?php

namespace App\Http\Requests;

use App\Models\Group;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Http\FormRequest;

class updateUserRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $prefix = Config('srm_config.installer.table_prefix', 'srm_');
        return [
            'first_name' => ['required', 'min:3', 'max:50'],
            'last_name' => ['required', 'min:3', 'max:50'],
            'user_name' => ['required', 'min:3', 'max:50', Rule::unique("{$prefix}Users", 'user_name')->ignore($this->route('user')->user_ID, 'user_ID')],
            'role' => ['required', 'in:user,admin'],
            "Phone" => ["nullable", "string", "max:50"],
            'Email' => ['required', 'email', Rule::unique("{$prefix}Users", 'Email')->ignore($this->route('user')->user_ID, 'user_ID')],
            'group' => [
                'required_if:role,user', function ($attribute, $value, $fail) {

                    $group = Group::find($value);

                    if ($this->input('role') === "admin" && isset($value)) {
                        $fail("Admin users can't have a group");
                    } elseif (
                        $this->input('role') === "user" &&
                        (!isset($value) || !$group)
                    ) {
                        $fail("Please select a group of this user");
                    }
                },
            ],
            'status' => ['required', 'in:active,pended,blocked'],
            'email_confirm' => ['required', function ($attribute, $value, $fail) {

                if (
                    $this->input('send_activation') === null && $value == "no"
                ) {
                    $fail("Please check the \"Send email activation\" box since you specified that \"User didn't verify his email\".");
                } elseif (
                    $this->input('send_activation') === "on" && $value == "yes"
                ) {
                    $fail("Please uncheck the \"Send email activation\" box since you specified that \"User  verified his email\".");
                }
            }],
            'reset_password' => ['nullable'],
            'send_activation' => ['nullable'],
        ];
    }


    public function messages()
    {
        return [
            "group.required_if" => "Please select a group of this user",
            "status.required" => "Please select the status of this user",
            "role.required" => "Please select the role of this user",
            "*.required" => ":attribute can't be empty.",
            "Email.email" => "Please enter a valid email address",
            "*.unique" => ":attribute is already existed",
            "*.min" => "The :attribute must be at least :min characters.",
            "*.max" => "The :attribute may not be greater than :max characters.",
            "*.digits_between" => "The :attribute may not be greater than :max characters.",
        ];
    }
}
