<?php

namespace App\Http\Controllers;

use App\Models\Module;
use App\Models\Resource;
use App\Services\UserService;
use App\Services\GroupService;
use App\Services\CategoryService;
use App\Services\PivotTableService;
use App\Services\DataConnectionService;
use App\Http\Middleware\OnlyFullVersion;
use App\Http\Requests\UpdatePivotTableCategoryRequest;
use App\Http\Requests\PivotTable\pivotTableShowRequest;

class PivotTableController extends Controller
{

    private $dataConnectionService;
    private $categoryService;
    private $groupService;
    public $userService;
    public $pivotService;

    public function __construct(
        DataConnectionService $dataConnectionService,
        CategoryService $categoryService,
        GroupService $groupService,
        UserService $userService,
        PivotTableService $pivotService
    ) {
        $this->dataConnectionService = $dataConnectionService;
        $this->categoryService       = $categoryService;
        $this->groupService          = $groupService;
        $this->userService           = $userService;
        $this->pivotService          = $pivotService;

        $this->middleware(OnlyFullVersion::class)->only([
            'changeCategory',
            'destroy'
        ]);

    }

    public function index()
    {
        $categories       = $this->categoryService->getCategories();
        $users            = $this->userService->getAllUsers();
        $pivotTableModule = Module::getPivotTableModule();

        $pivotTables = $this->pivotService->getPivotTables();

        return view('dashboard.pivot-tables.index', compact('categories', 'users', 'pivotTables', 'pivotTableModule'));
    }

    public function create()
    {
        $connections = $this->dataConnectionService->getAllConnections();
        $url         = $this->pivotService->create();

        $categories = $this->categoryService->getCategories();
        $groups     = auth()->user()->isAdminOrOwner() ? $this->groupService->getAllGroups() : [auth()->user()->group];
        return view('dashboard.pivot-tables.create', compact('connections', 'categories', 'groups'));
    }

    public function show(Resource $pivotTable)
    {
        $pivotTable = $this->pivotService->show($pivotTable);

        if (is_null(auth()->user()) && strtolower($pivotTable->access_control_type) == "public") {
            return view('dashboard.pivot-tables.show-public', compact('pivotTable'));
        }

        return view('dashboard.pivot-tables.show', compact('pivotTable'));
    }

    public function edit(Resource $pivotTable)
    {
        $connections = $this->dataConnectionService->getAllConnections();
        $categories  = $this->categoryService->getCategories();
        $groups      = auth()->user()->isAdminOrOwner() ? $this->groupService->getAllGroups() : [auth()->user()->group];

        $url = $this->pivotService->edit($pivotTable);

        return view('dashboard.pivot-tables.edit', compact('pivotTable', 'connections', 'categories', 'groups'));
    }

    public function build(pivotTableShowRequest $request, Resource $pivotTable)
    {
        $buildResult = $this->pivotService->build($pivotTable, $request);

        if (is_array($buildResult) && count($buildResult) === 2 && isset($buildResult[0], $buildResult[1])) {
            [$pivotTableData, $statusCode] = $buildResult;
        } else {
            $pivotTableData = $buildResult;
            $statusCode     = 200;
        }

        return response()->json($pivotTableData, $statusCode);

    }

    public function getCategories(Resource $pivotTable)
    {
        $categories = $this->categoryService->getCategories();

        return view('dashboard.pivot-tables.change-category', compact('pivotTable', 'categories'));
    }

    public function changeCategory(Resource $pivotTable, UpdatePivotTableCategoryRequest $request)
    {
        $this->pivotService->changePivotTableCategory($pivotTable, $request->validated());
        return response()->json([
            'message' => trans('pivot-tables.messages.update')
        ]);
    }

    public function destroy(Resource $pivotTable)
    {
        $this->pivotService->destroy($pivotTable);

        return response()->json([
            'message' => trans('pivot-tables.messages.delete', [
                'report' => $pivotTable->name
            ])
        ]);
    }

}
