<?php

namespace App\Http\Controllers;

use App\Models\HomeDashboard;
use App\Services\HomeService;
use App\Services\ResourceService;
use App\Http\Middleware\OnlyFullVersion;
use App\Http\Requests\StoreHomeWidgetRequest;
use App\Http\Requests\UpdateHomeWidgetRequest;

class HomeController extends Controller
{
    private $homeService;
    private $resourceService;

    public function __construct(HomeService $homeService, ResourceService $resourceService)
    {
        $this->homeService     = $homeService;
        $this->resourceService = $resourceService;

        $this->middleware(OnlyFullVersion::class)->only([
            'store',
            'update',
            'destroy'
        ]);

    }

    public function index()
    {
        $homeDashboardItems = $this->homeService->index();
        $resourceTypes      = $this->homeService->getResourceTypes();

        return view('dashboard.home.index', compact('homeDashboardItems', 'resourceTypes'));
    }

    public function create()
    {
        $resourceTypes = $this->homeService->getResourceTypes();

        if (checkDemo()) {
            $widget       = $this->homeService->getRecentWidget();
            $resourceType = $this->homeService->getResourceType($widget);

            $resources = !is_null($widget->resource)
            ? $this->homeService->getAllResources($widget->resource->module->id)
            : $this->homeService->getAllResources(null);

            return view('dashboard.home.partials._add-widget-modal-demo', compact('widget', 'resourceTypes', 'resources', 'resourceType'));
        }

        return view('dashboard.home.partials._add-widget-modal', compact('resourceTypes'));

    }

    public function store(StoreHomeWidgetRequest $request)
    {
        $this->homeService->storeWidget($request);

        return response()->json([
            'success'   => true,
            'message'   => trans('home.messages.store_widget'),
            'last_page' => $this->homeService->getHomeLastPage($request)
        ]);
    }

    public function edit(HomeDashboard $widget)
    {

        $resourceTypes = $this->homeService->getResourceTypes();
        $resourceType  = $this->homeService->getResourceType($widget);

        $resources = !is_null($widget->resource)
        ? $this->homeService->getAllResources($widget->resource->module)
        : $this->homeService->getAllResources(null);
        return view('dashboard.home.partials._edit-widget-modal', compact('widget', 'resourceTypes', 'resources', 'resourceType'));
    }

    public function update(UpdateHomeWidgetRequest $request, HomeDashboard $widget)
    {
        $this->homeService->updateWidget($request, $widget);

        return response()->json([
            'success' => true,
            'message' => trans('home.messages.save_widget')
        ]);

    }

    public function destroy(HomeDashboard $widget)
    {
        $this->homeService->destroy($widget);

        return response()->json([
            'message'   => trans('home.messages.delete'),
            'last_page' => $this->homeService->getHomeLastPage(request())
        ]);

    }

}