<?php
/**
 * Simplified Column Info Extractor
 * Gets only column name, type, and enum values
 */

// handling super globals
define("DIRECTACESS", 1);
require_once "../request.php";
require_once "functions.php";

if (online_demo_mode === 1) {
    echo "success";
    exit();
}

if (!is_connected()) {
    echo(" Must be connected to run this script");
    exit();
}

$_GET = array();
if (isset($_POST["columnsInfo"])) {
    $_POST = array("columnsInfo" => $_POST["columnsInfo"]);
} else {
    $_POST = array();
}
$_ENV    = array();
$_FILES  = array();
$_COOKIE = array();

require_once "../lib.php";
require_once "sessionCleaner.php";

// Function to extract table names from SQL query
function extractTableNamesFromSQL($sql) {
    $originalSql = trim($sql);
    $tables = array();
    
    // Remove extra spaces and normalize
    $originalSql = preg_replace('/\s+/', ' ', $originalSql);
    
    // More comprehensive regex patterns for different scenarios:
    
    // 1. FROM clause - handles: FROM table, FROM `table`, FROM database.table, FROM `database`.`table`
    if (preg_match('/from\s+(?:`?[a-zA-Z_][a-zA-Z0-9_]*`?\.)?`?([a-zA-Z_][a-zA-Z0-9_]*)`?/i', $originalSql, $matches)) {
        $tables[] = $matches[1];
    }
    
    // 2. JOIN clauses - handles all JOIN types with backticks
    $joinPattern = '/(?:inner\s+join|left\s+join|right\s+join|full\s+join|join)\s+(?:`?[a-zA-Z_][a-zA-Z0-9_]*`?\.)?`?([a-zA-Z_][a-zA-Z0-9_]*)`?/i';
    if (preg_match_all($joinPattern, $originalSql, $matches)) {
        foreach ($matches[1] as $tableName) {
            $tables[] = $tableName;
        }
    }
    
    return array_unique($tables);
}


// Function to get column information using DESCRIBE
function getColumnsFromTables($tables, $dbHandler, $srm_dashboard) {
    $allColumnsInfo = array();
    
    foreach ($tables as $tableName) {
        $tableName = clean_input($tableName);
        
        try {
            $result = $dbHandler->query("DESCRIBE `$tableName`", "ASSOC", array(), $srm_dashboard);
            foreach ($result as $columnData) {
                $columnName = $columnData["Field"];
                $fullType = $columnData["Type"];
                
                // Extract base type
                $type = $fullType;
                if (strpos($fullType, "(") !== false) {
                    $type = substr($fullType, 0, strpos($fullType, "("));
                }
                $type = strtolower(trim($type));
                
                // Build simple column info
                $columnInfo = array(
                    'name' => $columnName,
                    'type' => $type
                );
                
                // Handle enum values
                if ($type === 'enum') {
                    $enumValues = array();
                    if (preg_match('/enum\((.*)\)/i', $fullType, $matches)) {
                        $enumString = $matches[1];
                        // Parse enum values
                        $pattern = "/(?:^|,)(['\"])((?:[^'\"\\\\]|\\\\.)*)\\1/";
                        if (preg_match_all($pattern, $enumString, $valueMatches)) {
                            foreach ($valueMatches[2] as $value) {
                                $enumValues[] = stripslashes($value);
                            }
                        }
                    }
                    $columnInfo['enum_values'] = $enumValues;
                }
                
                $allColumnsInfo[$tableName][$columnName] = $columnInfo;
            }
        } catch (Exception $e) {
            error_log("Error describing table $tableName: " . $e->getMessage());
        }
    }
    
    return $allColumnsInfo;
}


function getSimpleColumnsFromTables($tables, $dbHandler, $srm_dashboard) {
    $allColumnsInfo = array();
    
    foreach ($tables as $tableName) {
        $tableName = clean_input($tableName);
        
        try {
            $result = $dbHandler->query("DESCRIBE `$tableName`", "ASSOC", array(), $srm_dashboard);
            foreach ($result as $columnData) {
                $columnName = $columnData["Field"];
                $fullType = $columnData["Type"];
                
                // Extract base type
                $type = $fullType;
                if (strpos($fullType, "(") !== false) {
                    $type = substr($fullType, 0, strpos($fullType, "("));
                }
                $type = strtolower(trim($type));
                
                // Build simple column info
                $columnInfo = array(
                    'name' => $columnName,
                    'type' => $type
                );
                
                $allColumnsInfo[$columnName] = $type;
            }
        } catch (Exception $e) {
            error_log("Error describing table $tableName: " . $e->getMessage());
        }
    }
    
    return $allColumnsInfo;
}


// Function to match query columns with table columns
function matchColumnsWithTypes($queryResult, $tableColumnsInfo) {
    $matchedColumns = array();
    
    if (empty($queryResult)) {
        return $matchedColumns;
    }
    
    $queryColumns = array_keys($queryResult[0]);
    
    foreach ($queryColumns as $queryColumn) {
        $found = false;
        
        // Search in all tables
        foreach ($tableColumnsInfo as $tableName => $tableColumns) {
            foreach ($tableColumns as $columnName => $columnInfo) {
                if (strtolower($columnName) === strtolower($queryColumn)) {
                    $matchedColumns[$queryColumn] = $columnInfo;
                    $found = true;
                    break 2;
                }
            }
        }
        
        // If not found, create unknown entry
        if (!$found) {
            $matchedColumns[$queryColumn] = array(
                'name' => $queryColumn,
                'type' => 'unknown'
            );
        }
    }
    
    return $matchedColumns;
}

// SQL validation function
function is_valid_select_sql($sql) {
    $sql = strtolower($sql);
    $must = array("select", "from");
    $forbidden = array("drop ", "delete ", "insert ", "update ", "describe ", "desc ", "show ", "create ");

    foreach ($must as $value) {
        if (!strstr($sql, $value)) {
            return '\'' . $value . '\' must be used in sql statement';
        }
    }

    foreach ($forbidden as $value) {
        if (strstr($sql, $value)) {
            return '\'' . $value . '\' is not allowed in the sql statement';
        }
    }

    return true;
}

// Main execution
if (isset($_POST["columnsInfo"])) {
    // if(!$_SESSION[Session_wizard_key]["edit_mode"])
    //     unsetSessionStartFromDataSource();
    $sql = make_valid($_POST['columnsInfo']);

    if (is_valid_select_sql($sql) !== true) {
        echo is_valid_select_sql($sql);
        exit();
    }

    if ($sql !== '') {
        // Add LIMIT 1 if not present
        if (!strpos(strtolower($sql), 'limit')) {
            $sql = $sql . ' LIMIT 1';
        }

        try {
            $result = $dbHandler->query($sql, 'ASSOC', array(), $srm_dashboard);
            $rows = $dbHandler->get_num_rows();

            // Extract table names from the SQL query
            $tableNames = extractTableNamesFromSQL($_POST['columnsInfo']);
            // Get column information from all tables
            $tableColumnsInfo = getColumnsFromTables($tableNames, $dbHandler, $srm_dashboard);
            
            // Match query result columns with their types
            $matchedColumns = matchColumnsWithTypes($result, $tableColumnsInfo);
            
            // Extract only enum columns for easy access
            $enumColumns = array();
            foreach ($matchedColumns as $columnName => $columnInfo) {
                if (isset($columnInfo['enum_values']) && !empty($columnInfo['enum_values'])) {
                    $enumColumns[$columnName] = $columnInfo['enum_values'];
                }
            }
            
            // Set session
            $_SESSION[Session_report_Settings_key]["sql"] = trim(str_replace(";", "", $_POST['columnsInfo']));
            
            $_SESSION[Session_report_Settings_key]["columns"] = getSimpleColumnsFromTables($tableNames, $dbHandler, $srm_dashboard);
            // Return simple response
            $response = array(
                "message" => "success",
                "columns" => $matchedColumns,
                "enum_values" => $enumColumns
            );
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            echo "Error executing query: " . $e->getMessage();
        }
    }
    
    exit();
}
?>