<?php

namespace App\Services;

use App\Models\Module;
use App\Models\Resource;
use App\Models\AccessKey;
use App\Models\Dashboard;
use Illuminate\Support\Facades\URL;

class EmbedResourceService
{

    public function generate($request)
    {

        switch ($request['type_of_embed']) {
            case 'php_embed':
                return route('embed_resource.generate_php', [
                    'type'      => $request['resource_type'],
                    'resource'  => $request['resources'],
                    'accessKey' => $request['access_key'] ?? 'public'
                ]);
            case 'html_embed':
                return route('embed_resource.generate_html', [
                    'type'     => $request['resource_type'],
                    'resource' => $request['resources']
                ]);

            case 'share_resource_url':
            case 'qr_code':
                return $this->getUrl($request['resource_type'], $request['resources'], true);
                break;
        }

    }

    public function preparePHPFileData($resourceType, $resource, $accessKey)
    {
        $decryptedAccessKey = $accessKey != 'public' ? decrypt(AccessKey::find($accessKey)->access_key) : '';

        $isPublic = $accessKey == 'public';
        return [
            'srm_resource_name'       => $resourceType == 'dashboards'
            ? $this->prepareDashboardName(Dashboard::find($resource)->name)
            : Resource::find($resource)->name,
            'srm_resource_url'        => $this->getUrl($resourceType, $resource, $isPublic),
            'srm_access_key'          => $decryptedAccessKey,
            'srm_token_endpoint'      => route('get-embed-token'),
            'srm_access_type'         => $this->getAccessType($resourceType, $resource),
            'custom_php_before_embed' => getEmbedPreferencesValue('custom_php_before_embed'),
            'custom_php_after_embed'  => getEmbedPreferencesValue('custom_php_after_embed'),
            'iframe_width'            => getEmbedPreferencesValue('iframe_width'),
            'iframe_height'           => getEmbedPreferencesValue('iframe_height')
        ];
    }

    public function prepareHTMLFileData($resourceType, $resource)
    {
        return [
            'srm_resource_name'        => $resourceType == 'dashboards'
            ? $this->prepareDashboardName(Dashboard::find($resource)->name)
            : Resource::find($resource)->name,
            'srm_resource_url'         => $this->getUrl($resourceType, $resource, true),
            'custom_html_before_embed' => getEmbedPreferencesValue('custom_html_before_embed'),
            'custom_html_after_embed'  => getEmbedPreferencesValue('custom_html_after_embed'),
            'iframe_width'             => getEmbedPreferencesValue('iframe_width'),
            'iframe_height'            => getEmbedPreferencesValue('iframe_height')
        ];
    }

    public function getUrl($resourceType, $resource, $isPublic = null)
    {

        if ($resourceType != 'dashboards') {
            $module   = Module::find($resourceType);
            $resource = Resource::find($resource);
            $isPublic = strtolower($resource->access_control_type) == "public";

            switch (strtolower($module->name)) {
                case 'charts':
                    return resourceSignedUrl('charts.show', ['chart' => $resource->name], $isPublic);
                case 'kpis / metrics':
                    return resourceSignedUrl('metrics.show', ['metric' => $resource->name], $isPublic);
                case 'pivot tables':
                    return resourceSignedUrl('pivot-tables.show', ['pivotTable' => $resource->name], $isPublic);
                case 'blank reports':
                    return $this->getReportUrlFromRequest($resource->url) . '?mode=embed';
            }

        }

        $dashboard = Dashboard::find($resource);
        $isPublic  = strtolower($dashboard->access_type) == "public";

        return resourceSignedUrl('dashboards.show', ['dashboard' => $dashboard->name], $isPublic);
    }

    public function getAccessType($resourceType, $resource)
    {

        if ($resourceType != 'dashboards') {
            return $resource = Resource::find($resource)->access_control_type;
        }

        return Dashboard::find($resource)->access_type;
    }

    public function prepareDashboardName($dashboardName)
    {
        return explode("__", $dashboardName)[0];
    }

    private function getReportUrlFromRequest($path)
    {
        $currentUrl = request()->url();

        if (strpos($currentUrl, '/dashboard') !== false) {
            $baseUrl = strstr($currentUrl, '/dashboard', true) . '/dashboard/';
        } else {
            $baseUrl = request()->getScheme() . "://" . request()->host();
        }

        return $baseUrl . getReportPath($path);
    }

}

