<?php

namespace App\Services;

use App\Models\Category;

class CategoryService
{
    public function store($data)
    {
        $this->setPreviousCategoriesOrder();

        $defaultIcon = "#xf187;";
        $data        = !isset($data["icon"])
        ? array_merge($data, ["icon" => $defaultIcon])
        : $data;

        $maxOrder = Category::where('name', '!=', 'Unassigned')
            ->max('order');
        $order = $maxOrder ? $maxOrder + 1 : 1;

        $data = array_merge($data, ["order" => $order]);
        Category::create($data);
    }

    public function update($data, $category)
    {
        $this->setPreviousCategoriesOrder();

        $this->checkDefaultCategory("unassigned", $category);

        $defaultIcon = "#xf187;";

        $data = !isset($data["icon"])
        ? array_merge($data, ["icon" => $defaultIcon]) : $data;

        $category->update($data);
    }

    public function moveUp($category)
    {

// Prevent "Unassigned" category from moving up
        if (strtolower($category->name) === 'unassigned') {
            return;
        }

        $previousCategory = Category::where('order', '<', $category->order)
            ->where('id', '!=', 1)
            ->where('name', '!=', 'Unassigned') // Don't swap with Unassigned
            ->orderBy('order', 'desc')
            ->first();

        if ($previousCategory) {
            $tempOrder = $previousCategory->order;
            $previousCategory->update(['order' => $category->order]);
            $category->update(['order' => $tempOrder]);
        }

    }

    public function moveDown($category)
    {

// Prevent moving down if current category is "Unassigned"
        if (strtolower($category->name) === 'unassigned') {
            return;
        }

        $nextCategory = Category::where('order', '>', $category->order)
            ->where('id', '!=', 1)
            ->where('name', '!=', 'Unassigned') // Don't swap with Unassigned
            ->orderBy('order', 'asc')
            ->first();

        if ($nextCategory) {
            $tempOrder = $nextCategory->order;
            $nextCategory->update(['order' => $category->order]);
            $category->update(['order' => $tempOrder]);
        }

    }

    public function destroy($category)
    {
        $this->checkDefaultCategory("unassigned", $category);

        $this->checkHasResource($category);

        $category->delete();
    }

    public function getCategories()
    {
        return Category::withCount('resources')
            ->orderBy('order', 'asc')
            ->get();
    }

    public function getCategoriesCount()
    {
        return Category::where('name', '!=', 'Unassigned')->count();
    }

    private function checkDefaultCategory($defaultCategoryName, $category)
    {

        if (strtolower($category->name) == strtolower($defaultCategoryName)) {
            abort(400, "\"$defaultCategoryName\" category can't be edited or deleted.");
        }

    }

    private function checkHasResource($category)
    {

        if ($category->resources()->exists()) {
            abort(400, trans('categories.remove_resources_before',
                ['category' => $category->name]));
        }

    }

    private function setPreviousCategoriesOrder()
    {
        $categories = Category::all();

        if (Category::whereNull('order')->exists()) {
            $maxOrder = $categories->where('name', '!=', 'Unassigned')
                ->whereNotNull('order')
                ->max('order') ?? 0;

            $categories->each(function ($category) use (&$maxOrder) {

                if (is_null($category->order)) {
                    $maxOrder++;
                    $category->update(['order' => $maxOrder]);
                }

            });
        }

        $unassignedCategory = $categories->firstWhere('name', 'Unassigned');

        if ($unassignedCategory && $unassignedCategory->order !== 999999) {
            $unassignedCategory->update(['order' => 999999]);
        }

    }
}
