<?php

namespace App\Services;

use App\Models\Resource;
use App\Models\AccessKey;
use App\Models\Dashboard;
use App\Models\DashboardWidget;

class AccessKeyService
{
    public function index()
    {
        return AccessKey::orderBy('created_at', 'desc')->get();
    }

    public function store($request)
    {
        $access_key                 = bin2hex(random_bytes(32));
        $request['access_key']      = encrypt($access_key);
        $request['origin_type']     = isset($request['origin']) ? $request['origin'] : null;
        $request['allowed_origins'] = isset($request['allowed_origins']) ? $this->formatAllowedOrigins($request['allowed_origins']) : null;
        $request['group_id']        = isset($request['group']) ? $request['group'] : null;

        AccessKey::create($request);
        return $access_key;
    }

    private function formatAllowedOrigins($rawOrigins)
    {
        $originsArray = preg_split('/\r\n|\r|\n/', $rawOrigins);
        $originsArray = array_filter(array_map('trim', $originsArray));

        $validatedOrigins = [];

        foreach ($originsArray as $origin) {

            if (!filter_var($origin, FILTER_VALIDATE_URL)) {
                continue;
            }

            $parts = parse_url($origin);

            $cleanOrigin = $parts['scheme'] . '://' . $parts['host'];

            if (isset($parts['port'])) {
                $cleanOrigin .= ':' . $parts['port'];
            }

            $validatedOrigins[] = $cleanOrigin;
        }

        return implode(' ', $validatedOrigins);
    }

    public function update($data, $accessKey)
    {
        $accessKey->update($data);
    }

    public function destroy($accessKey)
    {
        $accessKey->tokens()->delete();
        $accessKey->delete();
    }

    public function revoke($accessKey)
    {
        $accessKey->tokens()->delete();
    }

    public function getAllAccessKeysByResourceId($resourceType, $resource)
    {

        if ($this->isPublic($resourceType, $resource)) {
            return [];
        }

        $fullKeys = AccessKey::where('type', 'full')
            ->select('id', 'label')
            ->get()
            ->toArray();

        if ($resourceType === 'dashboards') {
            $limitedAccessKeys = $this->getDashboardLimitedAccessKeys($resource)
                ->map(fn($item) => $item->only(['id', 'label']))
                ->toArray();

            return array_merge($fullKeys, $limitedAccessKeys);
        }

        $resourceKeys = $this->getAccessKeysWithResourcePermission($resource)
            ->map(fn($item) => $item->only(['id', 'label']))
            ->toArray();

        return array_merge($fullKeys, $resourceKeys);
    }

    public function isPublic($resourceType, $resource)
    {

        if ($resourceType == 'dashboards') {
            return Dashboard::where('id', $resource)->where(['access_type' => 'public'])->exists() ? true : false;
        }

        return Resource::where('id', $resource)->where(['access_control_type' => 'public'])->exists() ? true : false;
    }

    private function getDashboardLimitedAccessKeys($dashboardId)
    {
        $widgets = DashboardWidget::where('dashboard_id', $dashboardId)->get();

        // Collect all resource IDs from widgets & report_lists
        $resourceIds = [];

        foreach ($widgets as $widget) {

            if (!is_null($widget->resource_id)) {
                $resourceIds[] = $widget->resource_id;
            }

            if (is_null($widget->resource_id) && !empty($widget->report_lists) && is_array($widget->report_lists)) {

                foreach ($widget->report_lists as $report) {

                    if (!empty($report['id'])) {
                        $resourceIds[] = $report['id'];
                    }

                }

            }

        }

        $resourceIds = array_unique($resourceIds);

        if (empty($resourceIds)) {
            return collect(); // no resources -> no limited access keys
        }

        // Load resources for public/private check
        $resources = Resource::whereIn('id', $resourceIds)->get();

// If all are public, then all keys are valid
        if ($resources->every(fn($r) => strtolower($r->access_control_type) === "public")) {
            return AccessKey::all();
        }

        // Now filter access keys that can access *all* resources
        $accessKeys = AccessKey::where('type', 'limited')->get()->filter(function ($key) use ($resources) {

            foreach ($resources as $resource) {

                if (strtolower($resource->access_control_type) === "public") {
                    continue; // skip check for public resources
                }

                $hasAccess = $key->group
                    ->resource_permissions()
                    ->where('resource_id', $resource->id)
                    ->exists();

                if (!$hasAccess) {
                    return false; // fail fast if one resource is not accessible
                }

            }

            return true; // this key has access to all resources
        });

        return $accessKeys->values(); // reindex collection
    }

    private function getAccessKeysWithResourcePermission($resourceId)
    {
        $resource = Resource::find($resourceId);

        if (!$resource) {
            return collect(); // no resource found
        }

        // Get all access keys that have permission for this resource
        $accessKeys = AccessKey::whereHas('group.resource_permissions', function ($query) use ($resource) {
            $query->where('resource_id', $resource->id);
        })->get();

// Also include "public" access
        if (strtolower($resource->access_control_type) === "public") {
            return AccessKey::all();
        }

        return $accessKeys;
    }

    public function getAccessKeyById($accessKey)
    {
        return AccessKey::find($accessKey);
    }

}

