<?php

namespace App\Repositories;

use App\Models\Module;
use App\Models\Resource;
use Illuminate\Support\Str;
use App\Models\ResourcePermission;
use Illuminate\Support\Facades\DB;
use App\Models\ResourceConfiguration;

class PivotTableRepository
{
    public function store($request)
    {
        $resource = DB::transaction(function () use ($request) {
            $resource = Resource::create([
                'name'                => $request['pivot_table_name'],
                'url'                 => $this->getUrl(Str::slug($request['pivot_table_name'], '_')),
                'access_control_type' => ucfirst($request["access_level"]),
                'module_id'           => $this->getPivotTableModule(),
                'category_id'         => intval($request["category"]),
                'resource_creator'    => auth()->id(),
                'data_connection_id'  => strtolower($request["connection"]) != "default" ? intval($request["connection"]) : null
            ]);

            if (!empty($request["groups"])) {

                if (auth()->user()->isAdminOrOwner()) {

                    foreach ($request["groups"] as $group) {
                        ResourcePermission::create([
                            'group_id'    => $group,
                            'resource_id' => $resource->id,
                            'can_view'    => true
                        ]);

                    }

                } else {
                    ResourcePermission::create([
                        'group_id'    => auth()->user()->group->group_ID,
                        'resource_id' => $resource->id,
                        'can_view'    => true
                    ]);

                }

            }

            ResourceConfiguration::create([
                'Resource_id'         => $resource->id,
                'Json_configurations' => $this->getPivotTableResourceConfiguration($request)
            ]);
            return $resource->id;
        });

        return $resource;
    }

    public function duplicate($request, $originalPivotTable)
    {
        return DB::transaction(function () use ($request, $originalPivotTable) {

            $newName = $request['name'];

            // Create a new resource with a modified name and URL
            $newResource = Resource::create([
                'name'                => $newName,
                'url'                 => $this->getUrl(Str::slug($newName, '_')),
                'access_control_type' => $originalPivotTable->access_control_type,
                'module_id'           => $originalPivotTable->module_id,
                'category_id'         => $originalPivotTable->category_id,
                'resource_creator'    => auth()->id(),
                'data_connection_id'  => $originalPivotTable->data_connection_id
            ]);

            $originalPermissions = ResourcePermission::where('resource_id', $originalPivotTable->id)->get();

            foreach ($originalPermissions as $permission) {
                ResourcePermission::create([
                    'group_id'    => $permission->group_id,
                    'resource_id' => $newResource->id,
                    'can_view'    => $permission->can_view
                ]);
            }

            $originalConfiguration = ResourceConfiguration::where('Resource_id', $originalPivotTable->id)->first();

            if ($originalConfiguration) {
                ResourceConfiguration::create([
                    'Resource_id'         => $newResource->id,
                    'Json_configurations' => $originalConfiguration->Json_configurations
                ]);
            }

            return $newResource;
        });
    }

    public function update($request, $pivotTable)
    {
        DB::transaction(function () use ($request, $pivotTable) {
            $resource = $pivotTable->update([
                'access_control_type' => ucfirst($request["access_level"]),
                'module_id'           => $this->getPivotTableModule(),
                'category_id'         => intval($request["category"]),
                'data_connection_id'  => strtolower($request["connection"]) != "default" ? intval($request["connection"]) : null
            ]);

            if (isset($request["groups"])) {

                ResourcePermission::where('resource_id', $pivotTable->id)->delete();

                if (auth()->user()->isAdminOrOwner()) {

                    foreach ($request["groups"] as $group) {

                        ResourcePermission::create([
                            'group_id'    => $group,
                            'resource_id' => $pivotTable->id,
                            'can_view'    => true
                        ]);

                    }

                } else {
                    ResourcePermission::create([
                        'group_id'    => auth()->user()->group->group_ID,
                        'resource_id' => $pivotTable->id,
                        'can_view'    => true
                    ]);

                }

            }

            ResourceConfiguration::updateOrCreate(
                ['Resource_id' => $pivotTable->id],
                [
                    'Json_configurations' => $this->getPivotTableResourceConfiguration($request)
                ]
            );
        });
    }

    private function getPivotTableModule()
    {
        return Module::where('name', 'Pivot tables')->first()->id;
    }

    private function getUrl($pivotName)
    {
        return substr(getBaseUrl(), -1) == "/"
        ? getBaseUrl() . "pivot-tables/" . $pivotName
        : getBaseUrl() . "/pivot-tables/" . $pivotName;
    }

    private function getUpdatedSession($session, $chart)
    {
        return array_merge($chart->configurations->Json_configurations, $session);
    }

    private function getPivotTableResourceConfiguration($request)
    {

        return [
            "connection"         => strtolower($request["connection"]) != "default" ? intval($request["connection"]) : null,
            "pivot_table_title"  => $request['pivot_table_title'] ?? "",
            "pivot_table_name"   => $request['pivot_table_name'] ?? "",
            "alignment"          => $request['alignment'] ?? "",
            "theme"              => $request['theme'] ?? "",
            "grand_totals_label" => $request['grand_totals_label'] ?? "",
            // "grand_totals_function" => $request['grand_totals_function'] ?? "",
            "show_grand_totals"  => $request['show_grand_totals'] ?? "",
            "access_level"       => $request['access_level'] ?? "",
            "combine_filters"    => $request['combine_filters'] ?? "",
            "values_format"      => $request['values_format'] ?? "",
            "currency"           => $request['currency'] ?? "",
            "custom_unit"        => $request['custom_unit'] ?? "",
            "row_headers"        => $request['row_headers'] ?? "",
            "values_fields"      => $request['values_fields'] ?? "",
            "column_headings"    => $request['column_headings'] ?? "",
            "joins"              => $request['joins'] ?? "",
            "static_filters"     => $request['static_filters'] ?? "",
            "parameters"         => $request['parameters'] ?? ""
        ];

    }

}
