<?php

namespace App\Models;

use App\Models\Resource;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Dashboard extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $primaryKey = 'id';
    protected $connection = 'mysql';

    protected $fillable = [
        'name',
        'icon',
        'title',
        'user_id',
        'url',
        'order',
        'access_type'

    ];

    // Expose a computed "clean" name for views/URLs/JSON
    protected $appends = ['clean_name'];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
        $prefix      = config('srm_config.installer.table_prefix', 'srm_');
        $this->table = "{$prefix}dashboards";
    }

    protected static function booted()
    {
        // CREATE: add UUID if it's not already there
        static::creating(function (Dashboard $dashboard) {
            $raw = $dashboard->getAttributes()['name'] ?? '';

            if (!self::hasUuidSuffix($raw)) {
                $dashboard->attributes['name'] = self::stripUuidSuffix($raw) . '__' . (string) generateUUID();
            }

        });

        // UPDATE: preserve UUID if old value had one, else never add
        static::updating(function (Dashboard $dashboard) {
            $oldRaw = $dashboard->getRawOriginal('name') ?? '';
            $newRaw = $dashboard->getAttributes()['name'] ?? '';

            if (self::hasUuidSuffix($oldRaw)) {
                // reuse the old UUID
                $uuid                          = self::extractUuidSuffix($oldRaw);
                $base                          = self::stripUuidSuffix($newRaw);
                $dashboard->attributes['name'] = $base . '__' . $uuid;
            } else {
                // keep as-is without UUID
                $dashboard->attributes['name'] = self::stripUuidSuffix($newRaw);
            }

        });
    }

    /* ===================== Helpers ===================== */

    protected static function uuidRegex(): string
    {
        return '/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i';
    }

    protected static function hasUuidSuffix(string $name): bool
    {
        $parts = explode('__', $name);
        $last  = end($parts) ?: '';
        return (bool) preg_match(self::uuidRegex(), $last);
    }

    protected static function extractUuidSuffix(string $name): ?string
    {
        $parts = explode('__', $name);
        $last  = end($parts) ?: '';
        return preg_match(self::uuidRegex(), $last) ? $last : null;
    }

    protected static function stripUuidSuffix(string $name): string
    {
        $parts = explode('__', $name);
        $last  = end($parts) ?: '';

        if (preg_match(self::uuidRegex(), $last)) {
            array_pop($parts);
            return implode('__', $parts);
        }

        return $name;
    }

/* ===================== Accessors ===================== */

    // Clean version for forms/UI
    public function getCleanNameAttribute(): string
    {
        $raw = $this->getAttributes()['name'] ?? '';
        return self::stripUuidSuffix($raw);
    }

    // Raw DB version (with UUID) — used for URLs
    public function getRawName(): string
    {
        return $this->getAttributes()['name'] ?? '';
    }

/* ===================== Routing ===================== */

    public function resolveRouteBinding($value, $field = null)
    {

// Match uuid after the last "__"
        if (preg_match('/__([0-9a-fA-F\-]{36})$/', $value, $matches)) {
            $uuid = $matches[1];
            return $this->where('name', 'LIKE', "%__{$uuid}%")->firstOrFail();
        }

        // fallback: search by name only
        return $this->where('name', $value)->firstOrFail();
    }

    public function getRouteKey()
    {
        return $this->name . '__' . $this->uuid;
    }

    public function resource()
    {
        return $this->hasOne(Resource::class, 'resource_id');
    }

    public function widgets()
    {
        return $this->hasMany(DashboardWidget::class, 'dashboard_id');
    }

}
