<?php

namespace App\Http\Middleware;

use Closure;
use Carbon\Carbon;
use App\Models\EmbedLog;
use App\Models\Resource;
use App\Models\Dashboard;
use App\Models\EmbedToken;
use Illuminate\Http\Request;
use App\Models\DashboardWidget;
use Symfony\Component\HttpFoundation\Response;

class loginOrPublic
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $token     = $this->getTokenFromRequestOrReferer();
        $embedMode = config('srm_config.dashboard.embed_mode');

        if ($token) {

            if (is_null($embedMode) || $embedMode != "include") {
                return abort(Response::HTTP_FORBIDDEN, 'Feature not enabled');
            }

            $embed = EmbedToken::withTrashed()->where('token', $token)->first();

            if (!$embed || $embed->deleted_at) {
                $this->logEmbed($request, $embed, false, 'Invalid token');
                return abort(Response::HTTP_FORBIDDEN, 'Invalid embed token.');
            }

            if (Carbon::now()->greaterThan($embed->expire_at)) {
                $this->logEmbed($request, $embed, false, 'Expired token');
                return abort(Response::HTTP_FORBIDDEN, 'Embed Token is expired. Please refresh the page to generate a new token.');
            }

            if (!$embed->hasValidAccessKey()) {
                $this->logEmbed($request, $embed, false, 'Access denied');
                return abort(Response::HTTP_FORBIDDEN, 'You are not authorized to access this embedded resource.');
            }
            if (strtolower($embed->accessKey->type) == "limited" && !$this->hasFullAccess($embed, $embed->accessKey, $request->url())) {
                $this->logEmbed($request, $embed, false, 'Access denied');

                return abort(Response::HTTP_FORBIDDEN, trans('api.messages.not_authorized_to_access_this_resource'));
            }

            $this->logEmbed($request, $embed, true, 'Success');
            session()->put('loginType', 'anonymous_group_user');

            return $next($request);
        }

        if (auth()->check()) {
            return $next($request);
        }

        $routeName = $request->route()->getName();

        $resourceTypeMap = [
            'pivot-tables.' => 'pivotTable',
            'charts.'       => 'chart',
            'metrics.'      => 'metric',
            'reports.'      => 'report'
        ];

        foreach ($resourceTypeMap as $prefix => $param) {

            if (str_starts_with($routeName, $prefix)) {
                $resource = $this->resolveResource($request->route($param));

                if ($prefix === 'reports.') {

                    if (strtolower($resource->access_control_type) === 'public') {
                        return $next($request);
                    }

                    return redirect()->route('login');
                }

                if (strtolower($resource->access_control_type) !== 'public') {
                    return redirect()->route('login');
                }

                break;
            }

        }

        return $next($request);
    }

    public function getTokenFromRequestOrReferer()
    {
        $token       = request()->query('token') ?? request()->input('token');
        $extraParams = [];

        if (!$token) {
            $referrer = request()->headers->get('referer');

            if ($referrer) {
                $query = parse_url($referrer, PHP_URL_QUERY);
                parse_str($query, $params);

                if (isset($params['token'])) {
                    $token = $params['token'];
                    unset($params['token']);
                    $extraParams = $params;
                }

            }

        }

        if ($token) {
            $parts = preg_split('/[?&]/', $token, 2);
            $token = $parts[0];

            if (isset($parts[1])) {
                parse_str($parts[1], $extraFromToken);
                $extraParams = array_merge($extraParams, $extraFromToken);
            }

            request()->merge(['token' => $token] + $extraParams);
        }

        return $token;
    }

    private function isAllowedIp($accessKeyRecord, $ip)
    {

        if (empty($accessKeyRecord->allowed_ip)) {
            return true;
        }

        $allowedIps = array_map('trim', explode(',', $accessKeyRecord->allowed_ip));

        return in_array($ip, $allowedIps, true);
    }

    private function resolveResource($resource)
    {

        if (is_string($resource)) {
            return Resource::where('name', $resource)->first();
        }

        return $resource;
    }

    private function getOriginFromReferer($referer){
        
        if(empty($referer) || is_null($referer)) {
            $origin = 'Not Detected';
        }else{
            
            $origin = $referer ? parse_url($referer, PHP_URL_SCHEME) . '://' . parse_url($referer, PHP_URL_HOST) : 'Not Detected';
        }

        return $origin;
    }

    private function logEmbed(Request $request, $token, $success, $reason)
    {
        $origin = $this->getOriginFromReferer(request()->server('HTTP_REFERER'));

        EmbedLog::create([
            'token_id'      => $token->id ?? null,
            'access_key_id' => isset($token) ? $token->access_key_id : null,
            'origin'        => $origin,
            'resource_url'  => $request->url(),
            'attembed_ip'   => getClientIpv4($request),
            'success'       => $success,
            'reason'        => $reason
        ]);

        if ($success) {
            try {
                $token->accessKey->increment('uses_count');

            } catch (\Exception $e) {

            }

        }

    }

    private function hasFullAccess($token, $accessKeyRecord, $resourceUrl)
    {

        if (isDashboard($resourceUrl)) {
            $dashboard = Dashboard::where('url', $resourceUrl)->first();

            foreach ($this->getDashboardWidgets($dashboard) as $widget) {

                if (!$this->hasResourcePermissionById($accessKeyRecord, $widget)) {
                    return false;
                }

            }

        } else {

            if (!$this->hasResourcePermission($accessKeyRecord, $resourceUrl)) {
                return false;
            }

        }

        return true;
    }

    private function getDashboardWidgets($dashboard)
    {
        return DashboardWidget::where('dashboard_id', $dashboard->id)->get();
    }

    private function hasResourcePermission($accessKeyRecord, $resourceUrl)
    {
        $cleanResourceUrl = removeStringsFromUrl($resourceUrl, ["/build", "/apply", "/print", "/export", "xValue"]);
        $resourceName     = getLastSegment($cleanResourceUrl);

        $resource = Resource::where('url', $cleanResourceUrl)->orWhere('name', $resourceName)->first();

        if ($resource) {
            return $accessKeyRecord->group
                ->resource_permissions()
                ->where('resource_id', $resource->id)
                ->exists() || strtolower($resource->access_control_type) === 'public';
        }

        return false;
    }

    private function hasResourcePermissionById($accessKeyRecord, $widget)
    {

// dd($widget);
        if (!is_null($widget->resource_id)) {
            $resource = Resource::find($widget->resource_id);
            if ($resource) {
                return $accessKeyRecord->group
                    ->resource_permissions()
                    ->where('resource_id', $resource->id)
                    ->exists() || strtolower($resource->access_control_type) === "public";
            }

            return false;
        }

        if (is_null($widget->resource_id) && !empty($widget->report_lists)) {
            if (is_array($widget->report_lists)) {
                foreach ($widget->report_lists as $report) {
                    if (!empty($report['id'])) {
                        $resource = Resource::find($report['id']);
                        if (!$resource) {
                            return false;
                        }

                        $hasAccess = $accessKeyRecord->group
                            ->resource_permissions()
                            ->where('resource_id', $resource->id)
                            ->exists()
                        || strtolower($resource->access_control_type) === "public";

                        if (!$hasAccess) {
                            return false;
                        }

                    }

                }

                return true;
            }

        }

        return false;
    }

}
