<?php

namespace App\Http\Controllers;

use App\Models\Resource;
use App\Models\AccessKey;
use App\Models\Dashboard;
use App\Models\EmbedToken;
use App\Models\DashboardWidget;
use App\Models\EmbedPreference;
use App\Services\EmbedResourceService;
use App\Http\Requests\CheckEmbededTokenRequest;

class GetEmbededTokenController extends Controller
{
    private $embedResourceService;
    public function __construct(EmbedResourceService $embedResourceService)
    {

        if (config('srm_config.dashboard.embed_mode') != "include") {
            return response()->json(['message' => trans('api.messages.embed_mode_disabled')], 403);
        }

        $this->embedResourceService = $embedResourceService;
    }

    public function __invoke(CheckEmbededTokenRequest $request)
    {
        $data        = $request->validated();
        $resourceUrl = $data['resource_url'];

        $accessKeyRecord = $this->getValidAccessKey($data);

        if (!$accessKeyRecord) {
            return response()->json(['message' => trans('api.messages.invalid_access_key')], 401);
        }

        if (!$this->isValidResourceUrl($resourceUrl)) {
            return response()->json(['message' => trans('api.messages.not_authorized_to_access_this_resource')], 403);
        }

        if (strtolower($accessKeyRecord->type) == "limited" && !$this->hasFullAccess($accessKeyRecord, $resourceUrl)) {
            return response()->json(['message' => trans('api.messages.not_authorized_to_access_this_resource')], 403);
        }

// if ($accessKeyRecord->allowed_ip && !$this->isAllowedIp($accessKeyRecord, getClientIpv4($request))) {

//     return response()->json(['message' => trans('api.messages.unregistered_origin')], 403);
        // }

        $token = $this->createEmbedToken($accessKeyRecord, $resourceUrl, $data);

        return response()->json(['token' => $token], 200);
    }

    private function getValidAccessKey($request)
    {
        $authHeader = $request['Authorization'];

        if (!$authHeader || !preg_match('/Bearer\s+(\S+)/', $authHeader, $matches)) {
            return null;
        }

        $accessKey = $matches[1];

        return AccessKey::all()->first(function ($record) use ($accessKey) {
            try {
                return decrypt($record->access_key) === $accessKey;
            } catch (\Exception $e) {
                return false;
            }

        });
    }

    private function isValidResourceUrl($url)
    {

        if (empty($url)) {
            return false;
        }

        return !preg_match('/(settings|create|edit|wizard|apply|export|print)/i', $url);
    }

    private function hasFullAccess($accessKeyRecord, $resourceUrl)
    {
        $resourceUrl = $this->getCleanUrl($resourceUrl);

        if (isDashboard($resourceUrl)) {
            $dashboard = Dashboard::where('url', $resourceUrl)->first();

            foreach ($this->getDashboardWidgets($dashboard) as $widget) {

                if (!$this->hasResourcePermissionById($accessKeyRecord, $widget)) {
                    return false;
                }

            }

        } else {

            if (!$this->hasResourcePermission($accessKeyRecord, $resourceUrl)) {
                return false;
            }

        }

        return true;
    }

    private function getDashboardWidgets($dashboard)
    {
        return DashboardWidget::where('dashboard_id', $dashboard->id)->get();
    }

    private function hasResourcePermission($accessKeyRecord, $resourceUrl)
    {

        if (strpos($resourceUrl, 'SRM9') !== false) {
            $parts       = explode('SRM9', $resourceUrl, 2);
            $resourceUrl = 'SRM9' . $parts[1];
        }

        $resource  = Resource::where('url', $resourceUrl)->first();
        $reportUrl = getReportUrl($resourceUrl);

        if ($resource) {
            return $accessKeyRecord->group
                ->resource_permissions()
                ->where('resource_id', $resource->id)
                ->exists();
        }

        return false;
    }

    private function getCleanUrl($url)
    {
        $parts = parse_url($url);

        $cleanUrl = $parts['scheme'] . "://" . $parts['host'] . (isset($parts['port']) ? ':' . $parts['port'] : '') . $parts['path'];

        return $cleanUrl;

    }

    private function hasResourcePermissionById($accessKeyRecord, $widget)
    {

// dd($widget);
        if (!is_null($widget->resource_id)) {
            $resource = Resource::find($widget->resource_id);
            if ($resource) {
                return $accessKeyRecord->group
                    ->resource_permissions()
                    ->where('resource_id', $resource->id)
                    ->exists() || strtolower($resource->access_control_type) === "public";
            }

            return false;
        }

        if (is_null($widget->resource_id) && !empty($widget->report_lists)) {
            if (is_array($widget->report_lists)) {
                foreach ($widget->report_lists as $report) {
                    if (!empty($report['id'])) {
                        $resource = Resource::find($report['id']);
                        if (!$resource) {
                            return false;
                        }

                        $hasAccess = $accessKeyRecord->group
                            ->resource_permissions()
                            ->where('resource_id', $resource->id)
                            ->exists()
                        || strtolower($resource->access_control_type) === "public";

                        if (!$hasAccess) {
                            return false;
                        }

                    }

                }

                return true;
            }

        }

        return false;
    }

    private function isAllowedIp($accessKeyRecord, $ip)
    {

        if (empty($accessKeyRecord->allowed_ip)) {
            return true;
        }

        $allowedIps = array_map('trim', explode(',', $accessKeyRecord->allowed_ip));

        return in_array($ip, $allowedIps, true);
    }

    private function getEmbedTokenLifeSpan()
    {
        $prefs = EmbedPreference::where('key', 'embed_token_life_span')->first();
        return $prefs->value ?? 60;
    }

    private function createEmbedToken($accessKeyRecord, $url, $data)
    {
        $token      = bin2hex(random_bytes(32));
        $expiration = now()->addMinutes($this->getEmbedTokenLifeSpan());
        EmbedToken::create([
            'token'                => $token,
            'allowed_resource_url' => $data['resource_url'],
            'expire_at'            => $expiration,
            'access_key_id'        => $accessKeyRecord->id
        ]);

        $signedUrl = $this->getSingedUrl($url, $token);
        return $signedUrl;
    }

    public function getSingedUrl($url, $token, $isPublic = null)
    {

        if (strpos($url, 'srm_charts') !== false) {
            $params = is_null($token) 
            ? ['chart' => $this->getResourceNameFromUrl($url)] 
            : ['chart' => $this->getResourceNameFromUrl($url), 'token' => $token ];

            return resourceSignedUrl('charts.show', $params, $isPublic);

        } elseif (strpos($url, 'srm_metrics') !== false) {
            $params = is_null($token) 
            ? ['metric' => $this->getResourceNameFromUrl($url)] 
            : ['metric' => $this->getResourceNameFromUrl($url), 'token' => $token ];

            return resourceSignedUrl('metrics.show', $params, $isPublic);

        } elseif (strpos($url, 'pivot-tables') !== false) {

            $params = is_null($token) 
            ? ['pivotTable' => $this->getResourceNameFromUrl($url)] 
            : ['pivotTable' => $this->getResourceNameFromUrl($url), 'token' => $token];

            return resourceSignedUrl('pivot-tables.show', $params, $isPublic);

        } elseif (strpos($url, 'srm_modules') !== false) {
            $token = !is_null($token) ? '&token=' . $token : '';
            return $url . $token;

        } elseif (strpos($url, 'dashboards') !== false) {
            $params = is_null($token) 
            ? ['dashboard' => $this->getResourceNameFromUrl($url)] 
            : ['dashboard' => $this->getResourceNameFromUrl($url), 'token' => $token];

            return resourceSignedUrl('dashboards.show', $params, $isPublic);
        }

    }

    private function getResourceNameFromUrl($url)
    {
        $path = parse_url($url, PHP_URL_PATH);

        $segments = explode('/', trim($path, '/'));

        // $index   = array_search('dashboards', $segments);
        $segment = $segments[count($segments) - 1] ?? null;

        return $segment;
    }

}
