<?php

namespace App\Builders;

use Carbon\Carbon;
use Illuminate\Http\Response;
use App\Generators\ParametersGenerator;
use App\Services\DataConnectionService;
use Illuminate\Database\QueryException;
use App\Generators\DataFiltersGenerator;

class PivotTableBuilder
{
    private $pivotTable;
    private $connection;
    private $rowHeaders;
    private $columnHeadings;
    private $valuesFields;
    private $options;
    private $joins;
    private $parameters;
    private $parametersBlocks;
    private $filters;
    private $combineFiltersBy;
    private $query;
    private $dataConnectionService;

    public function __construct($pivotTable)
    {
        $this->pivotTable       = $pivotTable;
        $this->connection       = $pivotTable['connection'] ?? "default";
        $this->rowHeaders       = !empty($pivotTable['row_headers']) ? destructFieldBlocks($pivotTable['row_headers']) : "";
        $this->columnHeadings   = !empty($pivotTable['column_headings']) ? destructFieldBlocks($pivotTable['column_headings']) : "";
        $this->valuesFields     = !empty($pivotTable['values_fields']) ? destructFieldBlocks($pivotTable['values_fields']) : "";
        $this->joins            = !empty($pivotTable['joins']) ? destructJoins($pivotTable['joins']) : "";
        $this->parameters       = $pivotTable['parameters'] ?? '';
        $this->parametersValues = $pivotTable['parametersValues'] ?? '';
        $this->staticFilters    = $pivotTable['static_filters'] ?? '';
        $this->combineFiltersBy = $pivotTable['combine_filters'] ?? 'and';
        $this->options          = [
            "operation"             => $pivotTable['operation'] ?? 'apply',
            "pivot_table_title"     => $pivotTable['pivot_table_title'] ?? 'Pivot table title',
            "grand_totals_function" => "sum",
            "grand_totals_label"    => $pivotTable['grand_totals_label'] ?? "",
            "show_grand_totals"     => $pivotTable['show_grand_totals'] ?? false,
            "values_format"         => $pivotTable['values_format'] ?? "",
            "currency"              => $pivotTable['currency'] ?? "",
            "custom_unit"           => $pivotTable['custom_unit'] ?? ""
        ];

    }

    private function initializeConnection()
    {
        $this->query = (new DataConnectionService())
            ->getConnection(
                $this->connection
            );

        return $this;

    }

    private function getNewConnection()
    {
        return (new DataConnectionService())
            ->getConnection(
                $this->connection
            );
    }

    private function getHeaders($columns = false)
    {

        if ($columns) {
            return $this->columnHeadings;
        }

        return !empty($this->rowHeaders) ? $this->rowHeaders : $this->columnHeadings;
    }

    private function addTables()
    {
        $table = $this->getMainTable();

        $this->query = $this->applyJoins($this->query->table($table), $table);
        return $this;
    }

    private function getXAxisLabel($columns = false)
    {
        $headers  = $columns ? $this->columnHeadings : $this->getHeaders();
        $function = \ucfirst($this->getXAxisFunction());
        return strtolower($function) != "raw" ?
        "$function of {$headers[0]['label']}" :
        "{$headers[0]['label']}";
    }

    private function addAxis()
    {

        if ($this->getXAxisFunction() != "raw") {

            $this->query = $this->query->selectRaw(
                "{$this->getSelectTimeScale()} as '{$this->getXAxisLabel()}'," .
                $this->getYAxisWithFunction()
            );
        } else {

            $this->query = $this->query->selectRaw(
                "{$this->getXAxis()} as '{$this->getXAxisLabel()}'," .
                $this->getYAxisWithFunction()
            );

        }

        return $this;
    }

    private function applyJoins($query, $table)
    {
        $mainTable    = $table;
        $joinedTables = [$mainTable];

        if (!is_null($this->joins)) {
            $pendingJoins = $this->joins;

            while (!empty($pendingJoins)) {
                $processed = [];

                foreach ($pendingJoins as $index => $join) {
                    $leftTable   = $join["left"];
                    $rightTable  = $join["right"];
                    $columnLeft  = $join["columnLeft"];
                    $columnRight = $join["columnRight"];
                    $type        = $join["type"];

                    if (in_array($leftTable, $joinedTables)) {
                        $baseTable    = $leftTable;
                        $targetTable  = $rightTable;
                        $baseColumn   = $columnLeft;
                        $targetColumn = $columnRight;
                    } elseif (in_array($rightTable, $joinedTables)) {
                        $baseTable    = $rightTable;
                        $targetTable  = $leftTable;
                        $baseColumn   = $columnRight;
                        $targetColumn = $columnLeft;
                    } else {
                        continue;
                    }

                    if (!in_array($targetTable, $joinedTables)) {
                        $query = ($type == "inner") ?
                        $query->join($targetTable, "{$baseTable}.{$baseColumn}", "=", "{$targetTable}.{$targetColumn}") :
                        $query->leftJoin($targetTable, "{$baseTable}.{$baseColumn}", "=", "{$targetTable}.{$targetColumn}");

                        $joinedTables[] = $targetTable;
                    }

                    $processed[] = $index;
                }

                foreach ($processed as $index) {
                    unset($pendingJoins[$index]);
                }

                if (empty($processed)) {
                    break;
                }

            }

        }

        return $query;
    }

    public function getMainTable()
    {

        if (!empty($this->joins)) {

            foreach ($this->joins as $join) {

                if (
                    isset($join["type"], $join["left"]) &&
                    $join["type"] === "left"
                ) {
                    return $join["left"];
                }

            }

        }

        return $this->rowHeaders[0]['table'] ?? $this->columnHeadings[0]['table'];
    }

    private function get3DColumnValues()
    {
        $table = $this->valuesFields[0]['table'];
        $query = $this->applyJoins($this->getNewConnection()->table($table), $table);

        return $query
            ->when(!empty($this->getStaticFilters()), function ($query) {

                $filters = $this->getStaticFilters();
                $query->whereRaw($filters["sql"], $filters["bindings"]);
            })

            ->when($this->options['operation'] != "apply" && !empty($this->getSelectedParameters()), function ($query) {
                $filters = $this->getSelectedParameters();

                if ($filters['sql'] == "") {
                    return;
                }

                if ($this->combineFiltersBy == "and") {

                    $query->whereRaw($filters["sql"], $filters["bindings"]);

                } else {

                    $query->orWhereRaw($filters["sql"], $filters["bindings"]);

                }

            })

            ->distinct()
            ->orderBy($this->columnHeadings[0]['table'] . '.' . $this->columnHeadings[0]['column'], 'asc')
            ->pluck($this->columnHeadings[0]['table'] . '.' . $this->columnHeadings[0]['column'])
            ->toArray();
    }

    private function columnHasNull()
    {

        $table = $this->valuesFields[0]['table'];

        $query = $this->applyJoins($this->getNewConnection()->table($table), $table);

        $hasNull = $query

            ->when(!empty($this->getStaticFilters()), function ($query) {

                $filters = $this->getStaticFilters();

                $query->whereRaw($filters["sql"], $filters["bindings"]);

            })

            ->when($this->options['operation'] != "apply" && !empty($this->getSelectedParameters()), function ($query) {

                $filters = $this->getSelectedParameters();

                if ($this->combineFiltersBy == "and") {

                    $query->whereRaw($filters["sql"], $filters["bindings"]);

                } else {

                    $query->orWhereRaw($filters["sql"], $filters["bindings"]);

                }

            })

            ->whereNull($this->columnHeadings[0]['table'] . '.' . $this->columnHeadings[0]['column'])
            ->exists();

        return $hasNull;
    }

    private function composeSelect()
    {
        $headers = $this->columnHeadings ? $this->columnHeadings : $this->getHeaders();

        if ($headers[0]['function'] == "raw") {
            return $this->composeRawSelect();
        } else {
            return $this->composeSelectDateTime();
        }

    }

    private function composeRawSelect()
    {
        $grandTotalFunction = $this->options['grand_totals_function'];
        $columnValues       = $this->get3DColumnValues();
        $selectRawParts     = ["{$this->rowHeaders[0]['table']}.{$this->rowHeaders[0]['column']} as '{$this->rowHeaders[0]['label']}'"];
        $label              = $this->getGrandTotalsLabel();
        $headingsTable      = $this->columnHeadings[0]['table'];
        $headingsColumn     = $this->columnHeadings[0]['column'];
        $valuesTable        = $this->valuesFields[0]['table'];
        $valuesColumn       = $this->valuesFields[0]['column'];
        $sqlFunction        = $this->getSelectFunction();

        foreach ($columnValues as $origin) {

            if (!is_null($origin)) {
                $escapedOrigin = addslashes($origin);
                $comparison    = "$headingsTable.$headingsColumn = '{$escapedOrigin}'";
                $alias         = "`{$escapedOrigin}`";
            } else {
                $comparison = "$headingsTable.$headingsColumn IS NULL";
                $alias      = "`Null`"; // Do not wrap NULL in backticks
            }

            if ($this->valuesFields[0]['function'] == "count") {
                $sqlFunction      = "SUM";
                $selectRawParts[] = "{$sqlFunction}(CASE WHEN $comparison AND $valuesTable.$valuesColumn IS NOT NULL THEN 1 ELSE 0 END) AS $alias";
            } elseif ($this->valuesFields[0]['function'] == "count-distinct") {
                $sqlFunction      = "COUNT";
                $selectRawParts[] = "{$sqlFunction}(DISTINCT CASE WHEN $comparison THEN $valuesTable.$valuesColumn END) AS $alias";
            } elseif (in_array($this->valuesFields[0]['function'], ["min", "first-value"])) {
                $selectRawParts[] = "{$this->getSelectFunction()}(CASE WHEN $comparison THEN COALESCE($valuesTable.$valuesColumn, 0) ELSE NULL END) AS $alias";
            } elseif (in_array($this->valuesFields[0]['function'], ["average"]) && in_array($grandTotalFunction, ["sum", "max", "min", "count", "avg"])) {
                $selectRawParts[] = "{$this->getSelectFunction()}(DISTINCT CASE WHEN $comparison THEN COALESCE($valuesTable.$valuesColumn, 0) ELSE NULL END) AS $alias";
            } else {
                $selectRawParts[] = "{$this->getSelectFunction()}(CASE WHEN $comparison THEN COALESCE($valuesTable.$valuesColumn, 0) ELSE 0 END) AS $alias";
            }

        }

        $query = implode(", ", $selectRawParts);

        return $query;
    }

    private function composeSelectDateTime()
    {

        $columnValues        = $this->get3DColumnValues();
        $aggregatedDateTimes = $this->aggregateBy($columnValues);
        $selectRawParts      = ["{$this->rowHeaders[0]['table']}.{$this->rowHeaders[0]['column']} AS '{$this->rowHeaders[0]['label']}'"];
        $function            = $this->valuesFields[0]['function'];
        $columnTable         = $this->columnHeadings[0]['table'];
        $columnName          = $this->columnHeadings[0]['column'];
        $columnFunction      = $this->columnHeadings[0]['function'];
        $valueTable          = $this->valuesFields[0]['table'];
        $valueColumn         = $this->valuesFields[0]['column'];
        $selectFunction      = $this->getSelectFunction();

        $cases = function ($condition, $fallback = 0) use ($function, $valueTable, $valueColumn, $selectFunction) {

            switch ($function) {
                case "count":
                    return "SUM(CASE WHEN $condition AND $valueTable.$valueColumn IS NOT NULL THEN 1 ELSE 0 END)";
                case "count-distinct":
                    return "COUNT(DISTINCT CASE WHEN $condition THEN $valueTable.$valueColumn END)";
                case "first-value":
                case "average":
                case "min":
                    return "$selectFunction(CASE WHEN $condition THEN $valueTable.$valueColumn ELSE NULL END)";
                default:
                    return "$selectFunction(CASE WHEN $condition THEN $valueTable.$valueColumn ELSE $fallback END)";
            }

        };

        switch ($columnFunction) {
            case "weekday":

                foreach (["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"] as $day) {
                    $condition        = "DATE_FORMAT($columnTable.$columnName, '%W') = '$day'";
                    $selectRawParts[] = "{$cases($condition)} AS `$day`";
                }

                break;
            case "hour":

                foreach (range(0, 23) as $hour) {
                    $formattedHour    = str_pad($hour, 2, "0", STR_PAD_LEFT);
                    $condition        = "HOUR($columnTable.$columnName) = $hour";
                    $selectRawParts[] = "{$cases($condition)} AS `$formattedHour`";
                }

                break;
            default:

                foreach ($aggregatedDateTimes as $dateTimeLabel => $dateTime) {
                    $condition        = "$columnTable.$columnName BETWEEN '{$dateTime['start']}' AND '{$dateTime['end']}'";
                    $selectRawParts[] = "{$cases($condition)} AS `$dateTimeLabel`";
                }

        }

        $query = implode(", ", $selectRawParts);

        if ($this->columnHasNull()) {

            return $query . ", " . $cases("$columnTable.$columnName IS NULL") . " AS `Null`";
        }

        return $query;

    }

    private function add3DAxis($selectRawParts)
    {

        if ($this->getXAxisFunction() != "raw") {

            $this->query = $this->query->selectRaw(
                "{$this->getSelectTimeScale(true)} as '{$this->getXAxisLabel(true)}'," .
                $this->getYAxisWithFunction()
            );

        } else {
            $this->query = $this->query->selectRaw(
                "$selectRawParts"
            );
        }

        return $this;
    }

    private function addGroupBy()
    {
        $this->query = $this->query->groupByRaw(
            $this->getXAxisWithTimeScaleWithoutAlias()
        );

        return $this;
    }

    private function addOrderBy()
    {
        $this->query = $this->query->orderByRaw($this->getYAxisWithFunctionWithoutAlias() . " DESC");

        return $this;
    }

    private function getStaticFilters()
    {
        return (new DataFiltersGenerator($this->staticFilters, $this->combineFiltersBy))->build();
    }

    private function getSelectedParameters()
    {
        return (new ParametersGenerator($this->parameters, $this->parametersValues, $this->combineFiltersBy))->build();
    }

    private function addFilters()
    {

        if (!empty($this->staticFilters)) {
            $filters     = $this->getStaticFilters();
            $this->query = $this->query->whereRaw($filters["sql"], $filters["bindings"]);

        }

        return $this;
    }

    private function addParameters()
    {

        if (($this->options['operation'] != "apply" || empty($this->options['operation'])) && !empty($this->parameters)) {
            $filters = $this->getSelectedParameters();

// dd($this->getSelectedParameters());
            if (!empty($filters["sql"]) && $filters["sql"] != "") {
                $this->query->{$this->combineFiltersBy == "and" ? "whereRaw" : "orWhereRaw"}

                ($filters["sql"], $filters["bindings"]);
            }

        }

        return $this;
    }

    private function addOrderBy3D()
    {
        $this->query = $this->query->orderByRaw($this->getXAxisWithTimeScaleWithoutAlias() . " ASC");

        return $this;
    }

    private function getXAxisFunction()
    {
        return $this->getHeaders()[0]['function'];
    }

    private function getXAxisWithTimeScaleWithoutAlias()
    {

        if ($this->getXAxisFunction() != "raw") {
            return "{$this->getSelectTimeScale()}";
        }

        return "{$this->getSelectTimeScale()}({$this->getXAxis()})";

    }

    private function getYAxisWithFunctionWithoutAlias()
    {

        return "{$this->getSelectFunction()}({$this->getYAxis()})";

    }

    private function aggregateByQuarters(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {
            if (!$timestamp) {
                continue;
            }

            $date    = Carbon::parse($timestamp);
            $quarter = "Q" . $date->quarter . " " . $date->year;
            $start   = $date->copy()->firstOfQuarter()->format('Y-m-d H:i:s');
            $end     = $date->copy()->lastOfQuarter()->format('Y-m-d H:i:s');

            if (!isset($aggregated[$quarter])) {
                $aggregated[$quarter] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$quarter]['timestamps'][] = $timestamp;
        }

        uksort($aggregated, function ($a, $b) {
            [$qA, $yearA] = explode(" ", $a);
            [$qB, $yearB] = explode(" ", $b);

            $qA = (int) str_replace("Q", "", $qA);
            $qB = (int) str_replace("Q", "", $qB);

            return ($yearA == $yearB) ? $qA - $qB : $yearA - $yearB;
        });

        return $aggregated;
    }

    private function isDistinct()
    {
        return $this->valuesFields[0]['function'] == "count-distinct" ? "distinct" : "";
    }

    private function getYAxisFunction()
    {
        return $this->valuesFields[0]['function'] != "raw"
        ? \ucwords(str_replace("-", " ", $this->valuesFields[0]['function']))
        : "";
    }

    private function getYAxisLabel()
    {
        $function = $this->getYAxisFunction();

        return "$function of {$this->valuesFields[0]['label']}";
    }

    private function getYAxisWithFunction()
    {

        $isDistinct = $this->isDistinct();

        return "{$this->getSelectFunction()}($isDistinct {$this->getYAxis()}) as '{$this->getYAxisLabel()}'";
    }

    private function getSelectFunction()
    {

        switch ($this->valuesFields[0]['function']) {
            case 'count':
            case 'count-distinct':
                return 'COUNT';
                break;
            case 'sum':
                return 'SUM';
                break;
            case 'average':
                return 'AVG';
                break;
            case 'max':
                return 'MAX';
                break;
            case 'min':
            case 'first-value':
                return 'MIN';
            default:
                break;
        }

    }

    private function getSelectTimeScale($columns = false)
    {
        $headers = $columns ? $this->columnHeadings : $this->getHeaders();

        switch ($headers[0]['function']) {
            case 'hour':
                return "DATE_FORMAT({$this->getXAxis()},'%b %d, %Y %h %p')";
                break;
            case 'day':
                return "date_format({$this->getXAxis()},'%b %d, %Y')";
                break;
            case 'weekday':
                return "date_format({$this->getXAxis()},'%W')";
                break;
            case 'week':
                return "concat('Week ',cast(mid(yearweek({$this->getXAxis()}),5,2) as char),', ',cast(mid(yearweek({$this->getXAxis()}),1,4) as char))";
                break;
            case 'month':
                return "date_format({$this->getXAxis()},'%b %Y')";
                break;
            case 'quarter':
                return "concat('Q',cast(quarter({$this->getXAxis()}) as char),' ',date_format({$this->getXAxis()},'%Y'))";
                break;
            case 'year':
                return "year({$this->getXAxis()})";
                break;

            default:
                break;
        }

    }

    private function aggregateByYear(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {
            if (!$timestamp) {
                continue;
            }

            $date  = Carbon::parse($timestamp);
            $year  = $date->year;
            $start = $date->copy()->startOfYear()->format('Y-m-d H:i:s');
            $end   = $date->copy()->endOfYear()->format('Y-m-d H:i:s');

            if (!isset($aggregated["Year " . $year])) {
                $aggregated["Year " . $year] = ['start' => $start, 'end' => $end];
            }

        }

        ksort($aggregated);
        return $aggregated;
    }

    public function aggregateByMonth(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {
            if (!$timestamp) {
                continue;
            }

            $date  = Carbon::parse($timestamp);
            $month = $date->format('F Y');
            $start = $date->copy()->startOfMonth()->format('Y-m-d H:i:s');
            $end   = $date->copy()->endOfMonth()->format('Y-m-d H:i:s');

            if (!isset($aggregated[$month])) {
                $aggregated[$month] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$month]['timestamps'][] = $timestamp;
        }

        uksort($aggregated, function ($a, $b) {
            return strtotime("1 $a") - strtotime("1 $b");
        });

        return $aggregated;
    }

    public function aggregateByWeek(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {

            if (!$timestamp) {
                continue;
            }

            $date  = Carbon::parse($timestamp);
            $week  = "Week " . $date->weekOfYear . " " . $date->year;
            $start = $date->copy()->startOfWeek()->format('Y-m-d H:i:s');
            $end   = $date->copy()->endOfWeek()->format('Y-m-d H:i:s');

            if (!isset($aggregated[$week])) {
                $aggregated[$week] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$week]['timestamps'][] = $timestamp;
        }

        uksort($aggregated, function ($a, $b) {
            preg_match('/Week (\d+) (\d+)/', $a, $matchesA);
            preg_match('/Week (\d+) (\d+)/', $b, $matchesB);
            return ($matchesA[2] == $matchesB[2]) ? $matchesA[1] - $matchesB[1] : $matchesA[2] - $matchesB[2];
        });

        return $aggregated;
    }

    public function aggregateByWeekday(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {
            if (!$timestamp) {
                continue;
            }

            $date    = Carbon::parse($timestamp);
            $weekday = $date->format('l');
            $start   = $date->copy()->startOfDay()->format('Y-m-d');
            $end     = $date->copy()->endOfDay()->format('Y-m-d');

            if (!isset($aggregated[$weekday])) {
                $aggregated[$weekday] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$weekday]['timestamps'][] = Carbon::parse($timestamp)->format('Y-m-d');
        }

        $orderedWeekdays = ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"];

        uksort($aggregated, function ($a, $b) use ($orderedWeekdays) {
            return array_search($a, $orderedWeekdays) - array_search($b, $orderedWeekdays);
        });

        return $aggregated;
    }

    public function aggregateByDay(array $timestamps): array
    {
        $aggregated = [];
        foreach ($timestamps as $timestamp) {
            if (!$timestamp) {
                continue;
            }

            $date  = Carbon::parse($timestamp);
            $day   = $date->format('Y-m-d');
            $start = $date->copy()->startOfDay()->format('Y-m-d H:i:s');
            $end   = $date->copy()->endOfDay()->format('Y-m-d H:i:s');

            if (!isset($aggregated[$day])) {
                $aggregated[$day] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$day]['timestamps'][] = $timestamp;
        }

        ksort($aggregated);
        return $aggregated;
    }

    public function aggregateByHour(array $timestamps): array
    {
        $aggregated = [];

        foreach ($timestamps as $timestamp) {

            if (!$timestamp) {
                continue;
            }

            $date  = Carbon::parse($timestamp);
            $hour  = $date->format('Y-m-d H:00');
            $start = $date->copy()->startOfHour()->format('Y-m-d H:i:s');
            $end   = $date->copy()->endOfHour()->format('Y-m-d H:i:s');

            if (!isset($aggregated[$hour])) {
                $aggregated[$hour] = ['start' => $start, 'end' => $end, 'timestamps' => []];
            }

            $aggregated[$hour]['timestamps'][] = $timestamp;
        }

        ksort($aggregated);

        return $aggregated;
    }

    private function aggregateBy($timestamps)
    {
        $headers = $this->columnHeadings ?? $this->getHeaders();

        switch ($headers[0]['function']) {
            case 'hour':
                return $this->aggregateByHour($timestamps);
                break;
            case 'day':
                return $this->aggregateByDay($timestamps);
                break;
            case 'weekday':
                return $this->aggregateByWeekday($timestamps);
                break;
            case 'week':
                return $this->aggregateByWeek($timestamps);
                break;
            case 'month':
                return $this->aggregateByMonth($timestamps);
                break;
            case 'quarter':
                return $this->aggregateByQuarters($timestamps);
                break;
            case 'year':
                return $this->aggregateByYear($timestamps);
                break;
            default:
                break;
        }

    }

    private function getXAxis($columns = false)
    {
        $headers = $columns ? $this->columnHeadings : $this->getHeaders();
        return "{$headers[0]['table']}.{$headers[0]['column']}";
    }

    private function getYAxis()
    {
        return "{$this->valuesFields[0]['table']}.{$this->valuesFields[0]['column']}";
    }

    private function getGrandTotalsLabel()
    {
        return $this->options['grand_totals_label'];
    }

    private function add2DGrandTotals($data, $values)
    {
        if (!$this->options['show_grand_totals']) {
            return $data;
        }

        $function = $this->options['grand_totals_function'];
        $label    = $this->getGrandTotalsLabel();

        switch ($function) {
            case 'sum':
                $data['options']['total'] = [
                    $this->getXAxisLabel() => $label,
                    $this->getYAxisLabel() => $this->formatSingleValue($this->getYAxisLabel(), array_sum($values))
                ];
                break;
            case 'count':
                $data['options']['total'] = [
                    $this->getXAxisLabel() => $label,
                    $this->getYAxisLabel() => count($values)
                ];
                break;
            case 'avg':
                $data['options']['total'] = [
                    $this->getXAxisLabel() => $label,
                    $this->getYAxisLabel() => count($values) > 0 ? array_sum($values) / count($values) : 0
                ];
                break;
            case 'max':
                $data['options']['total'] = [
                    $this->getXAxisLabel() => $label,
                    $this->getYAxisLabel() => max($values)
                ];
                break;
            case 'min':
                $data['options']['total'] = [
                    $this->getXAxisLabel() => $label,
                    $this->getYAxisLabel() => min($values)
                ];
                break;
            default:
                break;
        }

        return $data;
    }

    private function prepare3dGrandTotals($data)
    {
        $label               = $this->getGrandTotalsLabel();
        $aggregationFunction = $this->options['grand_totals_function'];

        if (empty($data) || empty($data['data'])) {
            return [];
        }

        $data          = $data['data'];
        $results       = [];
        $overallTotals = [];
        $rowTotals     = [];
        foreach ($data as $row) {
            $row             = (array) $row;
            $rowValues       = [];
            $yearTotals      = [];
            $totals          = 0;
            $hasYears        = false;
            $identifierKey   = array_key_first($row);
            $identifierValue = $row[$identifierKey];

            foreach ($row as $key => $value) {
                if ($key === $identifierKey) {
                    continue;
                }

                if (is_numeric($key) && is_array($value)) {
                    $hasYears   = true;
                    $yearValues = [];

                    foreach ($value as $subKey => $subValue) {
                        if (!is_numeric($subValue) && !is_null($subValue)) {
                            continue;
                        }

                        $yearValues[] = !is_null($subValue) ? $subValue : 0;
                        $rowValues[]  = $subValue;

// Skip null row with zero value
                        if (!($identifierValue === 'Null' && $subValue == 0)) {
                            $overallTotals[$key][$subKey][] = $subValue;
                        }

                    }

                    $yearTotal                  = $this->applyAggregationFunction($aggregationFunction, $yearValues);
                    $row[$key]["{$key} $label"] = $yearTotal;
                    $yearTotals[]               = $yearTotal;

                    if (!($identifierValue === 'Null' && $yearTotal == 0)) {
                        $overallTotals[$key][$label][] = $yearTotal;
                    }

                } elseif (is_numeric($value)) {
                    $rowValues[] = $value;
                    if (!($key === 'Null' && $value == 0)) {
                        $yearTotals[] = $value;
                    }

// Skip null row with zero value
                    if (!($identifierValue === 'Null' && $value == 0)) {
                        $overallTotals[$key][] = $value;
                    }

                } elseif (is_null($value)) {
                    $rowValues[] = 0;
                    if (!($key === 'Null' && $value == 0)) {
                        $yearTotals[] = $value;
                    }

                }

            }

            $totals   = !empty($yearTotals) ? $yearTotals : $rowValues;
            $rowTotal = $this->applyAggregationFunction($aggregationFunction, $totals);
            if (!$hasYears && !($identifierValue === 'Null' && $rowTotal == 0)) {
                $overallTotals[$label][] = $rowTotal;
            }

            $row[$label] = $rowTotal;
            $results[]   = $row;

            $rowTotals[] = $rowTotal;
        }

        $grandTotalRow = [];
        if ($identifierKey) {
            $grandTotalRow[$identifierKey] = $label;
        }

        foreach ($overallTotals as $yearKey => $yearData) {
            if (is_array($yearData)) {
                foreach ($yearData as $subKey => $values) {
                    if (is_array($values)) {
                        $grandTotalRow[$yearKey][$subKey] = $this->applyAggregationFunction($aggregationFunction, $values);
                    } else {
                        $grandTotalRow[$yearKey] = $this->applyAggregationFunction($aggregationFunction, $yearData);
                        break;
                    }

                }

            } else {
                $grandTotalRow[$yearKey] = $yearData;
            }

        }

        $grandTotalRow[$label] = $this->applyAggregationFunction($aggregationFunction, $rowTotals);
        $results[]             = $grandTotalRow;
        return $results;
    }

    private function applyAggregationFunction($function, $values)
    {
        switch ($function) {
            case 'sum':return array_sum($values);
            case 'count':return count($values);
            case 'avg':return count($values) > 0 ? array_sum($values) / count($values) : 0;
            case 'max':return max($values);
            case 'min':return min($values);
            default:return null;
        }

    }

    private function add3dGrandTotals($data)
    {
        if (!$this->options['show_grand_totals']) {
            return $data;
        }

        $data['data']             = $this->prepare3dGrandTotals($data);
        $data['options']['total'] = [
            'label' => $this->getGrandTotalsLabel()
        ];
        return $data;
    }

    private function addValuesLabel($data)
    {
        $data['options']['value'] = $this->getYAxisLabel();
        return $data;
    }

    private function is2DTable()
    {
        return empty($this->rowHeaders) || empty($this->columnHeadings);
    }

    private function is3DTable()
    {
        return !empty($this->rowHeaders) && !empty($this->columnHeadings);
    }

    private function addGrandTotals($data, $values)
    {

        if ($this->is3DTable()) {

            $data = $this->add3dGrandTotals($data);
            $data = $this->addValuesLabel($data);

        } else {

            $data = $this->add2DGrandTotals($data, $values);

        }

        return $data;
    }

    private function reformatRowKeys($data)
    {

        foreach ($data as &$row) {
            $grandTotals = null;

            foreach ($row as $key => $value) {

                if (preg_match('/^\d+$/', $key)) {
                    $newKey       = " " . $key;
                    $row[$newKey] = $value;
                    unset($row[$key]);
                }

                if ($key === $this->getGrandTotalsLabel()) {
                    $grandTotals = $value;
                    unset($row[$key]);
                }

            }

            if ($grandTotals !== null) {
                $row[$this->getGrandTotalsLabel()] = $grandTotals;
            }

        }

        unset($row);

        return $data;
    }

    private function isComplexTable($data)
    {
        $datTimeDataTypes = ["date", "datetime", "timestamp"];

        return (bool) array_filter(array_keys($data), function ($key) {
            return preg_match('/\d{4}$/', $key);
        })
        && !in_array($this->columnHeadings[0]['function'], ["year"])
        && in_array($this->columnHeadings[0]['type'], $datTimeDataTypes)
        && !$this->hasSingleYear($data);
    }

    private function hasSingleYear($data)
    {

        if (!is_array($data) || empty($data)) {
            return false;
        }

        $years = [];

        if (array_keys($data) !== range(0, count($data) - 1)) {

            foreach ($data as $key => $value) {

                if (preg_match('/^(\d{4})[\s-]*(.*)$/', $key, $matches) ||
                    preg_match('/^(.*)[\s-](\d{4})$/', $key, $matches)) {
                    $year    = is_numeric($matches[1]) ? $matches[1] : $matches[2];
                    $years[] = $year;
                }

            }

        } else {

            foreach ($data as $item) {

                if (!is_array($item) && !is_object($item)) {
                    continue;
                }

                foreach ((array) $item as $key => $value) {

                    if (preg_match('/^(\d{4})[\s-]*(.*)$/', $key, $matches) ||
                        preg_match('/^(.*)[\s-](\d{4})$/', $key, $matches)) {
                        $year    = is_numeric($matches[1]) ? $matches[1] : $matches[2];
                        $years[] = $year;
                    }

                }

            }

        }

        return count(array_unique($years)) === 1;
    }

    private function transformArray($data)
    {
        $result     = [];
        $singleYear = $this->hasSingleYear($data);
        $nullItems  = [];

        foreach ($data as $item) {
            $newItem = [];
            $hasNull = false;

            foreach ($item as $key => $value) {

                if (is_object($item) && array_key_first((array) $item) === $key && $value === null) {
                    $value = "Null";
                }

                if ($value === "Null") {
                    $hasNull = true;
                }

                if ($singleYear) {
                    $newItem[$key] = $value;
                    continue;
                }

                if ((preg_match('/^(\d{4})[\s-]*(.*)$/', $key, $matches) ||
                    preg_match('/^(.*)[\s-](\d{4})$/', $key, $matches)) &&
                    !in_array($this->columnHeadings[0]['function'], ["year"])) {

                    $year = is_numeric($matches[1]) ? $matches[1] : $matches[2];
                    $rest = trim(is_numeric($matches[1]) ? $matches[2] : $matches[1], "- ");

                    if (!isset($newItem[$year])) {
                        $newItem[$year] = [];
                    }

                    if ($rest === '') {
                        $newItem[$year] = $value;
                    } else {
                        $newItem[$year][$rest] = $value;
                    }

                    continue;
                }

                $newItem[$key] = $value;
            }

            uksort($newItem, function ($a, $b) {
                return is_numeric($a) && is_numeric($b) ? $a - $b : 0;
            });

            if ($hasNull) {
                $nullItems[] = $newItem;
            } else {
                $result[] = $newItem;
            }

        }

        return array_merge($result, $nullItems);
    }

    private function getValuesPrefix()
    {
        return empty($this->options['currency']) ? $this->options['custom_unit'] : $this->options['currency'];
    }

    private function formatSingleValue($key, $value)
    {
        $prefix = $this->getValuesPrefix();

        if ($this->options['values_format'] == "standard_currency") {
            return ($key === $this->getHeaders()[0]['column'])
            ? $value
            : $prefix . " " . number_format($value, 2, '.', ',');
        }

        if ($this->options['values_format'] == "custom_unit") {
            return ($key === $this->getHeaders()[0]['column'])
            ? $value
            : round($value, 2) . " $prefix";
        }

        return is_numeric($value) ? round((float) $value, 2) : (empty($value) ? 0 : $value);
    }

    private function formatValues($array)
    {
        $result = [];

        foreach ($array as $key => $value) {

            if (is_array($value)) {
                $result[$key] = $this->formatValues($value);
            } else {
                $prefix = $this->getValuesPrefix();

                if ($this->options['values_format'] == "standard_currency") {

                    $result[$key] = ($key === $this->getHeaders()[0]['column']) ? $value : $prefix . " " . number_format($value, 2, '.', ',');

                } elseif ($this->options['values_format'] == "custom_unit") {

                    $result[$key] = ($key === $this->getHeaders()[0]['column']) ? $value : round($value, 2) . " $prefix";

                } else {

                    $result[$key] = is_numeric($value) ? round((float) $value, 2) : (empty($value) ? 0 : $value);

                }

            }

        }

        return $result;
    }

    private function getParameters()
    {
        $parameters = destructParametersBlocks($this->parameters);
        $data       = [];

        if (empty($parameters)) {
            return $data;
        }

        foreach ($parameters as $key => $parameter) {

            $table    = $parameter['table'];
            $column   = $parameter['column'];
            $function = $parameter['function'];

            $data[$parameter['label']]['type'] = $parameter['function'];

            $data[$parameter['label']]['data'] = $this->getParameterValues($data, $table, $column, $function);
        }

        return $data;

    }

    private function getParameterValues($data, $table, $column, $type)
    {

        switch ($type) {

            case 'multi-select':
                $values = $this->getNewConnection()
                    ->table($table)
                    ->select($column)
                    ->distinct()
                    ->orderBy($column, 'asc')
                    ->pluck($column)
                    ->toArray();

                $values = array_map(function ($value) {
                    return is_null($value) ? 'Null' : $value;
                }, $values);

                return $values;
            case 'numeric-slider':
                return [
                    'min' => $this->getNewConnection()
                        ->table($table)
                        ->selectRaw("MIN($column) as min")
                        ->value('min'),

                    'max' => $this->getNewConnection()
                        ->table($table)
                        ->selectRaw("MAX($column) as max")
                        ->value('max')
                ];

            default:
                return "";
        }

    }

    private function reorderColumns($data)
    {
        foreach ($data as &$row) {
            if (isset($row['Null'])) {
                $newRow          = [];
                $nullInserted    = false;
                $grandTotalLabel = $this->getGrandTotalsLabel();

                foreach ($row as $key => $value) {

// Insert Null just before the Total column
                    if ($key === $grandTotalLabel) {
                        $newRow['Null'] = $row['Null'];
                        $nullInserted   = true;
                    }

// Skip Null when copying to avoid duplication
                    if ($key !== 'Null') {
                        $newRow[$key] = $value;
                    }

                }

// If Total column doesn't exist, append Null at the end
                if (!$nullInserted) {
                    $newRow['Null'] = $row['Null'];
                }

                $row = $newRow;
            }

        }

        unset($row); // break the reference
        return $data;
    }

    public function removeNullKeyIfAllZero($data)
    {
        $shouldRemoveNullKey = true;

        foreach ($data as $subArray) {

            if (is_array($subArray) && array_key_exists('Null', $subArray)) {

                if ($subArray['Null'] != 0 && $subArray['Null'] != 0.0) {
                    $shouldRemoveNullKey = false;
                    break;
                }

            } else {
                $shouldRemoveNullKey = false;
                break;
            }

        }

        if ($shouldRemoveNullKey) {

            foreach ($data as &$subArray) {
                unset($subArray['Null']);
            }

            unset($subArray);
        }

        return $data;
    }

    public function convertNullsToZero($data)
    {

        foreach ($data as $key => &$value) {

            if (is_array($value)) {
                $value = $this->convertNullsToZero($value);
            } elseif (is_null($value)) {
                $value = 0;
            }

        }

        return $data;
    }

    private function formatData()
    {
        $results = $this->query->get()->toArray();
        $results = $this->transformArray($results);
        $data    = [
            'data' => $results
        ];

        $values = array_column($data['data'], $this->getYAxisLabel());

        $data['options']['column']  = !empty($this->columnHeadings);
        $data['options']['row']     = !empty($this->rowHeaders);
        $data['options']['complex'] = $this->isComplexTable($data['data'][0] ?? []);

        $data['options']['parameters']        = $this->getParameters();
        $data['options']['pivot_table_title'] = $this->options['pivot_table_title'];

        $data['data'] = $this->reformatRowKeys($data['data']);
        $data['data'] = $this->convertNullsToZero($data['data']);
        $data         = $this->addGrandTotals($data, $values);
        $data['data'] = $this->formatValues($data['data']);

        $data['data'] = $this->reorderColumns($data['data']);
        $data['data'] = $this->removeNullKeyIfAllZero($data['data']);

        return $data;
    }

    private function build2DTable()
    {
        return $this->initializeConnection()
            ->addTables()
            ->addAxis()
            ->addGroupBy()
            ->addFilters()
            ->addParameters()
            ->addOrderBy();
    }

    private function build3DTable()
    {
        $selectRawParts = $this->composeSelect();
        return $this->initializeConnection()
            ->addTables()
            ->add3DAxis($selectRawParts)
            ->addGroupBy()
            ->addFilters()
            ->addParameters()
            ->addOrderBy3D();
    }

    private function handleQueryException(QueryException $e)
    {
        $errors['sqlErrors'] = [
            'details' => $e->getMessage(),
            'sql'     => method_exists($this->query, 'toRawSql') ? $this->query->toRawSql() : 'Unavailable'
        ];

        return [$errors, Response::HTTP_INTERNAL_SERVER_ERROR];
    }

    private function handleGeneralException(\Exception $e)
    {
        return response()->json($e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function build()
    {
        try {

            if ($this->is2DTable()) {
                $this->build2DTable();
            } else {
                $this->build3DTable();
            }

            return $this->formatData();
        } catch (QueryException $e) {
            return $this->handleQueryException($e);
        } catch (\Exception $e) {
            return $this->handleGeneralException($e);
        }

    }

}
