<?php

namespace App\Http\Controllers\ScheduledReports;

use App\Models\ScheduledTask;
use App\Services\GroupService;
use App\Services\ResourceService;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use App\Http\Middleware\OnlyFullVersion;
use App\Services\ScheduledReportService;
use App\Exceptions\DisabledTaskException;
use App\Services\ScheduledBatchReportService;
use App\Http\Requests\ScheduledTask\StoreScheduledTaskRequest;

class ScheduledReportsController extends Controller
{
    public $resourceService;
    public $groupService;
    public $scheduledReportService;

    public function __construct(
        ScheduledReportService $scheduledReportService,
        GroupService $groupService,
        ResourceService $resourceService
    ) {
        $this->scheduledReportService = $scheduledReportService;
        $this->groupService           = $groupService;
        $this->resourceService        = $resourceService;

        $this->middleware(OnlyFullVersion::class)->only([
            'stop',
            'send',
            'resume',
            'store',
            // 'update',
            'log',
            'destroy'
        ]);

    }

    public function index()
    {
        Artisan::call('config:clear');

        $tasks = $this->scheduledReportService->getScheduledReports();
        return view('dashboard.scheduled-reports.index', compact('tasks'));
    }

    public function log(ScheduledTask $scheduledTask)
    {
        $logs = $this->scheduledReportService->show($scheduledTask);
        return view('dashboard.scheduled-reports.logs', compact('scheduledTask', 'logs'));
    }

    public function send(ScheduledTask $scheduledTask)
    {
        try {
            (new ScheduledBatchReportService())->runTask($scheduledTask);

            return response()->json([
                'message' => trans('scheduled-report.messages.manual-send')
            ]);
        } catch (DisabledTaskException $e) {
            return response()->json([
                'message' => $e->getMessage()
            ]);

        }

    }

    public function create()
    {
        $groups  = $this->groupService->getAllGroups();
        $reports = $this->resourceService->getAllResources();

        if (checkDemo()) {
            $scheduledTask = $this->scheduledReportService->getRecentScheduledReport();
            return view('dashboard.scheduled-reports.add-demo', compact('scheduledTask', 'groups', 'reports'));

        }

        return view('dashboard.scheduled-reports.create', compact('groups', 'reports'));
    }

    public function edit(ScheduledTask $scheduledTask)
    {
        $groups  = $this->groupService->getAllGroups();
        $reports = $this->resourceService->getAllResources();
        return view('dashboard.scheduled-reports.edit', compact('scheduledTask', 'groups', 'reports'));
    }

    public function store(StoreScheduledTaskRequest $request)
    {
        $message = $this->scheduledReportService->store($request);
        session()->flash($message['type'], $message['body']);
        return redirect()->back();

    }

    public function update(ScheduledTask $scheduledTask, StoreScheduledTaskRequest $request)
    {

        if (checkDemo()) {
            return redirect()->route('scheduled-reports.index');
        }

        $message = $this->scheduledReportService->update($scheduledTask, $request);
        session()->flash($message['type'], $message['body']);
        return redirect()->back();

    }

    public function stop(ScheduledTask $scheduledTask)
    {
        $this->scheduledReportService->stop($scheduledTask);

        return response()->json([
            'message' => trans('scheduled-report.messages.stop', [
                'report' => $scheduledTask->name
            ])
        ]);
    }

    public function resume(ScheduledTask $scheduledTask)
    {
        $this->scheduledReportService->resume($scheduledTask);

        return response()->json([
            'message' => trans('scheduled-report.messages.resume', [
                'report' => $scheduledTask->name
            ])
        ]);
    }

    public function destroy(ScheduledTask $scheduledTask)
    {
        $this->scheduledReportService->destroy($scheduledTask);

        return response()->json([
            'message' => trans('scheduled-report.messages.delete', [
                'report' => $scheduledTask->name
            ])
        ]);
    }

}
