<?php

use App\Models\User;
use App\Models\Setting;
use App\Models\Resource;
use App\Services\ScheduledReportService;
use App\Http\Controllers\Auth\LoginController;

if (!function_exists("generateReport")) {

    function generateReport(Resource $report, $task)
    {
        if(strtolower(getInstallationMethod()) == "composer") {
            $url = $report->getReportUrl() . "?export=pdf1&save=1";
        } else {
            $url = getInternalDocker() . getReportPath($report->url)  . "?export=pdf1&save=1";
        }
        try {
            $ch = curl_init();

            curl_setopt($ch, CURLOPT_URL, $url);

            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'INTERNAL-KEY: ' . getSharedKey(),
                'TASK-NAME: ' . $task->title
            ]);

            $response = curl_exec($ch);

            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode == 200) {
                return true;
            } else {
                throw new Exception("Can't generate report");

            }

        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }
}


if (!function_exists("getSharedKey")) {
    function getSharedKey()
    {
        return Setting::where('settings_key', 'shared_key')->first()->settings_value;
    }
}

if (!function_exists("getReportPdf")) {

    function getReportPdf($task)
    {

        $reportPath =  getAttachmentPath();
        $reportName = str_replace(' ', '', $task->title).'.pdf';


        @mkdir($reportPath, 0755, true);

        return $reportPath  . $reportName;
    }
}

if (!function_exists("deleteAllAttachments")) {

    function deleteAllAttachments()
    {
        $attachments = glob(getAttachmentPath() . '*.pdf');
        foreach($attachments as $attachment) {
            if(is_file($attachment)) {
                unlink($attachment);
            }
        }

    }
}

if (!function_exists("deleteTaskAttachment")) {

    function deleteTaskAttachment($task)
    {
        $attachment = getAttachmentPath() . str_replace(' ', '', $task->title) .'.pdf';
        if(is_file($attachment)) {
            unlink($attachment);
        }

    }
}


if (!function_exists("getAttachmentPath")) {
    function getAttachmentPath()
    {
        return base_path() ."/". ltrim(Setting::where('settings_key', 'pdf_attachments')->first()->settings_value, "/") ;
    }
}

if (!function_exists("updateSuccessTask")) {


    function updateSuccessTask($task, $manual, $beginTime, $totalSent, $totalFailed, $sentIds)
    {
        $nextSendingTime = $manual
        ? $task->next_sending_date
        : ScheduledReportService::recaluclateNextSendingDate($task);
        // dd($totalSent);
        $task->update([
            'last_sending_datetime' => now()->toDateTimeString(),
            'last_status' => 'success',
            'next_sending_date' => $nextSendingTime,
            'retry_count' => 0,
            'recieved_users' => $sentIds,
            'worked_now' => 0
        ]);

        $task->taskHistory()->create([
            'start_sending_date' => $beginTime,
            'end_sending_date' => now()->toDateTimeString(),
            'error_message' => null,
            'status' => 'success',
            'next_sending_date' => $nextSendingTime,
            'total_sent' => $totalSent,
            'total_failed' => $totalFailed
        ]);

    }

}

if (!function_exists("updateFailedTask")) {
    function updateFailedTask(
        $task,
        $manual,
        $error,
        $beginTime,
        $totalSent,
        $totalFailed,
        $sendIds,
        $status
    ) {
        $errorMsg = buildErrorMsg($error->getMessage(), getExceptionMethod($error), $error->getLine(), $error->getTraceAsString());

        if($task->retry_count >= getMailRetriesCount()) {
            $retryCount = 0;

            $nextSendingTime = ScheduledReportService::recaluclateNextSendingDate($task);

            $sendIds = null;

        } else {
            $retryCount = $task->retry_count + 1;

            $nextSendingTime = $task->next_sending_date;

            if(in_array($task->last_status, ['incomplete','failed'])) {
                $task->recieved_users = !is_null($task->recieved_users) && is_array($task->recieved_users)
                ? $task->recieved_users
                : [];

                $sendIds = is_array($sendIds) && count($sendIds) > 0
                ? array_merge($task->recieved_users, $sendIds)
                : $task->recieved_users;
            }

        }

        if($manual) {
            $nextSendingTime = $task->next_sending_date;
        }

        $task->update([
            'last_sending_datetime' => now()->toDateTimeString(),
            'last_status' => $status,
            'recieved_users' => $sendIds,
            'retry_count' => $retryCount,
            'next_sending_date' => $nextSendingTime,
            'worked_now' => 0
        ]);

        $task->taskHistory()->create([
            'start_sending_date' => $beginTime,
            'end_sending_date' => now()->toDateTimeString(),
            'error_message' => $errorMsg,
            'status' => $status,
            'next_sending_date' => $task->next_sending_date,
            'total_sent' => $totalSent,
            'total_failed' => $totalFailed
        ]);

    }
}

if (!function_exists("updateTaskSendingTime")) {

    function updateTaskSendingTime($task)
    {
        $task->update([
            'next_sending_date' => ScheduledReportService::recaluclateNextSendingDate($task)
        ]);
    }
}

if (!function_exists("getMailRetriesCount")) {

    function getMailRetriesCount($default = 3)
    {
        return !is_null(Setting::where('settings_key', 'mail_retry_count')->first())
        ? Setting::where('settings_key', 'mail_retry_count')->first()->settings_value
        : $default;
    }

}

if (!function_exists("buildErrorMsg")) {

    function buildErrorMsg($errorMsg, $method, $line, $stackTrace)
    {
        return "The task failed at method: $method , Line: $line , Exact error message: $errorMsg ,  stack trace : $stackTrace";
    }
}


if (!function_exists("getResourceUrl")) {
    function getResourceUrl($resource)
    {
        switch ($resource->getResourceType()) {
            case 'Blank Reports':
                $url = app()->runningInConsole() ? rtrim(getBaseUrl(), '/') .route('reports.show', $resource, false) : route('reports.show', $resource, true);
                return makeLineHyperLink($url) ;
                break;
            case 'Charts':
                $url = app()->runningInConsole() ? rtrim(getBaseUrl(), '/') .route('charts.details', $resource, false) : route('charts.details', $resource, true);

                return makeLineHyperLink($url) ;
                break;
            case 'KPIs / Metrics':
                $url = app()->runningInConsole() ? rtrim(getBaseUrl(), '/') .route('metrics.drill', $resource, false) : route('metrics.drill', $resource, true);

                return makeLineHyperLink($url);
                break;
            default:
                break;
        }
    }
}

if (!function_exists("makeLineHyperLink")) {
    function makeLineHyperLink($url)
    {
        return '<a href="'.$url.'">'.$url.'</a>';
    }
}

if (!function_exists("getResourceStatement")) {
    function getResourceStatement($resource)
    {
        switch ($resource->getResourceType()) {
            case 'Blank Reports':
                return $resource->name . ' report : '. getResourceUrl($resource);
                break;
            case 'Charts':
                return  $resource->name . ' chart : '. getResourceUrl($resource);
                break;
            case 'KPIs / Metrics':
                return $resource->name . ' metric : '. getResourceUrl($resource);
                break;
            default:
                break;
        }
    }
}

if (!function_exists("getInternalDocker")) {
    function getInternalDocker()
    {
        return "http://localhost";
    }
}