<?php

namespace App\Services;

use Carbon\Carbon;
use App\Models\Resource;
use App\Models\SMTPMails;
use App\Models\ScheduledTask;
use App\Repositories\ScheduledTaskRepository;

class ScheduledReportService
{
    private $moduleService;
    private $groupService;
    private $scheduledTaskRepository;

    public function __construct(
        ModuleService $moduleService,
        GroupService $groupService,
        ScheduledTaskRepository $scheduledTaskRepository
    ) {
        $this->moduleService           = $moduleService;
        $this->groupService            = $groupService;
        $this->scheduledTaskRepository = $scheduledTaskRepository;
    }

    public function getScheduledReports()
    {
        return ScheduledTask::orderBy('created_at', 'desc')->get();
    }

    public function stop($scheduledTask)
    {
        $this->checkScheduledReportCreatorPermission($scheduledTask);

        $scheduledTask->update([
            'last_status' => 'stopped',
            'edited_by'   => auth()->user()->user_ID
        ]);
    }

    public function resume($scheduledTask)
    {
        $this->checkScheduledReportCreatorPermission($scheduledTask);

        $scheduledTask->update([
            'last_status' => 'active',
            'edited_by'   => auth()->user()->user_ID
        ]);
    }

    public function show($scheduledTask)
    {
        return $scheduledTask->taskHistory()->orderBy('end_sending_date')->get();
    }

    public function store($request)
    {

        if (is_null(SMTPMails::defaultSmtpMail()->first())) {
            return $this->errorMessage(trans('scheduled-report.messages.smtp-not-found'));
        }

        try {
            $this->scheduledTaskRepository->store($request->validated());
            return $this->successMessage();
        } catch (\Exception $e) {
            return $this->errorMessage($e->getMessage());
        }

    }

    public function update($scheduledTask, $request)
    {
        try {
            $this->scheduledTaskRepository->update($scheduledTask, $request->validated());
            return $this->successMessage();
        } catch (\Exception $e) {
            return $this->errorMessage($e->getMessage());
        }

    }

    public function destroy($scheduledTask)
    {
        $this->checkScheduledReportCreatorPermission($scheduledTask);

        $scheduledTask->update([
            'deleted_by' => auth()->user()->user_ID
        ]);

        $scheduledTask->resources()->delete();
        $scheduledTask->delete();
    }

    private function checkScheduledReportCreatorPermission(ScheduledTask $scheduledTask)
    {

        if (!auth()->user()->isAdminOrOwner()) {
            abort(403, trans('report.unauthorized'));
        }

    }

    private function checkReportAccessPermission(Resource $report)
    {

        if (
            !(auth()->user()->isAdminOrOwner() || strtolower($report->access_control_type) == "public" ||
                in_array(auth()->user()->group_id, $report->resource_permissions->pluck("group_ID")->toArray()))
        ) {
            abort(403, trans('report.access_unautorized'));
        }

    }

    private function successMessage()
    {
        $managePageUrl = route('scheduled-reports.index');

        return [
            "type" => "success",
            "body" => "Scheduled task is saved successfully, <a href='" . $managePageUrl . "'>Click Here</a> to return back to the Manage scheduled reports page"
        ];
    }

    private function errorMessage($error)
    {
        return [
            "type" => "failed",
            "body" => "Error in saving the scheduled task: $error"
        ];

    }

    public static function caluclateNextSendingDate($frequency, $beginDate, $sendingTime, $sendingWeeklyDay, $sendingMonthlyDay)
    {

        switch ($frequency) {
            case 'hourly':
                return static::calculateHourlySendingDate($beginDate);
                break;
            case 'daily':
                return static::calculateDailySendingDate($beginDate, $sendingTime);
                break;
            case 'weekly':
                return static::calculateWeeklySendingDate($beginDate, $sendingTime, $sendingWeeklyDay);
                break;
            case 'monthly':
                return static::calculateMonthlySendingDate($beginDate, $sendingTime, $sendingMonthlyDay);
                break;
            default:
                break;
        }

    }

    public static function recaluclateNextSendingDate($task)
    {

        switch ($task->frequency) {
            case 'hourly':
                return static::calculateHourlySendingDate($task->begin_date);
                break;
            case 'daily':
                return static::calculateDailySendingDate($task->begin_date, $task->sending_time);
                break;
            case 'weekly':
                return static::calculateWeeklySendingDate($task->begin_date, $task->sending_time, $task->sending_weekly_day);
                break;
            case 'monthly':
                return static::calculateMonthlySendingDate($task->begin_date, $task->sending_time, $task->sending_monthly_day);
                break;
            default:
                break;
        }

    }

    private static function calculateHourlySendingDate($beginDate)
    {
        $baseDate = static::getBaseDate($beginDate);
        return $baseDate->addHour()->minute(0)->second(0)->toDateTimeString();
    }

    private static function calculateDailySendingDate($beginDate, $sendingTime)
    {
        $baseDate = static::getBaseDate($beginDate);
        return $baseDate->hour($sendingTime) > now()
        ? $baseDate->startOfDay()->hour($sendingTime)->toDateTimeString()
        : $baseDate->addDay()->startOfDay()->hour($sendingTime)->toDateTimeString();
    }

    private static function calculateWeeklySendingDate($beginDate, $sendingTime, $sendingWeeklyDay)
    {
        $baseDate = static::getBaseDate($beginDate);

        return $baseDate->is($sendingWeeklyDay) && Carbon::createFromTime($sendingTime)->hour > now()->hour
        ? $baseDate->startOfDay()->hour($sendingTime)->toDateTimeString()
        : $baseDate->next($sendingWeeklyDay)->hour($sendingTime)->toDateTimeString();
    }

    private static function calculateMonthlySendingDate($beginDate, $sendingTime, $sendingMonthlyDay)
    {
        $baseDate = static::getBaseDate($beginDate);

        if ($sendingMonthlyDay != "last") {

            if ($baseDate->day < $sendingMonthlyDay) {
                return $baseDate->day($sendingMonthlyDay)->startOfDay()
                    ->hour($sendingTime)
                    ->toDateTimeString();
            } elseif ($baseDate->day == $sendingMonthlyDay && Carbon::createFromTime($sendingTime)->hour > now()->hour) {
                return $baseDate->startOfDay()->hour($sendingTime)->toDateTimeString();
            } else {
                return $baseDate->addMonthNoOverflow()->day($sendingMonthlyDay)->startOfDay()
                    ->hour($sendingTime)->toDateTimeString();
            }

        }

        if ($baseDate->isLastOfMonth() && Carbon::createFromTime($sendingTime)->hour <= now()->hour) {
            return $baseDate->addMonthNoOverflow()->endOfMonth()->hour($sendingTime)
                ->minute(0)->second(0)->toDateTimeString();
        } else {
            return $baseDate->endOfMonth()->startOfDay()
                ->hour($sendingTime)
                ->toDateTimeString();
        }

    }

    private static function getBaseDate($beginDate)
    {
        return Carbon::parse($beginDate) > now()->startOfDay() ? Carbon::parse($beginDate) : now();
    }

    private function mapStringDaysToCarbon($weekDay)
    {

        switch ($weekDay) {
            case 'monday':
                return Carbon::MONDAY;
                break;
            case 'tuesday':
                return Carbon::TUESDAY;
                break;
            case 'wednesday':
                return Carbon::WEDNESDAY;
                break;
            case 'thrusday':
                return Carbon::THURSDAY;
                break;
            case 'friday':
                return Carbon::FRIDAY;
                break;
            case 'saturday':
                return Carbon::SATURDAY;
                break;
            case 'sunday':
                return Carbon::SUNDAY;
                break;
            default:
                break;
        }

    }

    public function getRecentScheduledReport()
    {
        return ScheduledTask::orderBy('created_at', 'desc')->first();
    }

}
