<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Str;
use App\Models\ScheduledTask;
use App\Jobs\SendScheduledSingleReportJob;
use App\Models\ScheduledTaskSendingLog;
use Illuminate\Support\Facades\Storage;
use App\Exceptions\DisabledTaskException;
use Exception;
use Illuminate\Console\Scheduling\Schedule;

class ScheduledBatchReportService
{
    private $beginTime;
    public function __construct()
    {
        $this->beginTime = now();
    }

    public function runBatch()
    {
        $this->updateEndedTasks();
        $this->initializeTaskWorkedNow();

        deleteAllAttachments();

        $tasks = $this->getCurrentTasks();
        if(count($tasks)) {
            foreach ($tasks as $task) {

                try {
                    $this->setTaskWorkedNow($task);

                    $receivers = getReceivers($task, $task->groups);
                    $body = getBodies($task->resources->first()->resource->first(), $task->body, $receivers);

                    $this->checkBaseUrl();
                    generateReport($task->resources->first()->resource->first(), $task);
                    $report = getReportPdf($task);

                    dispatch(new SendScheduledSingleReportJob($task, $report, $receivers, $body));
                } catch(\Exception $exception) {

                    updateFailedTask(
                        $task,
                        false,
                        $exception,
                        $this->beginTime,
                        0,
                        count($receivers),
                        [],
                        'failed'
                    );


                }
            }
        }

        $this->createLog($this->beginTime, now()->toDateTimeString(), count($tasks));

    }

    public function runTask(ScheduledTask $task)
    {
        if($this->checkCurrentTaskCanRun($task)) {
            throw new DisabledTaskException();
        }

        deleteAllAttachments();

        $this->setTaskWorkedNow($task);
        try {
            $receivers = getReceivers($task, $task->groups, true);
            $body = getBodies($task->resources->first()->resource->first(), $task->body, $receivers);

            $this->checkBaseUrl();
            generateReport($task->resources->first()->resource->first(), $task);
            $report = getReportPdf($task);

            dispatch(new SendScheduledSingleReportJob($task, $report, $receivers, $body, true));

        } catch(\Exception $exception) {

            updateFailedTask(
                $task,
                false,
                $exception,
                $this->beginTime,
                0,
                count($receivers),
                [],
                'failed'
            );


        }

        $this->createLog($this->beginTime, now()->toDateTimeString(), 1);

    }


    private function createLog($created_at, $endedAt, $taskCount)
    {

        if($this->CheckLogStatus()) {
            ScheduledTaskSendingLog::create([
                'created_at' => $created_at,
                'ended_at' => $endedAt,
                'scheduled_tasks_count' => $taskCount,
            ]);
        }
    }


    private function updateEndedTasks()
    {
        ScheduledTask::whereDate('end_date', '<', now())->update([
            'last_status' => 'ended',
            'next_sending_date' => null
        ]);
    }

    private function setTaskWorkedNow($task)
    {
        $task->update(['worked_now' => 1]);
    }
    private function initializeTaskWorkedNow()
    {
        ScheduledTask::query()->update(['worked_now' => 0]);
    }

    private function getCurrentTasks()
    {
        return ScheduledTask::where('next_sending_date', '<=', now())
        ->where('worked_now', 0)
        ->whereNotIn('last_status', ['stopped','ended','deleted'])->get();
    }

    private function checkCurrentTaskCanRun($task)
    {
        if(in_array($task->last_status, ['stopped','ended','deleted'])) {
            return true;
        }

        return false;
    }

    private function CheckLogStatus()
    {
        return Setting::where('settings_key', 'log_sending_batch_reports_task')->first()->settings_value;
    }

    private function checkBaseUrl()
    {
        $baseUrl = Setting::where('settings_key', 'base_url')->first()->settings_value;
        if(is_null($baseUrl)) {
            throw new Exception("Can't generate a report");
        }
    }

}
