<?php

namespace App\Providers;

use App\Models\Module;
use App\Models\Setting;
use App\Models\Dashboard;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\ServiceProvider;
use App\Jobs\SendScheduledSingleReportJob;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register()
    {

        if (!Config::get("srm_install_config")) {
            $this->app['config']['app.key'] = "base64:" . base64_encode(random_bytes(32));
        } else {
            $this->app['config']['app.key'] = Config::get("srm_install_config.security_key");
        }

        $this->app['config']['database.connections.mysql'] = Config::get("srm_db_config");
    }

    /**
     * Bootstrap any application services.
     */
    public function boot()
    {

        Paginator::useBootstrapFour();

        if (!Config::get("srm_db_config")) {
            Config::set('session.driver', 'array');
        }

        if ($this->app['config']['database.connections.mysql'] && !is_null(Setting::where('settings_key', 'timezone')->first())) {

            $this->setTimeZone();
            $this->setJobsTable();
            $this->setFailedJobsTable();

            Queue::after(function ($event) {
                $job         = $event->job;
                $jobInstance = unserialize($job->payload()['data']['command']);

                if ($jobInstance instanceof SendScheduledSingleReportJob) {
                    $jobInstance->after();
                }

            });

            View::composer('*', function ($view) {

                if (!is_null(auth()->user())) {
                    $userDashboards = Dashboard::where('user_id', auth()->user()->user_ID)->orderBy('order', 'asc')->limit(5)->get();

                    $blankReportModule = Module::getBlankReportModule();

                    $chartModule = Module::getChartModule();

                    $metricModule = Module::getMetricModule();

                    $view->with([
                        'userDashboards'    => $userDashboards,
                        'blankReportModule' => $blankReportModule,
                        'chartModule'       => $chartModule,
                        'metricModule'      => $metricModule
                    ]);

                }

            });

        }

    }

    private function setTimeZone()
    {
        $timezone = Setting::where('settings_key', 'timezone')->first()->settings_value;
        config(['app.timezone' => $timezone]);
        date_default_timezone_set($timezone);
    }

    private function setJobsTable()
    {
        $prefix = config('srm_config.installer.table_prefix', 'srm_');
        config(['queue.connections.database.table' => $prefix . 'jobs']);
    }

    private function setFailedJobsTable()
    {
        $prefix = config('srm_config.installer.table_prefix', 'srm_');
        config(['queue.failed.table' => $prefix . 'failed_jobs']);
    }

}
