<?php

namespace App\Jobs;

use ArrayIterator;
use App\Models\User;
use MultipleIterator;
use App\Models\Resource;
use App\Models\SMTPMails;
use App\Mail\SendReportMail;
use Illuminate\Http\Request;
use App\Mails\ReportTestMail;
use App\Models\ScheduledTask;
use Illuminate\Bus\Queueable;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Queue\SerializesModels;
use App\Services\ScheduledReportService;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use App\Http\Middleware\DeleteAttachmentAfterJob;
use App\Mails\ScheduledTaskMail\ScheduledReportMail;

class SendScheduledSingleReportJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public $task;
    public $report;
    public $receivers;
    public $body;

    public $beginTime;
    public $totalSent;
    public $totalFailed;
    public $sendIds;
    public $manual;

    public function __construct(ScheduledTask $task, $report, $receivers, $body, $manual = false)
    {
        $this->task = $task;
        $this->report = $report;
        $this->receivers = $receivers;
        $this->body = $body;

        $this->beginTime = now();
        $this->totalSent = 0;
        $this->totalFailed = 0;
        $this->sendIds = [];
        $this->manual = $manual;
    }

    public function handle(): void
    {

        $mailInfo = $this->getEmailInfoIterator($this->receivers, $this->body);
        $this->updateIfManual($this->task, ['recieved_users' => null,'retry_count' => 0]);

        try {
            $this->checkSMTPExist();
        } catch (\Exception $exception) {

            $this->totalFailed = count($this->receivers);

            updateFailedTask(
                $this->task,
                $this->manual,
                $exception,
                $this->beginTime,
                $this->totalSent,
                $this->totalFailed,
                $this->sendIds,
                'failed'
            );

            return; // Exit the handle method early since SMTP check failed
        }

        foreach ($mailInfo as [$receiver, $body]) {
            try {
                (new ScheduledReportMail($this->task, $this->report, $receiver, $body))->send();
                $this->totalSent++;
                $this->sendIds[] = $receiver['id'];
            } catch (\Exception $error) {
                $this->totalFailed++;
                $lastError = $error; // Store the last occurred exception

            }
        }
        if ($this->totalSent > 0 && $this->totalFailed > 0) {
            updateFailedTask(
                $this->task,
                $this->manual,
                !is_null($lastError) ? $lastError : new \Exception('There is an error'),
                $this->beginTime,
                $this->totalSent,
                $this->totalFailed,
                $this->sendIds,
                'incomplete'
            );
        } elseif ($this->totalSent == 0) {
            updateFailedTask(
                $this->task,
                $this->manual,
                !is_null($lastError) ? $lastError : new \Exception('There is an error'),
                $this->beginTime,
                $this->totalSent,
                $this->totalFailed,
                $this->sendIds,
                'failed'
            );
        }
        if($this->totalFailed == 0 && $this->totalSent == count($this->receivers)) {
            updateSuccessTask($this->task, $this->manual, $this->beginTime, $this->totalSent, $this->totalFailed, $this->sendIds);
        }


    }

    private function getEmailInfoIterator($receivers, $body)
    {
        $mi = new MultipleIterator();
        $mi->attachIterator(new ArrayIterator($receivers));
        $mi->attachIterator(new ArrayIterator($body));

        return $mi;
    }

    private function updateIfManual($task, $data)
    {
        if($this->manual && is_array($data)) {
            $task->update($data);
        }
    }

    private function checkSMTPExist()
    {
        $smtp = SMTPMails::defaultSmtpMail()->first();

        if(is_null($smtp)) {
            throw new \Exception('SMTP is not exist');
        }
    }

    public function failed(\Throwable $exception)
    {
        updateFailedTask(
            $this->task,
            $this->manual,
            $exception,
            $this->beginTime,
            $this->totalSent,
            $this->totalFailed,
            $this->sendIds,
            'failed'
        );
    }

    public function after()
    {
        deleteTaskAttachment($this->task);
    }



}