<?php

namespace App\Http\Requests;

use App\Models\Group;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Http\FormRequest;

class StoreAccessKeyRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $prefix = config("srm_config.installer.table_prefix", "srm_");

        return [
            'label' => ["required", "max:50", "unique:{$prefix}access_keys,label"],
            "type"  => ["required", "in:limited,full"],
            "group" => [function ($attribute, $value, $fail) {
                if ($this->input('type') == 'limited') {
                    if (!$value || empty($value)) {
                        $fail(trans("access-keys.validations.custom.group"));
                    }

                }

            },Rule::In(Group::pluck('group_ID')->toArray())],
            "origin" => ["required"],
            "allowed_origins" => [function ($attribute, $value, $fail) {
                if ($this->input('origin') == 'on' && (empty($value) || is_null($value))) {
                    $fail(trans("access-keys.validations.custom.origins_empty"));

                }

                $origins = preg_split('/\r\n|\r|\n/', trim($value));
                foreach ($origins as $origin) {
                    if (!filter_var($origin, FILTER_VALIDATE_URL)) {
                        $fail(trans("access-keys.validations.custom.invalid_url"));
                    }
                }


            }],


        ];
    }

    public function prepareForValidation()
    {
        $this->merge([
            'display_value' => $this->has('display_value') ? $this->boolean('display_value') : 0
        ]);
    }

    public function messages()
    {
        return [
            "label.required" => trans("access-keys.validations.required.label"),
            "label.max" => trans("access-keys.validations.required.label"),
            "label.unique"   => trans("access-keys.validations.unique.label"),
            "type.required"  => "Please select barcode type."
        ];
    }

}
