<?php

namespace App\Http\Requests\PivotTable;

use Carbon\Carbon;
use App\Models\Resource;
use Illuminate\Foundation\Http\FormRequest;

class PivotTableRequest extends FormRequest
{

    protected $columnHeaders;
    protected $rowHeadings;
    protected $valueFields;
    protected $joins;
    protected $static_filters;
    protected $parameters;

    protected function prepareForValidation()
    {
        $this->columnHeaders = $this->input('column_headings') ?
        destructFieldBlocks($this->input('column_headings'))
        : [];

        $this->rowHeadings = $this->input('row_headers') ?
        destructFieldBlocks($this->input('row_headers'))
        : [];

        $this->valueFields = $this->input('values_fields') ?
        destructFieldBlocks($this->input('values_fields'))
        : [];

        $this->joins = !is_null($this->input('joins')) ? destructJoins($this->input('joins')) : [];

        $pivotTableName  = $this->input("pivot_table_name");
        $isSaveOperation = $this->input("operation") == "save";

        if ($this->has('pivot_table_name') && $isSaveOperation) {
            $this->merge([
                'pivot_table_name' => preg_replace('/[-\s]+/', '_', $this->input('pivot_table_name'))
            ]);
        }

    }

    public function rules(): array
    {
        $prefix = config("srm_config.installer.table_prefix", "srm_");

        return [
            'connection'              => ['required'],
            'grand_totals_label'      => ['nullable'],
            'access_level'            => ['required'],
            'category'                => ['required'],
            'alignment'               => ['required'],
            'theme'                   => ['required'],
            'groups'                  => ['nullable', 'array'],
            'row_headers'             => [function ($attribute, $value, $fail) {

                if (empty($this->column_headings) && empty($this->row_headers)) {
                    $fail(trans('pivot-tables.validations.required.row_headers'));
                }

            }

            ],
            'values_fields'           => ['required'],
            'column_headings'         => ['nullable'],
            'joins'                   => [function ($attribute, $value, $fail) {

                $tablesUsed = array_unique(array_merge(
                    array_column($this->rowHeadings, 'table'),
                    array_column($this->columnHeaders, 'table'),
                    array_column($this->valueFields, 'table')
                ));

                if (count($tablesUsed) > 1) {

                    if (empty($this->joins)) {
                        return $fail(trans('pivot-tables.validations.required.join'));
                    }

                    foreach ($this->joins as $join) {
                        $left  = $join['left'] ?? null;
                        $right = $join['right'] ?? null;

                        if ($left && $right) {
                            $adjacencyList[$left][]  = $right;
                            $adjacencyList[$right][] = $left;
                        }

                    }

                    $visited              = [];
                    $startTable           = reset($tablesUsed);
                    $queue                = [$startTable];
                    $visited[$startTable] = true;

                    while (!empty($queue)) {
                        $current = array_shift($queue);

                        foreach ($adjacencyList[$current] ?? [] as $neighbor) {

                            if (!isset($visited[$neighbor])) {
                                $visited[$neighbor] = true;
                                $queue[]            = $neighbor;
                            }

                        }

                    }

                    foreach ($tablesUsed as $table) {

                        if (!isset($visited[$table])) {
                            return $fail(trans('pivot-tables.validations.required.missing_relationship'));
                        }

                    }

                }

            }

            ],

            'static_filters'          => ["nullable"],
            'combine_filters'         => ["nullable"],
            'parameters'              => ["nullable"],
            'parametersValues.*'      => ["nullable"],
            'parametersValues.*.from' => ["nullable", function ($attribute, $value, $fail) {
                $this->validateDateRange($attribute, $value, $fail);
            }

            ],
            'parametersValues.*.to'   => ["nullable", function ($attribute, $value, $fail) {
                $this->validateDateRange($attribute, $value, $fail);
            }

            ],

            'values_format'           => ["nullable"],
            'currency'                => ["nullable"],
            'custom_unit'             => ["nullable"],
            'operation'               => ["nullable"],
            'show_grand_totals'       => ["nullable"],
            "pivot_table_title"       => ['required'],
            "pivot_table_name"        => [function ($attribute, $value, $fail) {

                $pivotTableName  = $this->input("pivot_table_name");
                $isSaveOperation = $this->input("operation") == "save";

                if ($isSaveOperation) {

                    if (empty($pivotTableName) || $pivotTableName == "Pivot_Table1") {
                        $fail(trans('pivot-tables.validations.required.pivot_table_name'));
                    } elseif (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_ \-]*$/', $pivotTableName)) {
                        $fail(trans('pivot-tables.validations.regex.pivot_table_name'));
                    } elseif (Resource::where('name', $pivotTableName)->exists()) {
                        $fail(trans('pivot-tables.validations.unique.pivot_table_name'));
                    }

                }

            }

            ],

        ];
    }

    private function validateDateRange($attribute, $value, $fail)
    {
        $data = $this->input('parametersValues');

        foreach ($data as $values) {
            $from = $values['from'] ?? null;
            $to   = $values['to'] ?? null;

            if (is_null($from) xor is_null($to)) {
                return $fail(trans("pivot-tables.validations.required.missing_filter_date"));
            }

            if ($to && $from && Carbon::parse($to)->lessThan(Carbon::parse($from))) {
                return $fail(trans("pivot-tables.validations.custom.to_date_less_than_from_date"));
            }

        }

    }

    public function messages()
    {
        return [
            'values_fields.required' => trans('pivot-tables.validations.required.values_fields'),
        ];
    }

}
