<?php

namespace App\Http\Controllers\Charts;

use App\Models\Module;
use App\Models\Resource;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Services\ChartService;
use App\Services\CategoryService;
use App\Http\Controllers\Controller;
use App\Http\Middleware\OnlyFullVersion;
use App\Http\Requests\UpdateChartCategoryRequest;
use App\Http\Requests\Chart\DuplicateChartRequest;

class ChartController extends Controller
{
    public $chartService;
    public $categoryService;

    public function __construct(ChartService $chartService, CategoryService $categoryService)
    {
        $this->chartService    = $chartService;
        $this->categoryService = $categoryService;
        $this->middleware(OnlyFullVersion::class)->only([
            'changeCategory',
            'duplicate',
            'destroy'
        ]);

    }

    public function index(Request $request)
    {
        $charts     = $this->chartService->index($request);
        $chartTypes = $this->chartService->chartTypes();

        if (isset($request->chartName) || isset($request->chartTitle) || isset($request->chartType)) {
            $search = true;
        } else {
            $search = false;
        }

        $chartModule = Module::getChartModule();

        return view('dashboard.charts.index', compact('charts', 'chartTypes', 'search', 'chartModule'));
    }

    public function getScales($dataRange)
    {
        return response()->json([
            "success" => true,
            "data"    => $this->chartService->getDateRangeTimeScale($dataRange)
        ]);
    }

    public function getCategories(Resource $chart)
    {
        $categories = $this->categoryService->getCategories();

        return view('dashboard.charts.change-category', compact('chart', 'categories'));
    }

    public function changeCategory(Resource $chart, UpdateChartCategoryRequest $request)
    {
        $this->chartService->changeChartCategory($chart, $request->validated());
        return response()->json([
            'message'   => trans('charts.messages.updated'),
            'last_page' => $this->chartService->getChartLastPage(request())
        ]);
    }

    public function show($chart)
    {
        $chart = $this->chartService->show($chart);
        return response()->view('dashboard.charts.show', compact('chart'))
                ->withHeaders(getHeaders());
    }


    public function details(Resource $chart)
    {
        $chart = $this->chartService->details($chart);

        if (
            (strtolower($chart->access_control_type) == "public" && is_null(auth()->user())) ||
            (!is_null(request()->query('token')) && !is_null(session('loginType'))) ||
            (!is_null(request()->query("mode")) && request()->query("mode") == "embed")
        ) {
            return view('dashboard.charts.details-public', compact('chart'));
        }

        return view('dashboard.charts.details', compact('chart'));
    }

    public function autoexport(Resource $chart)
    {
        $chart = $this->chartService->autoExport($chart);

// if(is_null(auth()->user()) && strtolower($chart->access_control_type) == "public") {

//     return view('dashboard.charts.details-public', compact('chart'));
        // }

        return view('dashboard.charts.details-auto', compact('chart'));
    }

    public function generateDetails(Request $request, Resource $chart)
    {
        $this->chartService->generateDetailsPdf($request, $chart);

    }

    public function drill(Resource $chart)
    {

        if (empty(request()->query()) || !isset(request()->query()['xValue'])) {
            abort(Response::HTTP_NOT_FOUND);
        }

        $xValue = request()->query()['xValue'];

        [$data, $keys, $label] = $this->chartService->drill($chart, $xValue);

        $formatedXValue = $this->chartService->formatXValue($chart, $xValue);

        if (
            (strtolower($chart->access_control_type) == "public" && is_null(auth()->user())) ||
            (!is_null(request()->query('token')) && !is_null(session('loginType'))) ||
            (!is_null(request()->query("mode")) && request()->query("mode") == "embed")
        ) {
            return view('dashboard.charts.drill-public', compact('chart', 'data', 'keys', 'xValue', 'label', 'formatedXValue'));
        }

        return view('dashboard.charts.drill', compact('chart', 'data', 'keys', 'xValue', 'label', 'formatedXValue'));
    }

    public function forgetSession()
    {
        session()->forget('chart_session_wizard_configuration');

    }

    public function destroy(Resource $chart)
    {
        $this->chartService->destroy($chart);

        return response()->json([
            'message'   => trans('charts.messages.delete', [
                'chart' => $chart->name
            ]),
            'last_page' => $this->chartService->getChartLastPage(request())
        ]);
    }

    public function duplicate(DuplicateChartRequest $request, Resource $chart)
    {
        $this->chartService->duplicate($request->validated(), $chart);

        return response()->json([
            'message' => trans('charts.messages.duplicate')
        ]);
    }

}
