#!/usr/bin/env bash
set -uo pipefail
IFS=$'\n\t'

# ============================================================================
# SRM Update Script - Enhanced Version
# ============================================================================

# Configuration Variables
readonly SCRIPT_VERSION="1.0.0"
readonly SRM_OLD_DIR="srm_old"
readonly SRM_NEW_DIR="srm"
readonly DASHBOARD_DIR="dashboard"
readonly CONFIG_DIR="config"
readonly MODULES_DIR="modules"
readonly REPORTS_DIR="SRM9/SRM/Reports9"
readonly STORAGE_DIR="storage/app/public"
readonly BACKUP_DIR="existed_resources"
readonly LOG_FILE="update.log"
readonly PROGRESS_FILE=".progress"
readonly MIN_PHP_VERSION="7.4"
readonly REQUIRED_PHP_EXTENSIONS=("curl" "zip")

# Runtime flags
DRY_RUN=false
VERBOSE=false
ROLLBACK_MODE=false
RETRY_MIGRATIONS_MODE=false
VALID_ARGS_PARSED=false
ARGS_PROVIDED=false

# Error codes
readonly ERR_GENERAL=1
readonly ERR_DEPENDENCY=2
readonly ERR_PERMISSION=3
readonly ERR_MIGRATION=4
readonly ERR_VALIDATION=5
readonly ERR_ROLLBACK=6
readonly ERR_RETRY_MIGRATIONS=7

# Colors for output
readonly RED='\033[0;31m'
readonly GREEN='\033[0;32m'
readonly YELLOW='\033[1;33m'
readonly BLUE='\033[0;34m'
readonly NC='\033[0m' # No Color

# ============================================================================
# Utility Functions
# ============================================================================

log() {
    local level="$1"
    shift
    local message="$*"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')

    case "$level" in
    "INFO") echo -e "${GREEN}[$timestamp] [INFO]${NC} $message" ;;
    "WARN") echo -e "${YELLOW}[$timestamp] [WARN]${NC} $message" ;;
    "ERROR") echo -e "${RED}[$timestamp] [ERROR]${NC} $message" ;;
    "DEBUG") [[ "$VERBOSE" == true ]] && echo -e "${BLUE}[$timestamp] [DEBUG]${NC} $message" ;;
    esac
}

error_exit() {
    local message="$1"
    local code="${2:-$ERR_GENERAL}"
    log "ERROR" "$message"
    cleanup_on_failure
    exit "$code"
}

show_usage() {
    cat <<EOF
Usage: $0 [OPTIONS]

SRM Update Script v$SCRIPT_VERSION

OPTIONS:
    --dry-run       Perform a dry run without making actual changes
    --verbose       Enable verbose logging
    --rollback      Roll back to previous version
    --retry_migrations      Roll back to previous version
    --help          Show this help message

EXAMPLES:
    $0                  # Normal update
    $0 --dry-run        # Test update without changes
    $0 --verbose        # Update with detailed logging
    $0 --rollback       # Roll back to previous version
    $0 --retry_migrations      

EOF
}

parse_arguments() {
    # Check if any arguments were provided
    if [[ $# -gt 0 ]]; then
        ARGS_PROVIDED=true
    fi

    while [[ $# -gt 0 ]]; do
        case $1 in
        --dry-run)
            DRY_RUN=true
            log "INFO" "Dry run mode enabled"
            shift
            ;;
        --verbose)
            VERBOSE=true
            log "INFO" "Verbose mode enabled"
            shift
            ;;
        --rollback)
            ROLLBACK_MODE=true
            log "INFO" "Rollback mode enabled"
            shift
            ;;
        --retry_migrations)
            RETRY_MIGRATIONS_MODE=true
            log "INFO" "Retry migration mode enabled"
            shift
            ;;
        --help)
            show_usage
            exit 0
            ;;
        *)
            log "ERROR" "Unknown option: $1. Use --help for usage information."
            exit "$ERR_GENERAL" # Direct exit without cleanup
            ;;
        esac
    done
    VALID_ARGS_PARSED=true
}
sanitize_path() {
    local path="$1"
    # Remove any potentially dangerous characters and resolve path
    path=$(echo "$path" | sed 's/[;&|`$]//g')
    realpath -m "$path" 2>/dev/null || echo "$path"
}

validate_path() {
    local path="$1"
    local description="$2"

    if [[ ! -d "$path" ]]; then
        error_exit "$description not found at: $path" "$ERR_VALIDATION"
    fi

    if [[ ! -r "$path" ]]; then
        error_exit "$description is not readable: $path" "$ERR_PERMISSION"
    fi
}

# ============================================================================
# System Validation Functions
# ============================================================================

check_php_version() {
    log "INFO" "Checking PHP version..."

    if ! command -v php &>/dev/null; then
        error_exit "PHP is not installed or not in PATH" "$ERR_DEPENDENCY"
    fi

    local php_version
    php_version=$(php -r "echo PHP_VERSION;" 2>/dev/null) || error_exit "Failed to get PHP version" "$ERR_DEPENDENCY"

    log "DEBUG" "Found PHP version: $php_version"

    if ! php -r "exit(version_compare(PHP_VERSION, '$MIN_PHP_VERSION', '>=') ? 0 : 1);"; then
        error_exit "PHP version $MIN_PHP_VERSION or higher is required. Found: $php_version" "$ERR_DEPENDENCY"
    fi

    log "INFO" "PHP version check passed: $php_version"
}

check_php_extensions() {
    log "INFO" "Checking required PHP extensions..."

    # Detect active php.ini
    local php_ini_file
    php_ini_file=$(php --ini | grep "Loaded Configuration" | awk -F': ' '{print $2}')
    echo ""
    echo "Active php.ini file: $php_ini_file"
    echo ""

    while true; do
        local missing_extensions=()

        for extension in "${REQUIRED_PHP_EXTENSIONS[@]}"; do
            local check=$(php -m | grep -q "^$extension$")

            if ! $check; then
                missing_extensions+=("$extension")
            fi
        done

        if [[ ${#missing_extensions[@]} -eq 0 ]]; then
            log "INFO" "All required PHP extensions are available"
            break
        fi

        echo ""
        log "ERROR" "Missing PHP extensions: ${missing_extensions[*]}"
        echo ""
        echo "To fix this:"
        echo "1. Open your php.ini file:"
        echo "   $php_ini_file"
        echo "2. Enable the following missing extensions:"
        for ext in "${missing_extensions[@]}"; do
            echo "   extension=$ext"
        done
        echo "3. Save the file and restart your web server (Apache, Nginx, or PHP-FPM)."
        echo ""
        echo "After doing that, press Enter to recheck..."
        read -r
    done
}

validate_composer() {
    log "INFO" "Validating Composer..."

    if [[ ! -f "$SRM_NEW_DIR/composer" ]]; then
        error_exit "Composer binary not found in $SRM_NEW_DIR/" "$ERR_DEPENDENCY"
    fi

    log "INFO" "Composer validation passed"
}

validate_database_config() {
    log "INFO" "Validating database configuration..."

    local config_file="$SRM_OLD_DIR/$DASHBOARD_DIR/$CONFIG_DIR/srm_db_config.php"

    if [[ ! -f "$config_file" ]]; then
        error_exit "Database configuration file not found: $config_file" "$ERR_VALIDATION"
    fi

    # Test PHP syntax
    if ! php -l "$config_file" &>/dev/null; then
        error_exit "Database configuration file has syntax errors: $config_file" "$ERR_VALIDATION"
    fi

    log "INFO" "Database configuration validation passed"
}

pre_update_checks() {
    log "INFO" "Performing pre-update system checks..."

    check_php_version
    check_php_extensions
    validate_composer

    # Check disk space
    local available_space
    available_space=$(df . | awk 'NR==2 {print $4}')
    local required_space=1048576 # 1GB in KB

    if [[ "$available_space" -lt "$required_space" ]]; then
        error_exit "Insufficient disk space. Required: 1GB, Available: $((available_space / 1024))MB" "$ERR_VALIDATION"
    fi

    log "INFO" "Pre-update checks completed successfully"
}

# ============================================================================
# Progress and Backup Functions
# ============================================================================

show_progress() {
    local pid=$1
    local message="${2:-Processing}"
    local interval=1
    local spinner='|/-\'
    local i=0

    echo -n "$message "
    while kill -0 $pid 2>/dev/null; do
        printf "\b${spinner:$((i % 4)):1}"
        sleep $interval
        ((i++))
    done
    printf "\b✓\n"
}

update_progress_state() {
    local state="$1"

    if [[ "$DRY_RUN" == false ]]; then
        sed -i.bak "/^migration_/d" "$PROGRESS_FILE" 2>/dev/null || true
        echo "$state" >>"$PROGRESS_FILE"
        rm -f "$PROGRESS_FILE.bak"
    fi

    log "DEBUG" "Progress state updated: $state"
}

calculate_checksum() {
    local file="$1"
    if command -v sha256sum &>/dev/null; then
        sha256sum "$file" | cut -d' ' -f1
    elif command -v shasum &>/dev/null; then
        shasum -a 256 "$file" | cut -d' ' -f1
    else
        log "WARN" "No checksum utility available"
        echo "unavailable"
    fi
}

verify_file_integrity() {
    local source="$1"
    local destination="$2"

    if [[ ! -f "$source" || ! -f "$destination" ]]; then
        return 1
    fi

    local source_checksum destination_checksum
    source_checksum=$(calculate_checksum "$source")
    destination_checksum=$(calculate_checksum "$destination")

    if [[ "$source_checksum" != "unavailable" && "$source_checksum" == "$destination_checksum" ]]; then
        log "DEBUG" "File integrity verified: $destination"
        return 0
    else
        log "WARN" "File integrity check failed: $destination"
        return 1
    fi
}

# ============================================================================
# Backup and Restore Functions
# ============================================================================

create_backup() {
    local source="$1"
    local backup_name="$2"
    local destination="$BACKUP_DIR/$backup_name"

    log "INFO" "Creating backup: $backup_name"

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would backup $source to $destination"
        return 0
    fi

    if [[ -d "$destination" ]]; then
        log "WARN" "Backup already exists: $destination - skipping"
        return 0
    fi

    if ! mkdir -p "$BACKUP_DIR"; then
        error_exit "Failed to create backup directory: $BACKUP_DIR" "$ERR_PERMISSION"
    fi

    if ! cp -r "$source" "$destination"; then
        error_exit "Failed to create backup from $source to $destination" "$ERR_GENERAL"
    fi

    log "INFO" "Backup created successfully: $destination"
}

# ============================================================================
# Directory and Permission Functions
# ============================================================================

prompt_for_path() {
    local var_name="$1"
    local prompt_msg="$2"
    local path_value=""

    while [[ ! -d "$path_value" ]]; do
        read -rp "$prompt_msg " path_value
        path_value=$(sanitize_path "$path_value")
        [[ -d "$path_value" ]] || log "WARN" "Invalid directory. Try again."
    done

    eval "$var_name=\"$path_value\""
}

detect_web_server_user() {
    local users=("www-data" "apache" "nginx" "httpd")

    for user in "${users[@]}"; do
        if id -u "$user" >/dev/null 2>&1; then
            echo "$user"
            return
        fi
    done

    echo "$USER"
}

check_and_set_permissions() {
    local directories=("$@")
    local failed_directories=()
    local non_existent_directories=()
    local web_server_user
    web_server_user=$(detect_web_server_user)

    log "INFO" "Updating permissions and ownership (web server user: $web_server_user)"

    for dir in "${directories[@]}"; do
        if [[ ! -d "$dir" ]]; then
            non_existent_directories+=("$dir")
            continue
        fi

        if [[ "$DRY_RUN" == true ]]; then
            log "INFO" "[DRY RUN] Would set permissions for: $dir"
            continue
        fi

        local current_owner
        current_owner=$(stat -c "%U" "$dir" 2>/dev/null) || current_owner="unknown"

        if [[ "$current_owner" != "$web_server_user" ]]; then
            log "DEBUG" "Changing ownership of $dir to $web_server_user"
            (chown -R "$web_server_user:$web_server_user" "$dir" &>/dev/null) &
            local pid=$!
            show_progress "$pid" "Setting ownership for $dir"
            wait "$pid"
            if [[ $? -ne 0 ]]; then
                failed_directories+=("$dir (ownership)")
            fi
        fi

        log "DEBUG" "Setting directory permissions for $dir"
        if ! find "$dir" -type d -exec chmod 2755 {} \; 2>/dev/null; then
            failed_directories+=("$dir (chmod-dirs)")
        fi

        log "DEBUG" "Setting file permissions for $dir"
        if ! find "$dir" -type f -exec chmod 0644 {} \; 2>/dev/null; then
            failed_directories+=("$dir (chmod-files)")
        fi
    done

    if [[ ${#failed_directories[@]} -gt 0 ]]; then
        log "WARN" "Failed to set permissions for: ${failed_directories[*]}"
    fi

    if [[ ${#non_existent_directories[@]} -gt 0 ]]; then
        log "WARN" "Non-existent directories: ${non_existent_directories[*]}"
    fi

    log "INFO" "Permissions update completed"
}

create_symlink() {
    local source="$1"
    local destination="$2"

    log "INFO" "Creating symbolic link: $destination -> $source"

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would create symlink: $destination -> $source"
        return 0
    fi

    if ! ln -sfnv "$source" "$destination"; then
        error_exit "Failed to create symbolic link: $destination -> $source" "$ERR_GENERAL"
    fi

    log "INFO" "Symbolic link created successfully"
}

# ============================================================================
# SELinux Functions
# ============================================================================

is_selinux_unlabeled() {
    local file="$1"
    if command -v ls &>/dev/null && command -v grep &>/dev/null; then
        local context
        context=$(ls -Zd "$file" 2>/dev/null | awk '{print $1}')
        [[ "$context" == *unlabeled_t* ]]
    else
        return 1
    fi
}

set_selinux_permissions() {
    if ! command -v chcon &>/dev/null; then
        log "DEBUG" "SELinux tools not available, skipping SELinux configuration"
        return 0
    fi

    log "INFO" "Setting SELinux permissions..."

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would set SELinux permissions"
        return 0
    fi

    local files_processed=0
    while IFS= read -r -d '' file; do
        if ! is_selinux_unlabeled "$file"; then
            chcon -t httpd_sys_rw_content_t "$file" 2>/dev/null || true
            ((files_processed++))
        fi
    done < <(find "$SRM_NEW_DIR" -type f -print0 2>/dev/null)

    log "INFO" "SELinux permissions set for $files_processed files"
}

# ============================================================================
# Installation and Migration Functions
# ============================================================================

install_dependencies() {
    log "INFO" "Installing PHP dependencies with Composer..."

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would install Composer dependencies"
        return 0
    fi

    if [[ "$(id -u)" -eq 0 ]]; then
        export COMPOSER_ALLOW_SUPERUSER=1
        log "DEBUG" "Running as root, set COMPOSER_ALLOW_SUPERUSER=1"
    fi

    export COMPOSER_MEMORY_LIMIT=-1

    log "INFO" "Running composer install..."
    if ! php "$SRM_NEW_DIR/composer" install --working-dir="$SRM_NEW_DIR" --no-interaction --prefer-dist --optimize-autoloader --no-dev; then
        error_exit "Composer install failed" "$ERR_DEPENDENCY"
    fi

    log "INFO" "Running composer update..."
    if ! php "$SRM_NEW_DIR/composer" update --working-dir="$SRM_NEW_DIR" --no-interaction --prefer-dist --optimize-autoloader --no-dev; then
        error_exit "Composer update failed" "$ERR_DEPENDENCY"
    fi

    log "INFO" "Running composer dump-autoload..."
    if ! php "$SRM_NEW_DIR/composer" dump-autoload --working-dir="$SRM_NEW_DIR"; then
        error_exit "Composer dump-autoload failed" "$ERR_DEPENDENCY"
    fi

    log "INFO" "Composer dependencies installed successfully"
}

migrate_tables() {
    log "INFO" "Starting database migration..."
    update_progress_state "migration_started"

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would run database migrations"
        update_progress_state "migration_success"
        return 0
    fi

    MIGRATION_OUTPUT=$(php "$SRM_NEW_DIR/srm_install/upgrade.php" migrate 2>&1)
    if [[ $? -eq 0 ]]; then
        update_progress_state "migration_success"
        log "INFO" "Database migrations completed successfully"
    else
        update_progress_state "migration_error"
        log "ERROR" "Database migration failed with error:"
        echo "$MIGRATION_OUTPUT"
        echo
        log "ERROR" "A migration error has occurred:"
        echo "$MIGRATION_OUTPUT"
        echo

        while true; do
            echo
            echo -e "To retry the migration, type 1. To roll back to the previous version of Smart Report Maker, type 2:"
            read -p "Enter 1 or 2: " choice

            case "$choice" in
            1)
                log "INFO" "Retrying database migration..."
                MIGRATION_OUTPUT=$(php "$SRM_NEW_DIR/srm_install/upgrade.php" migrate 2>&1)
                if [[ $? -eq 0 ]]; then
                    update_progress_state "migration_success"
                    log "INFO" "Migration succeeded on retry"
                    break
                else
                    update_progress_state "migration_error"
                    log "ERROR" "Migration retry failed with error:"
                    echo "$MIGRATION_OUTPUT"
                    echo
                    log "ERROR" "A migration error has occurred again:"
                    echo "$MIGRATION_OUTPUT"
                fi
                ;;
            2)
                log "INFO" "Rolling back database changes..."
                php "$SRM_NEW_DIR/srm_install/upgrade.php" rollback >>"$LOG_FILE" 2>&1 || true
                error_exit "Migration aborted by user" "$ERR_MIGRATION"
                ;;
            *)
                echo -e "Invalid input. Please enter 1 or 2."
                ;;
            esac
        done
    fi
}

# ============================================================================
# Cleanup Functions
# ============================================================================

cleanup_on_failure() {
    # Check if script was called with arguments (assuming SCRIPT_ARGC is set at script start)
    if [[ "$VALID_ARGS_PARSED" != true || "$ARGS_PROVIDED" == true ]]; then
        log "INFO" "No cleanup needed.."

        return 0
    fi
    log "WARN" "Cleaning up after failure..."

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would perform cleanup"
        return 0
    fi

    # Remove partial symlinks
    local symlinks=(
        "$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_storage"
        "$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_install"
        "$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_modules"
    )

    for symlink in "${symlinks[@]}"; do
        if [[ -L "$symlink" ]]; then
            rm -f "$symlink" && log "DEBUG" "Removed symlink: $symlink"
        fi
    done

    # Remove partial config copies (if backup exists)
    if [[ -d "$BACKUP_DIR/config" ]]; then
        rm -f "$SRM_NEW_DIR/$DASHBOARD_DIR/$CONFIG_DIR/srm_db_config.php" 2>/dev/null || true
        rm -f "$SRM_NEW_DIR/$DASHBOARD_DIR/$CONFIG_DIR/srm_install_config.php" 2>/dev/null || true
        log "DEBUG" "Removed partial config files"
    fi

    log "INFO" "Cleanup completed"
}
# ============================================================================
# Main Execution Functions
# ============================================================================

check_directories() {
    log "INFO" "Checking required directories..."

    if [[ ! -d "$SRM_OLD_DIR" && -d "$SRM_NEW_DIR" ]]; then
        error_exit "The '$SRM_OLD_DIR' directory was not found. Please rename your existing '$SRM_NEW_DIR' directory to '$SRM_OLD_DIR' and rerun this script." "$ERR_VALIDATION"
    elif [[ ! -d "$SRM_NEW_DIR" && -d "$SRM_OLD_DIR" ]]; then
        error_exit "The update script cannot locate the '$SRM_NEW_DIR' directory for the new version. Ensure '$SRM_OLD_DIR' and '$SRM_NEW_DIR' are alongside the updater script." "$ERR_VALIDATION"
    elif [[ ! -d "$SRM_OLD_DIR" && ! -d "$SRM_NEW_DIR" ]]; then
        error_exit "Neither '$SRM_NEW_DIR' nor '$SRM_OLD_DIR' directories were found. Rename your old SRM folder to '$SRM_OLD_DIR', place the new '$SRM_NEW_DIR' folder and this script together, then rerun." "$ERR_VALIDATION"
    fi

    validate_path "$SRM_OLD_DIR" "Old SRM directory"
    validate_path "$SRM_NEW_DIR" "New SRM directory"

    log "INFO" "Directory structure validation passed"
}

check_versions() {
    log "INFO" "Checking version information..."

    local old_ver_file="$SRM_OLD_DIR/$DASHBOARD_DIR/$CONFIG_DIR/srm_config.php"
    local new_ver_file="$SRM_NEW_DIR/$DASHBOARD_DIR/$CONFIG_DIR/srm_config.php"

    validate_path "$(dirname "$old_ver_file")" "Old version config directory"
    validate_path "$(dirname "$new_ver_file")" "New version config directory"

    if [[ ! -f "$old_ver_file" ]]; then
        error_exit "Could not find old version file at $old_ver_file" "$ERR_VALIDATION"
    fi

    if [[ ! -f "$new_ver_file" ]]; then
        error_exit "Could not find new version file at $new_ver_file" "$ERR_VALIDATION"
    fi

    local old_ver new_ver
    old_ver=$(sed -nE 's/.*"version_to_install"\s*=>\s*"([^"]+)".*/\1/p' "$old_ver_file")
    new_ver=$(sed -nE 's/.*"version_to_install"\s*=>\s*"([^"]+)".*/\1/p' "$new_ver_file")

    if [[ -z "$old_ver" ]]; then
        error_exit "Could not extract version from $old_ver_file" "$ERR_VALIDATION"
    fi

    if [[ -z "$new_ver" ]]; then
        error_exit "Could not extract version from $new_ver_file" "$ERR_VALIDATION"
    fi

    log "INFO" "Version check: $old_ver -> $new_ver"

    if [[ "$old_ver" == "$new_ver" ]]; then
        log "WARN" "No version difference detected between 'srm_old' and 'srm'; both are from the same release."
        exit 0
    fi

    # Initialize progress tracking
    echo "$old_ver -> $new_ver" >"$PROGRESS_FILE"
}

backup_existing_data() {
    log "INFO" "Backing up existing configuration and reports..."

    if [[ ! -d "$BACKUP_DIR" ]]; then
        if [[ "$DRY_RUN" == false ]]; then
            mkdir -p "$BACKUP_DIR" || error_exit "Failed to create backup directory" "$ERR_PERMISSION"
        fi
    fi

    # Backup configuration
    local config_src="$SRM_OLD_DIR/$DASHBOARD_DIR/$CONFIG_DIR"
    if [[ ! -d "$config_src" ]]; then
        prompt_for_path config_path "The updater could not locate the configuration directory at $config_src. Please enter the correct path:"
        config_src="$config_path"
    fi

    create_backup "$config_src" "config"

    # Backup reports
    local reports_src="$SRM_OLD_DIR/$MODULES_DIR/$REPORTS_DIR"
    if [[ ! -d "$reports_src" ]]; then
        prompt_for_path reports_path "The updater could not locate the Reports directory at $reports_src. Please enter the correct path:"
        reports_src="$reports_path"
    fi

    create_backup "$reports_src" "Reports9"

    # Remove shared folder from backup
    if [[ "$DRY_RUN" == false ]]; then
        if [[ -d "$BACKUP_DIR/Reports9/shared" ]]; then
            rm -rf "$BACKUP_DIR/Reports9/shared"
            log "INFO" "Removed 'shared' folder from Reports9 backup"
        fi
    fi

    # Backup storage
    local storage_src="$SRM_OLD_DIR/$DASHBOARD_DIR/storage/app"
    if [[ ! -d "$storage_src" ]]; then
        prompt_for_path storage_path "The updater could not locate the Reports directory at $storage_src. Please enter the correct path:"
        storage_src="$storage_path"
    fi

    create_backup "$storage_src" "public"

    log "INFO" "Backup completed successfully"
}

copy_configurations() {
    log "INFO" "Copying configurations to new version..."

    local config_src="$SRM_OLD_DIR/$DASHBOARD_DIR/$CONFIG_DIR"
    local config_dst="$SRM_NEW_DIR/$DASHBOARD_DIR/$CONFIG_DIR"

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would copy configuration files"
        return 0
    fi

    # Copy database config
    local db_config_src="$config_src/srm_db_config.php"
    local db_config_dst="$config_dst/srm_db_config.php"

    if [[ -f "$db_config_src" ]]; then
        cp "$db_config_src" "$db_config_dst" || error_exit "Failed to copy database configuration" "$ERR_GENERAL"
        verify_file_integrity "$db_config_src" "$db_config_dst" || log "WARN" "Database config integrity check failed"
        log "INFO" "Database configuration copied successfully"
    else
        error_exit "Database configuration file not found: $db_config_src" "$ERR_VALIDATION"
    fi

    # Copy install config
    local install_config_src="$config_src/srm_install_config.php"
    local install_config_dst="$config_dst/srm_install_config.php"

    if [[ -f "$install_config_src" ]]; then
        cp "$install_config_src" "$install_config_dst" || error_exit "Failed to copy install configuration" "$ERR_GENERAL"
        verify_file_integrity "$install_config_src" "$install_config_dst" || log "WARN" "Install config integrity check failed"
        log "INFO" "Install configuration copied successfully"
    else
        error_exit "Install configuration file not found: $install_config_src" "$ERR_VALIDATION"
    fi

    # Copy reports - move directories inside Reports9 to new location
    local reports_src="$SRM_OLD_DIR/$MODULES_DIR/$REPORTS_DIR"
    local reports_dst="$SRM_NEW_DIR/$MODULES_DIR/SRM9/SRM/Reports9"

    if [[ -d "$reports_src" ]]; then
        if [[ ! -d "$reports_dst" ]]; then
            mkdir -p "$reports_dst" || error_exit "Failed to create reports destination directory" "$ERR_GENERAL"
        fi
        
        # Move only directories inside Reports9, excluding 'shared' folder
        for dir in "$reports_src"/*; do
            if [[ -d "$dir" && "$(basename "$dir")" != "shared" ]]; then
                cp -r "$dir" "$reports_dst/" || error_exit "Failed to copy report directory $(basename "$dir")" "$ERR_GENERAL"
                log "DEBUG" "Copied report directory: $(basename "$dir")"
            fi
        done
        log "INFO" "Reports directories copied successfully"
    else
        error_exit "Reports directory not found: $reports_src" "$ERR_VALIDATION"
    fi

    # Copy storage
    local storage_src="$SRM_OLD_DIR/$DASHBOARD_DIR/$STORAGE_DIR"
    local storage_dst="$SRM_NEW_DIR/$DASHBOARD_DIR/storage/app"

    if [[ -d "$storage_src" ]]; then
        cp -r "$storage_src" "$storage_dst" || error_exit "Failed to copy reports directory" "$ERR_GENERAL"
        log "INFO" "Reports directory copied successfully"
    else
        error_exit "Reports directory not found: $storage_src" "$ERR_VALIDATION"
    fi

}

setup_symlinks() {
    log "INFO" "Setting up symbolic links..."

    local base_path
    base_path=$(pwd)

    create_symlink "$base_path/$SRM_NEW_DIR/$DASHBOARD_DIR/$STORAGE_DIR" "$base_path/$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_storage"
    create_symlink "$base_path/$SRM_NEW_DIR/srm_install" "$base_path/$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_install"
    create_symlink "$base_path/$SRM_NEW_DIR/$MODULES_DIR" "$base_path/$SRM_NEW_DIR/$DASHBOARD_DIR/public/srm_modules"

    log "INFO" "Symbolic links setup completed"
}

setup_permissions() {
    log "INFO" "Setting up permissions and ownership..."

    local directories_to_check=(
        "$(pwd)/$SRM_NEW_DIR"
        "$(pwd)/$SRM_NEW_DIR/$DASHBOARD_DIR"
        "$(pwd)/$SRM_NEW_DIR/$DASHBOARD_DIR/public"
        "$(pwd)/$SRM_NEW_DIR/$DASHBOARD_DIR/$CONFIG_DIR"
        "$(pwd)/$SRM_NEW_DIR/$DASHBOARD_DIR/storage/logs"
        "$(pwd)/$SRM_NEW_DIR/$MODULES_DIR/SRM9/SRM/Reports9"
    )

    check_and_set_permissions "${directories_to_check[@]}"
    set_selinux_permissions

    log "INFO" "Permissions setup completed"
}

setup_logging() {
    # Backup existing log file
    if [[ -f "$LOG_FILE" ]]; then
        mv "$LOG_FILE" "$LOG_FILE.bak" || true
    fi

    # Setup logging to both file and stdout
    exec > >(tee -a "$LOG_FILE") 2>&1
}

# ============================================================================
# Rollback Functions
# ============================================================================

check_rollback_conditions() {
    log "INFO" "Checking rollback conditions..."

    if [[ ! -f "$PROGRESS_FILE" ]]; then
        error_exit "No progress file found. Cannot determine rollback state." "$ERR_ROLLBACK"
    fi

    local progress_state
    progress_state=$(grep -E "^(pre_migration|migration_started|migration_error|migration_success)$" "$PROGRESS_FILE" | tail -1)

    if [[ -z "$progress_state" ]]; then
        error_exit "Invalid progress state in $PROGRESS_FILE" "$ERR_ROLLBACK"
    fi

    case "$progress_state" in
    "pre_migration")
        log "INFO" "Rollback detected in pre-migration state"
        rollback_pre_migration
        ;;
    "migration_started" | "migration_error")
        log "INFO" "Rollback detected during/after failed migration"
        rollback_with_migration
        ;;
    "migration_success")
        log "INFO" "Rollback detected after successful migration"
        confirm_rollback_after_success
        ;;
    *)
        error_exit "Unknown progress state: $progress_state" "$ERR_ROLLBACK"
        ;;
    esac
}

rollback_pre_migration() {
    log "INFO" "Performing pre-migration rollback..."

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would restore directories for pre-migration rollback"
        return 0
    fi

    if [[ ! -d "$SRM_OLD_DIR" ]]; then
        error_exit "Cannot find $SRM_OLD_DIR for rollback" "$ERR_ROLLBACK"
    fi

    if [[ -d "$SRM_NEW_DIR" ]]; then
        if ! mv "$SRM_NEW_DIR" "${SRM_NEW_DIR}_new"; then
            error_exit "Failed to rename $SRM_NEW_DIR during rollback" "$ERR_ROLLBACK"
        fi
    fi

    if ! mv "$SRM_OLD_DIR" "$SRM_NEW_DIR"; then
        error_exit "Failed to restore $SRM_OLD_DIR during rollback" "$ERR_ROLLBACK"
    fi

    log "INFO" "Pre-migration rollback completed successfully"
    echo ""
    echo "============================================================================"
    echo "ROLLBACK COMPLETED SUCCESSFULLY"
    echo "============================================================================"
    echo "The 'srm_old' directory has been successfully restored. Please log in to your"
    echo "dashboard to verify that everything is functioning correctly. Once confirmed,"
    echo "you may safely delete the 'srm_new' and 'existed_resources' directories."
    echo "============================================================================"

    exit 0
}

rollback_with_migration() {
    log "INFO" "Performing rollback with database migration..."

    if [[ "$DRY_RUN" == true ]]; then
        log "INFO" "[DRY RUN] Would perform database rollback and directory restoration"
        return 0
    fi

    # Check if srm_install directory exists
    if [[ ! -d "$SRM_NEW_DIR/srm_install" ]]; then

        error_exit "The 'srm_install' directory is missing, possibly due to automatic deletion for security purposes. Please copy it from the downloaded full version and place it under the '$SRM_NEW_DIR/' directory, so the path becomes '$SRM_NEW_DIR/srm_install/'." "$ERR_ROLLBACK"
        # error_exit "'srm_install' directory is missing. Please ensure it exists inside '$SRM_NEW_DIR' before proceeding." "$ERR_ROLLBACK"
    fi

    # Perform database rollback
    log "INFO" "Attempting database rollback..."
    if ! php "$SRM_NEW_DIR/srm_install/upgrade.php" rollback >>"$LOG_FILE" 2>&1; then
        error_exit "Database migration rollback failed. Check $LOG_FILE for details." "$ERR_ROLLBACK"
    fi

    # Restore directories
    if [[ ! -d "$SRM_OLD_DIR" ]]; then
        error_exit "Cannot find $SRM_OLD_DIR for rollback" "$ERR_ROLLBACK"
    fi

    if [[ -d "$SRM_NEW_DIR" ]]; then
        if ! mv "$SRM_NEW_DIR" "${SRM_NEW_DIR}_new"; then
            error_exit "Failed to rename $SRM_NEW_DIR during rollback" "$ERR_ROLLBACK"
        fi
    fi

    if ! mv "$SRM_OLD_DIR" "$SRM_NEW_DIR"; then
        error_exit "Failed to restore $SRM_OLD_DIR during rollback" "$ERR_ROLLBACK"
    fi

    log "INFO" "Rollback with migration completed successfully"
    echo ""
    echo "============================================================================"
    echo "ROLLBACK COMPLETED SUCCESSFULLY"
    echo "============================================================================"
    echo "The 'srm_old' directory has been successfully restored. Please log in to your"
    echo "dashboard to verify that everything is functioning correctly. Once confirmed,"
    echo "you may safely delete the 'srm_new' and 'existed_resources' directories."
    echo "============================================================================"

    exit 0
}

confirm_rollback_after_success() {
    log "WARN" "The migration was previously completed successfully."
    echo ""
    echo "============================================================================"
    echo "WARNING: Migration Was Previously Successful"
    echo "============================================================================"
    echo "According to the upgrade history, the migration was completed successfully."
    echo "Are you sure you want to roll back?"
    echo ""
    read -p "Type 'yes' to confirm rollback, or anything else to cancel: " user_input

    if [[ "$user_input" != "yes" ]]; then
        log "INFO" "Rollback has been canceled by user"
        echo ""
        echo "Rollback has been canceled"
        exit 0
    fi

    log "INFO" "User confirmed rollback after successful migration"
    rollback_with_migration
}
retry_migrations() {
    log "INFO" "Starting Retry Migration process..."

    # Pre-flight checks
    pre_update_checks

    check_directories
    validate_database_config
    check_versions

    # Migration phase
    migrate_tables

    # Final message
    log "INFO" "Update process completed successfully!"
    echo ""
    echo "============================================================================"
    echo "UPDATE COMPLETED SUCCESSFULLY"
    echo "============================================================================"
    echo "Please log in to your dashboard using your existing credentials and"
    echo "verify everything is working correctly, including your reports."
    echo ""
    echo "Once confirmed, you may delete '$SRM_OLD_DIR' and '$BACKUP_DIR'."
    echo "Keep backups in case of emergency."
    echo "============================================================================"

}
normal_update() {
    log "INFO" "Starting SRM update process..."

    # Pre-flight checks
    pre_update_checks

    check_directories
    validate_database_config
    check_versions
    update_progress_state "pre_migration"

    # Backup phase
    backup_existing_data

    # Copy phase
    copy_configurations

    # Install phase
    install_dependencies
    setup_symlinks
    setup_permissions
    # Migration phase
    migrate_tables

    # Final message
    log "INFO" "Update process completed successfully!"
    echo ""
    echo "============================================================================"
    echo "UPDATE COMPLETED SUCCESSFULLY"
    echo "============================================================================"
    echo "Please log in to your dashboard using your existing credentials and"
    echo "verify everything is working correctly, including your reports."
    echo ""
    echo "Once confirmed, you may delete '$SRM_OLD_DIR' and '$BACKUP_DIR'."
    echo "Keep backups in case of emergency."
    echo "============================================================================"

}

main() {
    echo "============================================================================"
    echo "SRM Update Script v$SCRIPT_VERSION"
    echo "============================================================================"

    parse_arguments "$@"

    setup_logging

    if [[ "$ROLLBACK_MODE" == true ]]; then
        log "INFO" "Starting SRM rollback process..."
        check_rollback_conditions
    elif [[ "$RETRY_MIGRATIONS_MODE" == true ]]; then
        log "INFO" "Starting SRM update process..."
        retry_migrations
    else
        log "INFO" "Starting SRM update process..."
        normal_update
    fi

}

# ============================================================================
# Script Entry Point
# ============================================================================

# Trap cleanup on script exit
trap cleanup_on_failure "$@" ERR

# Run main function
main "$@"
