<?php

namespace Webuccino\Install\Core;

use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Capsule\Manager as DB;

class Migration
{
    public $migrations;
    public $db;
    public $prefix;
    public function __construct()
    {
        writeImportLog("Migration class constructor method start.");

        $this->migrations = [
            'Webuccino\Install\migrations\CreateGroupsTable',
            'Webuccino\Install\migrations\CreateUsersTable',
            'Webuccino\Install\migrations\CreateInstalledVersionsTable',
            'Webuccino\Install\migrations\CreateMailTable',
            'Webuccino\Install\migrations\CreateOrganizationTable',
            'Webuccino\Install\migrations\CreatePasswordResetTokensTable',
            'Webuccino\Install\migrations\CreateSettingsTable',
            'Webuccino\Install\migrations\CreateSessionsTable',
            'Webuccino\Install\migrations\CreateModulesTable',
            'Webuccino\Install\migrations\CreateCategoriesTable',
            'Webuccino\Install\migrations\CreateDataConnectionsTable',
            'Webuccino\Install\migrations\CreateAnalyticsResourcesTable',
            'Webuccino\Install\migrations\CreateResourcePermissionsTable',
            'Webuccino\Install\migrations\CreateModulesPermissionsTable',
            'Webuccino\Install\migrations\CreateResourceConfigurationTable',
            'Webuccino\Install\migrations\CreateScheduledTasksTable',
            'Webuccino\Install\migrations\CreateScheduledTaskGroupsTable',
            'Webuccino\Install\migrations\CreateScheduledTaskResourcesTable',
            'Webuccino\Install\migrations\CreateScheduledTaskHistoryTable',
            'Webuccino\Install\migrations\CreateScheduledTasksSendingLogTable',
            'Webuccino\Install\migrations\CreateJobsTable',
            'Webuccino\Install\migrations\CreateFailedJobsTable',
            'Webuccino\Install\migrations\CreateHomeDashboardTable',
            'Webuccino\Install\migrations\CreateDashboardsTable',
            'Webuccino\Install\migrations\CreateWidgetsTable',
            'Webuccino\Install\migrations\EnablePivotTable',
            'Webuccino\Install\migrations\AddOrderColumnToCategoriesTable',
            'Webuccino\Install\migrations\AddRelationAndParametersToResourceConfiguration',
            'Webuccino\Install\migrations\CreateBarcodesDefinitionsTable',
            'Webuccino\Install\migrations\CreateBarcodesUsageTable',
            'Webuccino\Install\migrations\AddAccessTypeColumnToDashboardsTable',
            'Webuccino\Install\migrations\CreateAccessKeysTable',
            'Webuccino\Install\migrations\CreateEmbedTokensTable',
            'Webuccino\Install\migrations\CreateEmbedLogTable',
            'Webuccino\Install\migrations\CreateEmbeddingPreferenceTable'
        ];

        $testing_data = config_native("srm_config.dashboard.demo.testing_data", true) ?? true;

        if ((!is_docker() && $testing_data) ||
            (!is_null(getenv('ADD_TESTING_DATA')) && getenv('ADD_TESTING_DATA') == 'true')
        ) {
            $this->migrations[] = 'Webuccino\Install\migrations\CreateNorthWindTables';
        }

        App::resolve('Webuccino\Install\Core\Database')->initiateConnectionFromconfig_native();

        $this->db     = new DB();
        $this->prefix = config_native('srm_config.installer.table_prefix');

        writeImportLog("Migration class constructor method end.");
    }

    public function migrateMigrationsTable()
    {

        if (!DB::schema()->hasTable($this->prefix . 'migrations')) {
            (new \Webuccino\Install\migrations\CreateMigrationsTable())->run();
            writeImportLog("CreateMigrationsTable class migrated successfully.");
        }

    }

    public function runMigrations()
    {
        try {
            writeImportLog("run Migration method started.");

            $this->migrateMigrationsTable();

            $lastBatch = $this->getLastBatch() + 1;

            $numberOfTables = 0;

            foreach ($this->migrations as $migration) {

                if ($this->checkMigrationExists($migration) && $this->checkTableNamesExists($migration) ||
                ($this->checkNorthwindMigrationExists() && $migration == 'Webuccino\Install\migrations\CreateNorthWindTables')
                ) {
                    writeImportLog(basename(str_replace("\\", "/", $migration)) . " class already exists.");

                    continue;
                }

                (new $migration())->run();

                writeImportLog(basename(str_replace("\\", "/", $migration)) . " class migrated successfully.");

                $this->db->table($this->prefix . 'migrations')->insert([
                    'migration' => $this->getMigrationName($migration),
                    'batch'     => $lastBatch
                ]);

                writeImportLog(basename(str_replace("\\", "/", $migration)) . " added to migration table.");

            }

            writeImportLog('Migration Finished Successfully.');
        } catch (\Exception $e) {
            throw new \Exception('Exception message: ' . $e->getMessage());

            writeImportLog('Exception at Migration Class message: ' . $e->getMessage());
            writeImportLog('Exception Trace: ' . $e->getTraceAsString());
        }

    }

    public function rollbackMigrations()
    {
        try {

            writeImportLog("rollbackLastBatch started.");

            $lastBatch = $this->getLastBatch();


            if ($lastBatch <= 1) {
                writeImportLog("No batches found to roll back.");
                return;
            }

            // fetch all migrations in that batch, newest first
            $rows = $this->db
                ->table($this->prefix . 'migrations')
                ->where('batch', $lastBatch)
                ->orderBy('id', 'desc')
                ->get();

            if ($rows->isEmpty()) {
                writeImportLog("Batch {$lastBatch} has no migrations to roll back.");
                return;
            }

            foreach ($rows as $row) {
                $migrationName = $row->migration;
                $class         = $this->findMigrationClass($migrationName);

                if (!$class) {
                    writeImportLog("Could not find migration class for '{$migrationName}'. Skipping.");
                    continue;
                }

                // call your migration’s rollback (or down) method
                (new $class())->rollback();

                writeImportLog("{$migrationName} rolled back successfully.");

                // remove from migrations table
                $this->db
                    ->table($this->prefix . 'migrations')
                    ->where('migration', $migrationName)
                    ->where('batch', $lastBatch)
                    ->delete();

                writeImportLog("{$migrationName} removed from migrations table.");
            }

            writeImportLog("rollbackLastBatch of batch {$lastBatch} completed.");

        } catch (\Exception $e) {
            writeImportLog("Exception in rollbackLastBatch: " . $e->getMessage());
            writeImportLog("Trace: " . $e->getTraceAsString());
        }

    }

    protected function findMigrationClass(string $migrationName): ?string
    {

        foreach ($this->migrations as $fqcn) {

            if ($this->getMigrationName($fqcn) === $migrationName) {
                return $fqcn;
            }

        }

        return null;
    }

    public function getMigrationName($migration)
    {
        $migration_path = explode("\\", $migration);
        return $migration_path[sizeof($migration_path) - 1];
    }

    public function getTableName($migration)
    {
        return (new $migration())->getTableName();
    }

    public function checkMigrationExists($migration)
    {
        $table = $this->getMigrationName($migration);

        return $this->db->table($this->prefix . 'migrations')
            ->where('migration', $table)
            ->exists();
    }

    public function checkNorthwindMigrationExists()
    {
        $table = $this->getMigrationName('Webuccino\Install\migrations\CreateNorthWindTables');

        return $this->db->table($this->prefix . 'migrations')
            ->where('migration', $table)
            ->exists();
    }

    public function getLastBatch()
    {

        if ($this->db->table($this->prefix . 'migrations')->exists()) {
            return $this->db->table($this->prefix . 'migrations')
                ->latest('created_at')
                ->first()->batch;
        } else {
            return 0;
        }

    }

    public function checkTableNamesExists($table)
    {
        return DB::schema()
            ->hasTable($this->getTableName($table));
    }

}
