<?php

namespace App\Services;

use App\Models\Module;
use App\Models\Resource;
use Illuminate\Http\Response;
use App\Models\ResourcePermission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use App\Models\ResourceConfiguration;
use Illuminate\Support\Facades\Storage;

class ReportService
{
    private $moduleService;
    private $groupService;
    private $categoryService;

    public function __construct(
        ModuleService $moduleService,
        GroupService $groupService,
        CategoryService $categoryService
    ) {
        $this->moduleService   = $moduleService;
        $this->groupService    = $groupService;
        $this->categoryService = $categoryService;
    }

    public function getReports()
    {

        if (auth()->user()->isAdminOrOwner()) {
            $reports = $this->replaceReportCreator($this->getAllReports());
            return $reports->sortBy('category.order');
        } else {
            $reports = $this->replaceReportCreator($this->getAuthedUserReports());
            return $reports->sortBy('category.order');
        }

    }

    public function getAllReports()
    {
        return Resource::whereHas('module', function ($q) {

            return $q->where('name', 'Reports');
        })->orWhereHas('module.parent', function ($q) {
            return $q->where('name', 'Reports');
        })
            ->with('category')
            ->orderByDesc('created_at')

            ->get();
    }

    public function getAuthedUserReports()
    {
        $prefix = config("srm_config.installer.table_prefix", "srm_");

        return Resource::where(function ($q) use ($prefix) {
            $q->where(function ($q) {
                $q->whereHas('module', function ($q) {
                    return $q->where('name', 'Reports');
                })->orWhereHas('module.parent', function ($q) {
                    return $q->where('name', 'Reports');
                });
            })->where(function ($q) use ($prefix) {
                $q->whereHas('resource_permissions', function ($q) use ($prefix) {
                    return $q->where($prefix . 'resource_permissions.group_id', auth()->user()->group_id);
                })->orWhere(function ($q) {
                    $q->where('access_control_type', 'Public');
                });
            });
        })
            ->with('category')
            ->orderByDesc('created_at')
            ->get();
    }

    public function changeReportCategory($report, $data)
    {
        $this->checkReportCreatorPermission($report);

        $report->category_id = $data['report_category'];
        $report->save();
    }

    public function show($report)
    {

        switch (strtolower($report->module->name)) {
            case strtolower('Blank Reports'):
                $this->checkReportAccessPermission($report);
                $this->checkReportExists($report);
                return $report;
                break;
            default:
                abort(Response::HTTP_FORBIDDEN, trans('report.no_permission_to_access_report'));
                break;
        }

    }

    public function create()
    {

        if (
            !$this->moduleService->checkCurrentUserGroupCanCreateModule('Reports')
            && !auth()->user()->isAdminOrOwner()
        ) {
            abort(403, trans('report.permission_denied'));
        }

        // $this->checkThereIsGroupsAndCategories();

        $this->createReportSession();

        if (!$this->moduleService->checkModuleOnlyActiveInChildren('Reports', 'Blank Reports')) {
            return "";
        } else {
            return "SRM9/SRM/wizard/index.php";
        }

    }

    public function edit(Resource $report)
    {

        $this->checkReportCreatorPermission($report);

        // $this->checkThereIsGroupsAndCategories();

        $this->editReportSession($report);

        switch (strtolower($report->module->name)) {
            case 'blank reports':
                return "SRM9/SRM/wizard/index.php";
                break;
            default:
                return "";
                break;
        }

    }

    public function destroy($report)
    {
        $this->checkReportCreatorPermission($report);
        $this->checkScheduledReportExists($report);

        try {
            $this->deleteReportFolder($report);
        } catch (\Exception $e) {
            abort(Response::HTTP_BAD_REQUEST, trans('report.delete_error'));
        }

        $report->forceDelete();
    }

    private function getUrl($reportName)
    {
        $reportName = str_replace(" ", "", $reportName);

        return "SRM9/SRM/Reports9/$reportName/{$reportName}.php";
    }

    public function duplicate($request, $originalReport)
    {
        return DB::transaction(function () use ($request, $originalReport) {

            $newName = $request['name'];

            $newResource = Resource::create([
                'name'                => $newName,
                'url'                 => $this->getUrl($newName),
                'access_control_type' => $originalReport->access_control_type,
                'module_id'           => $originalReport->module_id,
                'category_id'         => $originalReport->category_id,
                'resource_creator'    => auth()->id(),
                'data_connection_id'  => $originalReport->data_connection_id
            ]);

            $originalPermissions = ResourcePermission::where('resource_id', $originalReport->id)->get();

            foreach ($originalPermissions as $permission) {
                ResourcePermission::create([
                    'group_id'    => $permission->group_id,
                    'resource_id' => $newResource->id,
                    'can_view'    => $permission->can_view
                ]);
            }

            $originalConfiguration = ResourceConfiguration::where('Resource_id', $originalReport->id)->first();

            if ($originalConfiguration) {
                $configData = $originalConfiguration->Json_configurations;

                if (isset($configData['file_name'])) {
                    $configData['file_name'] = $newName;
                }

                ResourceConfiguration::create([
                    'Resource_id'         => $newResource->id,
                    'Json_configurations' => $configData,
                    'relation' => $originalConfiguration->relation,
                    'parameters' => $originalConfiguration->parameters,
                ]);
            }

            $this->copyReportDirectory($originalReport->name, $newName, $newResource->id);

            return $newResource;
        });

    }

    private function copyReportDirectory($originalReportName, $newReportName, $newReportId = null)
    {
        $disk        = Storage::disk('reports');
        $reportsPath = "SRM9/SRM/Reports9";

        $originalReportName = str_replace(" ", "", $originalReportName);
        $newReportName = str_replace(" ", "", $newReportName);

        $originalReportPath = $reportsPath . "/" . $originalReportName;
        $newReportPath      = $reportsPath . "/" . $newReportName;

        if ($disk->exists($originalReportPath)) {

            if (!$disk->exists($newReportPath)) {
                $disk->makeDirectory($newReportPath);
                chmod($disk->path($newReportPath), 0755);
            }

            $files = $disk->allFiles($originalReportPath);

            foreach ($files as $file) {
                try {
                    $content = $disk->get($file);

                    $newFilePath = str_replace($originalReportPath, $newReportPath, $file);

                    $parentDir = dirname($newFilePath);

                    if ($parentDir !== '.' && !$disk->exists($parentDir)) {
                        $disk->makeDirectory($parentDir);
                        chmod($disk->path($parentDir), 0755);
                    }

                    $disk->put($newFilePath, $content);
                    chmod($disk->path($newFilePath), 0644);

                } catch (Exception $e) {
                    \Log::error("Failed to copy file {$file}: " . $e->getMessage());
                }

            }

            $originalIndexFile = $newReportPath . "/" . $originalReportName . ".php";
            $newIndexFile      = $newReportPath . "/" . $newReportName . ".php";

            if ($disk->exists($originalIndexFile)) {
                $content = $disk->get($originalIndexFile);
                $disk->put($newIndexFile, $content);
                chmod($disk->path($newIndexFile), 0644);
                $disk->delete($originalIndexFile);
            }

            $this->editConfigFile($disk, $newReportPath, $newReportName, $newReportId);

        } else {
            abort(Response::HTTP_NOT_FOUND, trans('report.cant_duplicate'));
        }

    }

    private function editConfigFile($disk, $reportPath, $fileName, $reportId = null)
    {
        $configPath = $reportPath . "/config.php";

        if ($disk->exists($configPath)) {
            try {
                $configContent = $disk->get($configPath);

                $currentDate = date('d-M-Y H:i:s');

                if ($reportId === null) {
                    $reportId = mt_rand(100, 999);
                }

                $patterns = [
                    '/(\$file_name\s*=\s*")[^"]*(";\s*)/'    => '${1}' . $fileName . '${2}',
                    '/(\$report_id\s*=\s*)\d+(;\s*)/'        => '${1}' . $reportId . '${2}',
                    '/(\$date_created\s*=\s*")[^"]*(";\s*)/' => '${1}' . $currentDate . '${2}'
                ];

                foreach ($patterns as $pattern => $replacement) {
                    $configContent = preg_replace($pattern, $replacement, $configContent);
                }

                $configContent = preg_replace(
                    '/^(\/\/[^,]*,)\d{2}-\w{3}-\d{4} \d{2}:\d{2}:\d{2}/m',
                    '${1}' . $currentDate,
                    $configContent
                );

                $disk->put($configPath, $configContent);
                chmod($disk->path($configPath), 0644);

            } catch (Exception $e) {
                \Log::error("Failed to edit config file {$configPath}: " . $e->getMessage());
            }

        }

    }

    private function checkScheduledReportExists($report)
    {

        if ($report->scheduledTaskResource()->exists()) {
            abort(400, trans('report.delete_scheduled_task_before_error', [
                'report'        => $report->name,
                'scheduledTask' => $report->scheduledTaskResource()->first()->scheduledTask()->first()->title
            ]));
        }

    }

    private function checkReportCreatorPermission(Resource $report)
    {

        if (
            !(auth()->user()->isAdminOrOwner() ||
                $report->resource_creator === auth()->user()->user_ID)
        ) {
            abort(403, trans('report.unauthorized'));
        }

    }

    private function checkReportAccessPermission(Resource $report)
    {
        if(!is_null(request()->query('token')) && !is_null(session('loginType')))
            return true;

        if (
            !((!is_null(auth()->user()) && auth()->user()->isAdminOrOwner()) || strtolower($report->access_control_type) == "public" ||
                in_array(auth()->user()->group_id, $report->resource_permissions->pluck("group_ID")->toArray()))
        ) {
            abort(403, trans('report.access_unautorized'));
        }

    }

    private function checkReportExists(Resource $report)
    {

        if (!File::exists(public_path() . "/srm_modules/" . $report->url)) {
            abort(Response::HTTP_NOT_FOUND, trans('report.not_found'));
        }

    }

    public function checkThereIsGroupsAndCategories()
    {

        if ($this->categoryService->getCategoriesCount() == 0 || $this->groupService->getGroupsCount() == 0) {
            abort(400, trans('report.create_group_and_category'));
        }

    }

    private function createReportSession()
    {
        $_SESSION["srm_wizard_config"]["edit_mode"] = false;
    }

    private function editReportSession($report)
    {
        $_SESSION["srm_wizard_config"]["edit_mode"] = 1;
        $_SESSION["srm_wizard_config"]["report_id"] = $report->id;
    }

    private function deleteReportFolder($report)
    {
        Storage::disk('reports')->deleteDirectory(
            dirname($report->url)
        );
    }

    private function replaceReportCreator($reports)
    {

        if (checkDemo()) {

            $userIdentifiers = [];
            $userCounter     = 1;

            foreach ($reports as $report) {

                if (isset($report->creator)) {
                    $creatorName = $report->creator->user_name;

                    if (!isset($userIdentifiers[$creatorName])) {
                        $userIdentifiers[$creatorName] = '<user ' . $userCounter . '>';
                        $userCounter++;
                    }

                    $report->report_creator = $userIdentifiers[$creatorName];
                } else {
                    $report->report_creator = '';
                }

            }

        }

        return $reports;
    }

}
