<?php

namespace App\Services;

use App\Models\Resource;
use Illuminate\Http\Response;
use App\Services\ModuleService;
use App\Builders\PivotTableBuilder;
use App\Repositories\PivotTableRepository;

class PivotTableService
{
    private $pivotTableRepository;
    private $moduleService;
    public function __construct(
        PivotTableRepository $pivotTableRepository,
        ModuleService $moduleService
    ) {
        $this->pivotTableRepository = $pivotTableRepository;
        $this->moduleService        = $moduleService;
    }

    public function create()
    {

        if (
            !$this->moduleService->checkCurrentUserGroupCanCreateModule('Pivot tables')
            && !auth()->user()->isAdminOrOwner()
        ) {
            abort(403, trans('pivot-tables.messages.unauthorized'));
        }

    }

    public function getPivotTables()
    {

        if (auth()->user()->isAdminOrOwner()) {
            $pivotTables = $this->replacePivotTableCreator($this->getAllPivotTables());
            return $pivotTables->sortBy('category.order');
        } else {
            $pivotTables = $this->replacePivotTableCreator($this->getAuthedUserPivotTables());
            return $pivotTables->sortBy('category.order');
        }

    }

    public function getAllPivotTables()
    {
        return Resource::whereHas('module', function ($q) {

            return $q->where('name', 'Pivot tables');
        })->orWhereHas('module.parent', function ($q) {
            return $q->where('name', 'Pivot tables');
        })
            ->with('category')
            ->orderByDesc('created_at')

            ->get();
    }

    public function getAuthedUserPivotTables()
    {
        $prefix = config("srm_config.installer.table_prefix", "srm_");

        return Resource::where(function ($q) use ($prefix) {
            $q->where(function ($q) {
                $q->whereHas('module', function ($q) {
                    return $q->where('name', 'Pivot tables');
                })->orWhereHas('module.parent', function ($q) {
                    return $q->where('name', 'Pivot tables');
                });
            })->where(function ($q) use ($prefix) {
                $q->whereHas('resource_permissions', function ($q) use ($prefix) {
                    return $q->where($prefix . 'resource_permissions.group_id', auth()->user()->group_id);
                })->orWhere(function ($q) {
                    $q->where('access_control_type', 'Public');
                });
            });
        })
            ->with('category')
            ->orderByDesc('created_at')
            ->get();
    }

    private function replacePivotTableCreator($pivotTables)
    {

        if (checkDemo()) {

            $userIdentifiers = [];
            $userCounter     = 1;

            foreach ($pivotTables as $pivotTable) {

                if (isset($pivotTable->creator)) {
                    $creatorName = $pivotTable->creator->user_name;

                    if (!isset($userIdentifiers[$creatorName])) {
                        $userIdentifiers[$creatorName] = '<user ' . $userCounter . '>';
                        $userCounter++;
                    }

                    $pivotTable->report_creator = $userIdentifiers[$creatorName];
                } else {
                    $pivotTable->report_creator = '';
                }

            }

        }

        return $pivotTables;
    }

    public function show($pivotTable)
    {

        switch (strtolower($pivotTable->module->name)) {
            case strtolower('Pivot tables'):
                $this->checkPivotTableAccessPermission($pivotTable);
                return $pivotTable;
                break;
            default:
                abort(Response::HTTP_FORBIDDEN, trans('pivot-tables.messages.access_unauthorized'));
                break;
        }

    }

    public function edit($pivotTable)
    {

        switch (strtolower($pivotTable->module->name)) {
            case strtolower('Pivot tables'):
                $this->checkPivotTableCreatorPermission($pivotTable);
                return $pivotTable;
                break;
            default:
                abort(Response::HTTP_FORBIDDEN, trans('pivot-tables.messages.access_unauthorized'));
                break;
        }

    }

    public function exportCreate($pivotTable)
    {
        return (new PivotTableBuilder($pivotTable))->build();
    }

    public function exportShow($pivotTable, $request)
    {
        $pivotTableConfiguration = $pivotTable->configurations->Json_configurations;
        $requestData             = $request->all();

        $pivotConfiguration = array_merge($pivotTableConfiguration, $requestData);

        $pivotConfigurationObject = json_decode(json_encode($pivotTableConfiguration));

        return [
            (new PivotTableBuilder($pivotConfiguration))->build(),
            $pivotConfigurationObject
        ];

    }

    public function buildCsv($pivotTable)
    {
        $result = flattenArray($pivotTable);
        $records = "";
        if (!empty($result)) {

            $labels       = array_keys($result[0]);
            $fields_count = count(array_keys($result[0]));

            $header = "";

            foreach ($labels as $k => $v) {

                $field = $v;
                $header .= str_replace(',', ';', $field) . ',';
            }

            $header = substr($header, 0, strlen($header) - 1);

            $header = $header . " " . PHP_EOL;

            $k       = 0;
            $records = $header;

            foreach ($result as $row) {

                foreach ($row as $key => $val) {
                    $field_data = $val;
                    $field_data = str_replace("\r\n", ' ', $field_data);
                    $field_data = str_replace(',', ';', $field_data);
                    $field_data = str_replace("\n", ' ', $field_data);

                    $field_data .= ',';

                    $records .= $field_data;
                }

                $records .= PHP_EOL;
            }

        } else {
            $records .= "" . PHP_EOL;
        }

        header('Pragma: public');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Content-Description: File Transfer');
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename=export.csv;');
        header('Content-Transfer-Encoding: binary');

        $fp = fopen('php://output', 'w');

        fputs($fp, $bom = (chr(0xEF) . chr(0xBB) . chr(0xBF)));

        if ($fp) {
            fwrite($fp, $records);
        }

        fclose($fp);

    }


    public function build($pivotTable, $request)
    {

        switch (strtolower($pivotTable->module->name)) {
            case strtolower('Pivot tables'):
                $data = array_merge($pivotTable->configurations->Json_configurations, $request->all());
                $this->checkPivotTableAccessPermission($pivotTable);
                return (new PivotTableBuilder($data))->build();
                break;
            default:
                abort(Response::HTTP_FORBIDDEN, trans('pivot-tables.messages.access_unauthorized'));
                break;
        }

    }

    private function checkPivotTableAccessPermission(Resource $pivotTable)
    {
        
        if(!is_null(request()->query('token')) && !is_null(session('loginType')))
            return true;

        $user = auth()->user();

        $isAuthorized = (
            ($user && $user->isAdminOrOwner()) ||
            strtolower($pivotTable->access_control_type) == "public" ||
            ($user && in_array($user->group_id, $pivotTable->resource_permissions->pluck("group_ID")->toArray()))
        );

        if (!$isAuthorized) {
            abort(403, trans('pivot-tables.messages.access_unauthorized'));
        }

    }

    public function changePivotTableCategory($pivotTable, $data)
    {
        $this->checkPivotTableCreatorPermission($pivotTable);

        $pivotTable->category_id = $data['pivot_table_category'];
        $pivotTable->save();
    }

    public function destroy($pivotTable)
    {
        $this->checkPivotTableCreatorPermission($pivotTable);

        $pivotTable->delete();
    }

    private function checkPivotTableCreatorPermission(Resource $pivotTable)
    {

        if (
            !(auth()->user()->isAdminOrOwner() ||
                $pivotTable->resource_creator === auth()->user()->user_ID)
        ) {
            abort(403, trans('pivot-tables.messages.unauthorized'));
        }

    }

    public function apply($request)
    {
        $this->sendPivotTableApplyNotification($request);
        return (new PivotTableBuilder($request))->build();
    }


    private function sendPivotTableApplyNotification($request)
    {
        if (!checkDemo() || strpos(request()->host(), 'demo.mysqlreports.com') === false) {
            return;
        }

        $url = config('srm_config.dashboard.demo.notification_url') . "/pivot-table";

        $request["ip"] = request()->ip();
        $request["referral"] = request()->headers->get('referer');
        $request["time"] = now()->toDateTimeString();

        try {
            sendApiNotification($url, $request);
        } catch (\Exception $e) {

        }
    }

    public function store($request)
    {
        return $this->pivotTableRepository->store($request);
    }

    public function duplicate($request, $pivotTable)
    {
        return $this->pivotTableRepository->duplicate($request, $pivotTable);
    }

    public function update($request, $pivotTable)
    {
        return $this->pivotTableRepository->update($request, $pivotTable);
    }

}
