<?php

namespace App\Services;

use App\Models\Resource;
use App\Models\HomeDashboard;
use App\Builders\ChartBuilder;
use App\Builders\MetricBuilder;
use App\Repositories\HomeRepository;

class HomeService
{
    private $moduleService;
    private $homeRepository;
    private $resourceService;
    public function __construct(ModuleService $moduleService, HomeRepository $homeRepository, ResourceService $resourceService)
    {
        $this->moduleService   = $moduleService;
        $this->homeRepository  = $homeRepository;
        $this->resourceService = $resourceService;

    }

    public function index()
    {
        $numberOfHomeWidgets = config('srm_config.dashboard.max_home_widgets', 20);

        $homeWidgets = HomeDashboard::where('user_id', auth()->user()->user_ID)
            ->with('resource')
            ->orderBy('created_at', 'asc')
            ->paginate($numberOfHomeWidgets);

        foreach ($homeWidgets as $widget) {

            if (isset($widget->resource_id)) {
                $this->buildResource($widget->resource);
            }

        }

        return $homeWidgets;

    }

    public function buildResource($resource)
    {
        if (isset($resource)) {

            switch ($resource->getResourceType()) {
                case 'KPIs / Metrics':
                    (new MetricBuilder($resource))->build();
                    break;
            }

        }
    }

    public function getResourceTypes()
    {
        return [
            ["value" => $this->moduleService->getModuleByName("Charts")->id, "id" => "charts", "label" => "Chart"],
            ["value" => $this->moduleService->getModuleByName("KPIs / Metrics")->id, "id" => "metrics", "label" => "Metric / KPI"],
            ["value" => $this->moduleService->getModuleByName("Pivot tables")->id, "id" => "pivot-tables", "label" => "Pivot Table"],
            ["value" => $this->moduleService->getModuleByName("Blank Reports")->id, "id" => "report-list", "label" => "Report List"]
        ];
    }

    public function getResourceType($widget)
    {

        if (!isset($widget->resource->module)) {
            return "Report List";
        }

        $resourceModule = $widget->resource->module->id;

        foreach ($this->getResourceTypes() as $type) {

            if ($type["value"] == $resourceModule) {
                return $type["label"];
            }

        }

    }

    public function storeWidget($request)
    {
        $this->homeRepository->store($request);
    }

    public function updateWidget($request, $widget)
    {
        $this->homeRepository->update($request, $widget);
    }

    public function destroy($widget)
    {
        $widget->delete();
    }

    public function getAllResources($moduleId)
    {

        if (is_null($moduleId)) {
            return $this->resourceService->getAllResourceByModuleId($this->moduleService->getModuleByName("Blank Reports")->id)->get();
        }

        return $this->resourceService->getAllResourceByModuleId($moduleId)->get();
    }

    private function getHomeWidgetsByPage()
    {
        return config('srm_config.dashboard.max_home_widgets', 20);
    }

    private function getNumberOfPages($request)
    {
        $widgetsPerPage = $this->getHomeWidgetsByPage();
        return HomeDashboard::where('user_id', auth()->user()->user_ID)
            ->with('resource')
            ->orderBy('created_at', 'asc')
            ->paginate($widgetsPerPage);
    }

    public function getHomeLastPage($request)
    {
        return $this->getNumberOfPages($request)->lastPage();
    }

    public function getRecentWidget()
    {
        return HomeDashboard::where('user_id', auth()->user()->user_ID)
            ->with('resource')
            ->orderBy('created_at', 'desc')
            ->first();
    }

}
