<?php

namespace App\Services;

class DataConnectionTablesService
{
    public $dataConnectionService;
    public function __construct(DataConnectionService $dataConnectionService)
    {
        $this->dataConnectionService = $dataConnectionService;
    }

    public function getTables($connection)
    {

        if (trim($connection) === "default") {
            return $this->getDefaultConnectionTables();
        }

        return $this->getConnectionTables($connection);

    }

    public function getColumns($connection, $table, $types = [])
    {

        if (trim($connection) === "default") {
            return $this->getDefaultConnectionColumns($table, $types);
        }

        return $this->getConnectionTableColumns($connection, $table, $types);
    }

    public function getColumnType($connection, $table, $column)
    {

        if (trim($connection) === "default") {
            return $this->getDefaultConnectionColumnType($table, $column);
        }

        return $this->getConnectionTableColumnType($connection, $table, $column);

    }

    public function getValues($connection, $table, $column)
    {

        if (trim($connection) === "default") {
            return array_map(function ($value) {
                return is_null($value) ? "null" : (string) $value;
            }, $this->getDefaultConnectionValues($table, $column));
        }

        return array_map(function ($value) {
            return is_null($value) ? "null" : (string) $value;
        }, $this->getConnectionTableValues($connection, $table, $column));

    }

    public function getDefaultConnectionTables()
    {
        $connection = $this->dataConnectionService->getDefaultConnection();
        $prefix     = config('srm_config.installer.table_prefix', 'srm_');

        $tablesAndViews = [];

        $databaseName = $connection->getDatabaseName();

        foreach ($connection->getSchemaBuilder()->getAllTables() as $table) {
            $tablesAndViews[] = $table->{"Tables_in_" . $databaseName};
        }

        $views = $connection->select("SELECT TABLE_NAME FROM INFORMATION_SCHEMA.VIEWS WHERE TABLE_SCHEMA = ?", [$databaseName]);

        foreach ($views as $view) {
            $tablesAndViews[] = $view->TABLE_NAME;
        }

        $tablesAndViews = array_filter($tablesAndViews, function ($element) use ($prefix) {
            return strpos($element, $prefix) !== 0;
        });

        return $tablesAndViews;
    }

    public function getDefaultConnectionColumns($table, $types = [])
    {
        $connection = $this->dataConnectionService->getDefaultConnection();

        $columns = [];

        if (empty($types)) {

            foreach ($connection->getSchemaBuilder()->getColumnListing($table) as $column) {
                $columns[] = $column;
            }

        } else {

            if (empty(array_diff($types, ["date", "datetime", "char", "varchar", "string", "timestamp"]))) {

                $columns = $this->getDateTimeColumns('default', $table, $types);
            } else {
                $columns = $this->getColumnsType('default', $table, $types);
            }

        }

        return $columns;
    }

    public function getDefaultConnectionColumnType($table, $column)
    {
        $connection = $this->dataConnectionService->getDefaultConnection();

        try {
            return $connection->getSchemaBuilder()->getColumnType($table, strtolower($column));

        } catch (\Exception $e) {
            return $connection->getSchemaBuilder()->getColumnType($table, $column);

        }

    }

    public function getTreeTables($connectionId)
    {
        $treeData   = [];
        $connection = $connectionId === "default"
        ? $this->dataConnectionService->getDefaultConnection()
        : $this->dataConnectionService->getConnection($connectionId);

        $databaseName = $connection->getDatabaseName();
        $tables       = $connectionId === "default"
        ? $this->getDefaultConnectionTables()
        : $this->getConnectionTables($connectionId);

        // Add database node
        $treeData[] = [
            'id'     => 'db_' . $databaseName,
            'text'   => $databaseName,
            'parent' => '#',
            'icon'   => 'fa fa-database'
        ];

        foreach ($tables as $table) {
            $treeData[] = [
                'id'     => 'table_' . $table,
                'text'   => $table,
                'parent' => 'db_' . $databaseName,
                'icon'   => 'fa fa-table'
            ];

            $columns = $this->getColumns($connectionId, $table);

            foreach ($columns as $column) {

                if ($this->isEnum($connection, $table, $column)) {
                    $columnType = "enum";

                } elseif ($this->isTinyIntegerOne($connection, $table, $column)) {
                    $columnType = "tinyint(1)";

                } elseif ($this->isYear($connection, $table, $column)) {
                    $columnType = "year";

                } else {
                    $columnType = $this->getColumnType($connectionId, $table, $column);

                }

                $treeData[] = [
                    'id'     => 'column_' . $table . '_' . $column,
                    'text'   => $column,
                    'table'  => $table,
                    'type'   => $columnType,
                    'enum'   => in_array(strtolower($columnType), ['enum', 'set'])
                    ? implode(',', $this->getEnumValues($connection, $table, $column))
                    : null,
                    'parent' => 'table_' . $table,
                    'icon'   => 'fa fa-columns'
                ];

            }

        }

        return $treeData;
    }

    public function getDemoTree($connectionId)
    {
        $treeData = [];

        $connection = $this->dataConnectionService->getDefaultConnection();

        $databaseName = $connection->getDatabaseName();

        $tables = [
            "test_Product"    => [
                "productId"    => ["type" => "bigint"],
                "supplierId"   => ["type" => "bigint"],
                "productName"  => ["type" => "varchar"],
                "MadeIn"       => ["type" => "varchar"],
                "unitPrice"    => ["type" => "decimal"],
                "unitsInStock" => ["type" => "smallint"]
            ],
            "test_SalesOrder" => [
                "orderId"     => ["type" => "bigint"],
                "custId"      => ["type" => "bigint"],
                "employeeId"  => ["type" => "bigint"],
                "productId"   => ["type" => "bigint"],
                "Paid"        => ["type" => "int"],
                "shipCountry" => ["type" => "varchar"],
                "shipRegion"  => ["type" => "varchar"],
                "orderDate"   => ["type" => "datetime"]
            ],

            "test_Customer"   => [
                "custId"      => ["type" => "bigint"],
                "LeadSource"  => ["type" => "enum", "enum" => ["Flyers or Brochures", "Billboards", "Events", "Direct Sales Pitch", "Word of Mouth and Referrals"]],
                "contactName" => ["type" => "varchar"],
                "companyName" => ["type" => "varchar"],
                "country"     => ["type" => "varchar"],
                "region"      => ["type" => "varchar"],
                "created_at"  => ["type" => "timestamp"]
            ],
            "test_Employee"   => [
                "employeeId" => ["type" => "bigint"],
                "gender"     => ["type" => "varchar"],
                "firstname"  => ["type" => "varchar"],
                "lastname"   => ["type" => "varchar"],
                "hireDate"   => ["type" => "datetime"],
                "city"       => ["type" => "varchar"],
                "country"    => ["type" => "varchar"],
                "status"     => ["type" => "enum", "enum" => ["Probation", "Active", "Terminated"]],
                "Department" => ["type" => "enum", "enum" => ["Management", "Sales", "Operation", "HR", "Customer Support"]],
                "salary"     => ["type" => "decimal"]
            ],
            "test_Supplier"   => [
                "supplierId"  => ["type" => "bigint"],
                "companyName" => ["type" => "varchar"],
                "contactName" => ["type" => "varchar"],
                "country"     => ["type" => "varchar"]
            ]
        ];

        // Add database node
        $treeData[] = [
            'id'     => 'db_' . $databaseName,
            'text'   => $databaseName,
            'parent' => '#',
            'icon'   => 'fa fa-database'
        ];

        foreach ($tables as $table => $columns) {
            $treeData[] = [
                'id'     => 'table_' . $table,
                'text'   => $table,
                'parent' => 'db_' . $databaseName,
                'icon'   => 'fa fa-table'
            ];

            foreach ($columns as $column => $columnInfo) {
                $treeData[] = [
                    'id'     => 'column_' . $table . '_' . $column,
                    'text'   => $column,
                    'table'  => $table,
                    'type'   => $columnInfo['type'],
                    'enum'   => in_array(strtolower($columnInfo['type']), ['enum'])
                    ? implode(',', $columnInfo['enum'])
                    : null,
                    'parent' => 'table_' . $table,
                    'icon'   => 'fa fa-columns'
                ];

            }

        }

        return $treeData;

    }

    public function getDefaultConnectionValues($table, $column)
    {
        $connection = $this->dataConnectionService->getDefaultConnection();

        return array_column($connection->table($table)->select($column)->distinct()->get()->toArray(), $column);
    }

    public function getConnectionTables($connection)
    {
        $connection = $this->dataConnectionService->getConnection($connection);
        $prefix     = config('srm_config.installer.table_prefix', 'srm_');

        $tablesAndViews = [];

        $databaseName = $connection->getDatabaseName();

        foreach ($connection->getSchemaBuilder()->getAllTables() as $table) {
            $tablesAndViews[] = $table->{"Tables_in_" . $databaseName};
        }

        $views = $connection->select("SELECT TABLE_NAME FROM INFORMATION_SCHEMA.VIEWS WHERE TABLE_SCHEMA = ?", [$databaseName]);

        foreach ($views as $view) {
            $tablesAndViews[] = $view->TABLE_NAME;
        }

        $tablesAndViews = array_filter($tablesAndViews, function ($element) use ($prefix) {
            return strpos($element, $prefix) !== 0;
        });

        return $tablesAndViews;
    }

    public function getConnectionTableColumns($connection, $table, $types = [])
    {
        $connectionID = $connection;
        $connection   = $this->dataConnectionService->getConnection($connection);
        $columns      = [];

        if (empty($types)) {

            foreach ($connection->getSchemaBuilder()->getColumnListing($table) as $column) {
                $columns[] = $column;
            }

        } else {

            if (empty(array_diff($types, ["date", "datetime", "char", "varchar", "string", "timestamp"]))) {

                $columns = $this->getDateTimeColumns($connectionID, $table, $types);
            } else {
                $columns = $this->getColumnsType($connectionID, $table, $types);
            }

        }

        return $columns;
    }

    public function getDateTimeColumns($connection, $table, $types)
    {
        $connectionID = $connection;
        $connection   = $this->dataConnectionService->getConnection($connection);
        $columns      = [];

        foreach ($connection->getSchemaBuilder()->getColumnListing($table) as $column) {

            if (
                in_array($this->getColumnType($connectionID, $table, $column), ["date", "datetime", "timestamp"])

            ) {
                $columns[] = $column;

            } elseif ($this->checkIsDate($connection, $table, $column)) {

                $columns[] = $column;
            }

        }

        return $columns;
    }

    public function checkIsDate($connection, $table, $column)
    {

        if (!$connection->table($table)->select($column)->get()->isEmpty()) {
            $firstRowValue = $connection->table($table)->select($column)->get()[0]->$column;
        } else {
            return false;
        }

        // Try to parse the date using strtotime
        $timestamp = strtotime($firstRowValue);

        // Check if the timestamp is valid
        if ($timestamp === false) {
            return false;
        }

        // Check if the resulting date is a valid Gregorian date
        $date  = date('Y-m-d', $timestamp);
        $parts = explode('-', $date);

        if (!checkdate($parts[1], $parts[2], $parts[0])) {
            return false;
        }

        return true;

    }

    public function getColumnsType($connection, $table, $types)
    {
        $connectionID = $connection;
        $connection   = $this->dataConnectionService->getConnection($connection);

        foreach ($connection->getSchemaBuilder()->getColumnListing($table) as $column) {

            if (
                in_array($this->getColumnType($connectionID, $table, $column), $types)
            ) {
                $columns[] = $column;
            }

        }

        return $columns;

    }

    public function getConnectionTableColumnType($connection, $table, $column)
    {
        $connection = $this->dataConnectionService->getConnection($connection);

        try {

            return $connection->getSchemaBuilder()->getColumnType($table, strtolower($column));
        } catch (\Exception $e) {

            return $connection->getSchemaBuilder()->getColumnType($table, $column);

        }

    }

    public function getEnumValues($connection, $table, $column)
    {
        $database = $connection->getDatabaseName();

        $type = $connection->select("SHOW COLUMNS FROM `{$database}`.`{$table}` WHERE Field = ?", [$column])[0]->Type;

        if (preg_match("/(enum|set)\((.*)\)/", $type, $matches)) {
            return array_map(fn ($value) => trim($value, "'"), explode(',', $matches[2]));
        }

        return null;
    }

    public function isTinyIntegerOne($connection, $table, $column)
    {
        $database = $connection->getDatabaseName();

        $columnInfo = $connection->select("SHOW COLUMNS FROM `{$database}`.`{$table}` WHERE Field = ?", [$column]);

        if (!empty($columnInfo)) {
            return strtolower($columnInfo[0]->Type) === 'tinyint(1)';
        }

        return false;
    }

    public function isYear($connection, $table, $column)
    {
        $database = $connection->getDatabaseName();

        $columnInfo = $connection->select("SHOW COLUMNS FROM `{$database}`.`{$table}` WHERE Field = ?", [$column]);

        if (!empty($columnInfo)) {
            return strtolower($columnInfo[0]->Type) === 'year(4)';
        }

        return false;
    }

    public function isEnum($connection, $table, $column)
    {
        $database = $connection->getDatabaseName();

        $columnInfo = $connection->select("SHOW COLUMNS FROM `{$database}`.`{$table}` WHERE Field = ?", [$column]);

        if (!empty($columnInfo)) {
            $type = strtolower($columnInfo[0]->Type);

            if (strpos($type, 'enum') !== false) {
                return 'enum';
            }

            if (strpos($type, 'set') !== false) {
                return 'set';
            }

        }

        return false;
    }

    public function getConnectionTableValues($connection, $table, $column)
    {
        $connection = $this->dataConnectionService->getConnection($connection);

        return array_column($connection->table($table)->select($column)->distinct()->get()->toArray(), $column);
    }

    public function getConnectionTree($connection)
    {
        return response()->json([
            "success" => true,
            "data"    => $this->dataConnectionTablesService->getTreeTables($connection)
        ]);
    }

    public function checkJoin($connection, $rowTable, $columnTable)
    {
        $connection = $this->dataConnectionService->getConnection($connection);
        $relation   = getTablesRelation($connection, $rowTable, $columnTable);

        if ($relation) {
            return [
                'success' => true,
                'data'    => $relation
            ];
        }

    }

}
