<?php

namespace App\Providers;

use App\Models\Module;
use App\Models\Setting;
use App\Models\Dashboard;
use Illuminate\Support\Facades\URL;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\ServiceProvider;
use App\Jobs\SendScheduledSingleReportJob;
use Illuminate\Support\Facades\Request;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register()
    {

        if (!config('srm_install_config')) {
            config(['app.key' => 'base64:' . base64_encode(random_bytes(32))]);
        } else {
            config(['app.key' => config('srm_install_config.security_key')]);
        }

        config(['database.connections.mysql' => config('srm_db_config')]);
    }

    /**
     * Bootstrap any application services.
     */
    public function boot()
    {

        Request::macro('hasValidSignatureWhileIgnoring', function (array $ignore = [], $absolute = true) {

            // Normalize ignore parameter: single or array
            $ignore = is_array($ignore) ? $ignore : [$ignore];

            // Pull out the signature query parameter
            $signature = $this->query('signature');

            if (!$signature) {
                return false;
            }

            // Extract the other query items, excluding 'signature' and the ignored ones
            $query = $this->query();

            // Remove signature
            unset($query['signature']);

// Remove ignored keys
            foreach ($ignore as $key) {
                unset($query[$key]);
            }

            // Rebuild the URL without those removed query parameters
            $baseUrl = $absolute
            ? $this->url() // scheme + host + path (no query)
            : '/' . ltrim($this->getPathInfo(), '/');

            $queryString = $query ? '?' . http_build_query($query) : '';

            $original = $baseUrl . $queryString;

            // If expires param exists, validate expiration
            $expires = $this->query('expires');

            // Compute the expected signature — using same logic as Laravel's URL generator
            $calculated = hash_hmac('sha256', $original, config('app.key'));

            // Compare the signature & check expiry if present
            return hash_equals($calculated, $signature)
                && (!$expires || $expires >= now()->getTimestamp());

        });

        if (request()->secure()) {
            URL::forceScheme('https');
        }

        Paginator::useBootstrap();

        if (!Config::get("srm_db_config")) {
            Config::set('session.driver', 'array');
        }

        if ($this->app['config']['database.connections.mysql'] && !is_null(Setting::where('settings_key', 'timezone')->first())) {

            $this->setTimeZone();
            $this->setJobsTable();
            $this->setFailedJobsTable();

            Queue::after(function ($event) {
                $job         = $event->job;
                $jobInstance = unserialize($job->payload()['data']['command']);

                if ($jobInstance instanceof SendScheduledSingleReportJob) {
                    $jobInstance->after();
                }

            });

            View::composer('*', function ($view) {

                $srmVersion = getSrmVersion();
                $view->with([
                    'srmVersion' => $srmVersion
                ]);

                if (!is_null(auth()->user())) {
                    $userDashboards = Dashboard::where('user_id', auth()->user()->user_ID)->orderBy('order', 'asc')->limit(5)->get();

                    $blankReportModule = Module::getBlankReportModule();

                    $chartModule = Module::getChartModule();

                    $metricModule = Module::getMetricModule();

                    $pivotTableModule = Module::getPivotTableModule();
                    $srmVersion       = getSrmVersion();

                    $view->with([
                        'userDashboards'    => $userDashboards,
                        'blankReportModule' => $blankReportModule,
                        'chartModule'       => $chartModule,
                        'metricModule'      => $metricModule,
                        'pivotTableModule'  => $pivotTableModule,
                        'srmVersion'        => $srmVersion
                    ]);

                }

            });

        }

    }

    private function setTimeZone()
    {
        $timezone = Setting::where('settings_key', 'timezone')->first()->settings_value;
        config(['app.timezone' => $timezone]);
        date_default_timezone_set($timezone);
    }

    private function setJobsTable()
    {
        $prefix = config('srm_config.installer.table_prefix', 'srm_');
        config(['queue.connections.database.table' => $prefix . 'jobs']);
    }

    private function setFailedJobsTable()
    {
        $prefix = config('srm_config.installer.table_prefix', 'srm_');
        config(['queue.failed.table' => $prefix . 'failed_jobs']);
    }

}
