<?php

namespace App\Http\Controllers;

use App\Models\Module;
use App\Models\Resource;
use App\Services\UserService;
use App\Services\ReportService;
use App\Services\CategoryService;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Artisan;
use App\Http\Middleware\OnlyFullVersion;
use App\Http\Requests\DuplicateReportRequest;
use App\Http\Requests\UpdateReportCategoryRequest;

class ReportController extends Controller
{
    public $reportService;
    public $categoryService;
    public $userService;

    public function __construct(
        ReportService $reportService,
        CategoryService $categoryService,
        UserService $userService
    ) {
        $this->reportService   = $reportService;
        $this->categoryService = $categoryService;
        $this->userService     = $userService;

        $this->middleware(OnlyFullVersion::class)->only([
            'changeCategory',
            'duplicate',
            'destroy'
        ]);
    }

    public function index()
    {
        Artisan::call('config:clear');

        addDataToSession(auth()->user());

        $categories        = $this->categoryService->getCategories();
        $users             = $this->userService->getAllUsers();
        $blankReportModule = Module::getBlankReportModule();

        $reports = $this->reportService->getReports();

        return view('dashboard.reports.index', compact('categories', 'users', 'reports', 'blankReportModule'));
    }

    public function show(Resource $report)
    {

        $report = $this->reportService->show($report);
        $url    = getReportUrl($report->url);

        if (is_null(auth()->user()) && (strtolower($report->access_control_type) == "public" || (!is_null(request()->query('token')) && !is_null(session('loginType'))))) {
            return view('dashboard.reports.show-public', compact('report', 'url'));
        }
        
        addDataToSession(auth()->user());

        if (config('srm_config.dashboard.report_display') == "inside_page") {
            return view('dashboard.reports.show', compact('report', 'url'));
        }

        return response()->json([
            'url' => $url
        ]);

    }

    public function create()
    {
        addDataToSession(auth()->user());

        $url = $this->reportService->create();
        return response()->json([
            'url' => getReportUrl($url)
        ]);
    }

    public function edit(Resource $report)
    {
        addDataToSession(auth()->user());

        $url = $this->reportService->edit($report);
        return response()->json([
            'url' => getReportUrl($url)
        ]);
    }

    public function getCategories(Resource $report)
    {
        $categories = $this->categoryService->getCategories();

        return view('dashboard.reports.change-category', compact('report', 'categories'));
    }

    public function changeCategory(Resource $report, UpdateReportCategoryRequest $request)
    {
        $this->reportService->changeReportCategory($report, $request->validated());
        return response()->json([
            'message' => trans('report.updated')
        ]);
    }



    public function duplicate(Resource $report, DuplicateReportRequest $request)
    {
        $this->reportService->duplicate( $request->validated(), $report);
        
        return response()->json([
            'message' => trans('report.duplicate')
        ]);
    }

    public function destroy(Resource $report)
    {
        $this->reportService->destroy($report);

        return response()->json([
            'message' => trans('report.delete', [
                'report' => $report->name
            ])
        ]);
    }

}
