<?php

namespace App\Http\Controllers;

use App\Models\Group;
use App\Models\AccessKey;
use App\Services\AccessKeyService;
use App\Http\Middleware\OnlyFullVersion;
use App\Http\Requests\StoreAccessKeyRequest;
use App\Http\Requests\UpdateBarCodeDefinitionRequest;

class AccessKeysController extends Controller
{
    public $barcodeService;
    public $accessKeyService;
    
    public function __construct(AccessKeyService $accessKeyService)
    {
        $this->accessKeyService = $accessKeyService;

        $this->middleware(OnlyFullVersion::class)->only(
            [
                'store',
                'revoke',
                'update',
                'destroy'
            ]
        );
    }

    public function index()
    {
        $accessKeys = $this->accessKeyService->index();
        $groups     = Group::all();

        return view('dashboard.access-keys.index', compact('accessKeys', 'groups'));
    }

    public function store(StoreAccessKeyRequest $request)
    {
        $key = $this->accessKeyService->store($request->validated());

        return response()->json([
            "message" => "Access Key created successfully. Copy and store it securely.",
            "key"     => $key
        ]);
    }

    public function show(AccessKey $accessKey)
    {
        return view('dashboard.access-keys.partials._view_modal', compact('accessKey'));
    }

    public function update(UpdateBarCodeDefinitionRequest $request, AccessKey $accessKey)
    {
        $this->accessKeyService->update($request->validated(), $accessKey);

        return response()->json([
            "message" => "\"{$accessKey->label}\" is edited successfully"
        ]);
    }

    public function destroy(AccessKey $accessKey)
    {
        $this->accessKeyService->destroy($accessKey);

        return response()->json([
            'message' => 'This access key has been successfully deleted!'
        ]);
    }

    public function revoke(AccessKey $accessKey)
    {
        $this->accessKeyService->revoke($accessKey);

        return response()->json([
            'message' => 'This access key has been successfully revoked!'
        ]);
    }

    public function getAllAccessKeysByResourceId($resourceType, $resource)
    {
        $keys = $this->accessKeyService->getAllAccessKeysByResourceId($resourceType, $resource);
        return response()->json([
            "success" => true,
            "data"    => [
                'resources' => $keys,
                'access' => $this->accessKeyService->isPublic($resourceType, $resource) ? "public" : "private"
            ]
        ]);
    }


    public function getAccessKeyById($accessKey)
    {
        $key = $this->accessKeyService->getAccessKeyById($accessKey);

        return response()->json([
            "success" => true,
            "data"    => [
                'accessKey' => decrypt($key->access_key)
            ]
        ]);
    }
}
