<?php

namespace App\Generators;

use Illuminate\Support\Carbon;

class DataFilter
{
    private $filter;
    private $table;
    private $column;
    private $dataType;

    public function __construct($filter, $dataType)
    {
        $this->filter   = $filter;
        $this->dataType = $dataType;
        $this->table    = $this->filter['table'] ?? '';
        $this->column   = $this->filter['column'] ?? '';
    }

    private function numericFilter()
    {
        $sql      = '';
        $bindings = [];
        $criteria = $this->filter['criteria'];
        $value    = $this->filter['value'] ?? '';
        $min      = $this->filter['min'] ?? '';
        $max      = $this->filter['max'] ?? '';

        if ($criteria == "greater_than") {
            $sql .= " {$this->table}.{$this->column} > ? ";
            $bindings[] = $value;
        } elseif ($criteria == "less_than") {
            $sql .= " {$this->table}.{$this->column} < ? ";
            $bindings[] = $value;
        } elseif ($criteria == "between") {
            $sql .= " {$this->table}.{$this->column} BETWEEN ? AND ? ";
            $bindings[] = $min;
            $bindings[] = $max;
        } elseif ($criteria == "equal") {
            $sql .= " {$this->table}.{$this->column} = ? ";
            $bindings[] = $value;
        } elseif ($criteria == "not_equal") {
            $sql .= " {$this->table}.{$this->column} != ? ";
            $bindings[] = $value;
        } elseif ($criteria == "is_null") {
            $sql .= " {$this->table}.{$this->column} IS NULL ";
        } elseif ($criteria == "is_not_null") {
            $sql .= " {$this->table}.{$this->column} IS NOT NULL ";
        }

        return [$sql, $bindings];
    }

    private function specialFilter()
    {
        $sql      = '';
        $bindings = [];
        $criteria = $this->filter['criteria'];
        $value    = $this->filter['value'] ?? '';

        if ($criteria == "is_null") {
            $sql .= " {$this->table}.{$this->column} IS NULL ";
        } elseif ($criteria == "is_not_null") {
            $sql .= " {$this->table}.{$this->column} IS NOT NULL ";
        } elseif ($criteria == "is_empty_binary") {
            $sql .= " OCTET_LENGTH({$this->table}.{$this->column}) = 0 ";
        } elseif ($criteria == "is_empty_json") {
            $sql .= " JSON_LENGTH({$this->table}.{$this->column}) = 0 ";
        } elseif ($criteria == "json_contains_key" || $criteria == "json_contains_value") {
            $sql .= " {$this->table}.{$this->column} LIKE ? ";
            $bindings[] = '%' . $value . '%';
        }

        return [$sql, $bindings];
    }


    private function textualFilter()
    {
        $sql      = '';
        $bindings = [];
        $criteria = $this->filter['criteria'];
        $value    = $this->filter['value'] ?? '';

        if ($criteria == "contains") {
            $sql .= " {$this->table}.{$this->column} LIKE ? ";
            $bindings[] = '%' . $value . '%';
        } elseif ($criteria == "does_not_contain") {
            $sql .= " {$this->table}.{$this->column} NOT LIKE ? ";
            $bindings[] = '%' . $value . '%';
        } elseif ($criteria == "starts_with") {
            $sql .= " {$this->table}.{$this->column} LIKE ? ";
            $bindings[] = $value . '%';
        } elseif ($criteria == "ends_with") {
            $sql .= " {$this->table}.{$this->column} LIKE ? ";
            $bindings[] = '%' . $value;
        } elseif ($criteria == "equals") {
            $sql .= " {$this->table}.{$this->column} = ? ";
            $bindings[] = $value;
        } elseif ($criteria == "not_equal") {
            $sql .= " {$this->table}.{$this->column} != ? ";
            $bindings[] = $value;
        } elseif ($criteria == "is_null") {
            $sql .= " {$this->table}.{$this->column} IS NULL ";
        } elseif ($criteria == "is_not_null") {
            $sql .= " {$this->table}.{$this->column} IS NOT NULL ";
        }

        return [$sql, $bindings];
    }

    private function dateFilter()
    {
        $sql      = '';
        $bindings = [];
        $criteria = $this->filter['typeOfRange'] == "relative_range"
            ? $this->filter['relative']
            : $this->filter['fixed'];

        $value   = $this->filter['min'] ?? '';
        $max   = $this->filter['max'] ?? '';

        $column = "{$this->table}.{$this->column}";
        $now    = Carbon::now();

        switch ($criteria) {
            case "greater_than":
                $sql .= "$column > ? ";
                $bindings[] = $value;
                break;
            case "less_than":
                $sql .= "$column < ? ";
                $bindings[] = $value;
                break;
            case "between":
                $sql .= "$column BETWEEN ? AND ? ";
                $bindings[] = $value;
                $bindings[] = $max;
                break;
            case "equal":
                $sql .= "$column = ? ";
                $bindings[] = $value;
                break;
            case "not_equal":
                $sql .= "$column != ? ";
                $bindings[] = $value;
                break;
            case "is_null":
                $sql .= "$column IS NULL ";
                break;
            case "is_not_null":
                $sql .= "$column IS NOT NULL ";
                break;
            case "this_year":
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $now->startOfYear()->toDateString();
                $bindings[] = $now->endOfYear()->toDateString();
                break;
            case "last_12_months":
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = now()->subMonths(12)->firstOfMonth()->toDateTimeString();
                $bindings[] = now()->addDay()->toDateTimeString();
                break;
            case "this_quarter":
                $thisQuarterStart = $now->startOfQuarter()->toDateString();
                $thisQuarterEnd = $now->endOfQuarter()->toDateString();
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $thisQuarterStart;
                $bindings[] = $thisQuarterEnd;
                break;
            case "last_quarter":
                $lastQuarterStart = $now->subQuarters(1)->startOfQuarter()->toDateString();
                $lastQuarterEnd = $now->endOfQuarter()->toDateString();
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $lastQuarterStart;
                $bindings[] = $lastQuarterEnd;
                break;
            case "this_month":
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $now->startOfMonth()->toDateString();
                $bindings[] = $now->endOfMonth()->toDateString();
                break;
            case "last_30_days":
                $sql .= "$column >= ? ";
                $bindings[] = $now->subDays(30)->toDateString();
                break;
            case "this_week":
                $sql .= "$column >= ? ";
                $bindings[] = $now->startOfWeek(getStartOfWeek())->toDateString();
                break;
            case "last_7_days":
                $sql .= "$column >= ? ";
                $bindings[] = $now->subDays(7)->toDateString();
                break;
            case "today":
                $sql .= "DATE($column) = ? ";
                $bindings[] = $now->toDateString();
                break;
            case "yesterday":
                $sql .= "DATE($column) = ? ";
                $bindings[] = $now->subDay()->toDateString();
                break;
            case "this_hour":
                $startHour = $now->startOfHour()->toDateTimeString();
                $endHour = $now->addHour()->startOfHour()->toDateTimeString();
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $startHour;
                $bindings[] = $endHour;
                break;
            case "previous_hour":
                $startHour = $now->subHour()->startOfHour()->toDateTimeString();
                $endHour = $now->addHour()->startOfHour()->toDateTimeString();
                $sql .= "$column >= ? AND $column < ? ";
                $bindings[] = $startHour;
                $bindings[] = $endHour;
                break;
        }

        return [$sql, $bindings];
    }

    private function booleanFilter()
    {
        $sql = " {$this->table}.{$this->column} = ? ";
        $bindings = [$this->filter['value'] == "true" ? 1 : 0];
        return [$sql, $bindings];
    }

    private function enumFilter()
    {
        $values = explode(",", $this->filter['value']);
        $placeholders = implode(',', array_fill(0, count($values), '?'));
        $sql    = " {$this->table}.{$this->column} IN ({$placeholders}) ";
        return [$sql, $values];
    }

    private function checkTypeAndBuild()
    {
        $numericTypes     = ["int", "tinyint", "smallint", "mediumint", "bigint", "decimal", "numeric", "float", "double","integer","year"];
        $specialDataTypes = ["blob", "longblob", "mediumblob", "tinyblob", "geometry", "point", "linestring", "polygon", "json"];
        $textualTypes     = ["varchar", "text", "char", "tinytext", "mediumtext", "longtext", "string"];
        $dateTypes        = ["date", "datetime", "timestamp", "time"];
        $booleanTypes     = ["boolean", "bit", "tinyint(1)"];
        $enumTypes        = ["enum", "set"];
        $sql              = '';
        $bindings         = [];

        if (in_array($this->dataType, $numericTypes)) {
            return $this->numericFilter();
        } elseif (in_array($this->dataType, $specialDataTypes)) {
            return $this->specialFilter();
        } elseif (in_array($this->dataType, $textualTypes)) {
            return $this->textualFilter();
        } elseif (in_array($this->dataType, $dateTypes)) {
            return $this->dateFilter();
        } elseif (in_array($this->dataType, $booleanTypes)) {
            return $this->booleanFilter();
        } elseif (in_array($this->dataType, $enumTypes)) {
            return $this->enumFilter();
        }

        return ['', []];
    }

    public function getSqlAndBindings()
    {
        return $this->checkTypeAndBuild();
    }
}
