#!/usr/bin/env bash

set -euo pipefail
IFS=$'\n\t'

readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly LOGFILE="${SCRIPT_DIR}/install_cli.log"
readonly SRM_DIR="${SCRIPT_DIR}/srm"
readonly REQUIRED_PHP_EXTENSIONS=("curl" "zip")

declare -g warning_count=0
declare -g step_status=""
declare -g error_message=""
declare -g APACHE_RUNNING=false
declare -g NGINX_RUNNING=false
declare -g PACKAGE_MANAGER="unknown"
declare -g PHP_FPM_SOCKET=""

# ============================================================================
# UTILITY FUNCTIONS
# ============================================================================

log_info() {
    echo "[INFO] $*"
}

log_warning() {
    echo "[WARNING] $*"
    ((warning_count++))
}

log_error() {
    echo "[ERROR] $*" >&2
}

show_progress() {
    local pid=$1
    local interval=1

    while kill -0 "$pid" 2>/dev/null; do
        printf "#"
        sleep "$interval"
    done
    echo
}

check_command() {
    command -v "$1" >/dev/null 2>&1
}

handle_error() {
    log_error "Setup halted at step: $step_status"
    log_error "Issue: $error_message"
    exit 1
}

# ============================================================================
# SELINUX FUNCTIONS
# ============================================================================

is_selinux_unlabeled() {
    local file="$1"
    if check_command ls && check_command grep; then
        local context
        context=$(ls -Zd "$file" 2>/dev/null | awk '{print $1}')
        [[ "$context" == *unlabeled_t* ]]
    else
        return 1
    fi
}

set_selinux_permissions() {
    if ! check_command chcon; then
        return 0
    fi

    log_info "Setting SELinux permissions..."
    while IFS= read -r -d '' file; do
        if ! is_selinux_unlabeled "$file"; then
            chcon -t httpd_sys_rw_content_t "$file" 2>/dev/null || true
        fi
    done < <(find "$SRM_DIR" -type f -print0 2>/dev/null)
}

# ============================================================================
# PHP DETECTION AND SETUP FUNCTIONS
# ============================================================================

search_php_in_directories() {
    local search_dirs=("/opt" "/usr/bin" "/usr/local/bin")

    log_info "Searching for PHP binary in specified directories..."

    for dir in "${search_dirs[@]}"; do
        if [[ ! -d "$dir" ]]; then
            continue
        fi

        local php_path
        php_path=$(find "$dir" -type f -name "php*" 2>/dev/null | grep -E '/php(-[0-9]+\.[0-9]+\.[0-9]+)?$' | head -n 1)
        if [[ -n "$php_path" && -x "$php_path" ]]; then
            echo "$php_path"
            return 0
        fi
    done

    return 1
}

search_php_system_wide() {
    log_info "Searching for PHP binary system-wide..."
    local php_path
    php_path=$(find / -type f -name "php*" 2>/dev/null | grep -E '/php(-[0-9]+\.[0-9]+\.[0-9]+)?$' | head -n 1)

    if [[ -n "$php_path" && -x "$php_path" ]]; then
        echo "$php_path"
        return 0
    fi

    return 1
}

link_php_binary() {
    local php_path="$1"
    local target="/usr/local/bin/php"

    log_info "Linking PHP binary to $target..."

    if ln -sfnv "$php_path" "$target"; then
        log_info "PHP binary successfully linked"
        export PATH="/usr/local/bin:$PATH"
        hash -r
        return 0
    else
        log_error "Failed to link PHP binary"
        return 1
    fi
}

check_php_availability() {
    if check_command php; then
        log_info "PHP is already installed and accessible"
        php -v | head -n 1
        return 0
    fi

    log_info "PHP command not found, searching for PHP binary..."

    local php_path
    if php_path=$(search_php_in_directories) || php_path=$(search_php_system_wide); then
        log_info "PHP binary found at: $php_path"
        if link_php_binary "$php_path"; then
            log_info "PHP is now accessible as 'php'"
            php -v | head -n 1
            return 0
        fi
    fi

    error_message="PHP is not installed or accessible on this system"
    return 1
}

check_php_extensions() {
    log_info "Checking required PHP extensions..."

    # Detect active php.ini
    local php_ini_file
    php_ini_file=$(php --ini | grep "Loaded Configuration" | awk -F': ' '{print $2}')
    echo ""
    echo "Active php.ini file: $php_ini_file"
    echo ""

    while true; do
        local missing_extensions=()
        for extension in "${REQUIRED_PHP_EXTENSIONS[@]}"; do
            local check=$(php -m | grep -q "^$extension$")

            if ! $check; then
                missing_extensions+=("$extension")
            fi
        done

        if [[ ${#missing_extensions[@]} -eq 0 ]]; then
            log_info "All required PHP extensions are available"
            break
        fi

        echo ""
        log_error "Missing PHP extensions: ${missing_extensions[*]}"
        echo ""
        echo "To fix this:"
        echo "1. Open your php.ini file:"
        echo "   $php_ini_file"
        echo "2. Enable the following missing extensions:"
        for ext in "${missing_extensions[@]}"; do
            echo "   extension=$ext"
        done
        echo "3. Save the file and restart your web server (Apache, Nginx, or PHP-FPM)."
        echo ""
        echo "After doing that, press Enter to recheck..."
        read -r
    done
}

# ============================================================================
# FILE SYSTEM FUNCTIONS
# ============================================================================

create_symlink() {
    local source="$1"
    local destination="$2"

    if [[ ! -e "$source" ]]; then
        log_warning "Source path does not exist: $source"
        return 1
    fi

    log_info "Creating symbolic link: $source -> $destination"

    if ln -sfnv "$source" "$destination"; then
        log_info "Symbolic link created successfully: $destination"
        return 0
    else
        log_warning "Failed to create symbolic link: $destination"
        return 1
    fi
}

verify_symbolic_links() {
    local links=("$@")
    local broken_links=()

    log_info "Verifying symbolic links..."

    for dst in "${links[@]}"; do
        if [[ ! -L "$dst" ]] || [[ ! -e "$dst" ]]; then
            log_error "Missing or broken symbolic link at $dst"
            broken_links+=("$dst")
        fi
    done

    if ((${#broken_links[@]} > 0)); then
        log_error "One or more symbolic links are broken"
        return 1
    fi

    log_info "All symbolic links are valid"
    return 0
}

# ============================================================================
# COMPOSER FUNCTIONS
# ============================================================================

setup_composer_environment() {
    if [[ "$(id -u)" -eq 0 ]]; then
        export COMPOSER_ALLOW_SUPERUSER=1
        log_info "Running as root - COMPOSER_ALLOW_SUPERUSER enabled"
    fi

    export COMPOSER_MEMORY_LIMIT=-1

    if [[ ":$PATH:" != *":/usr/local/bin:"* ]]; then
        export PATH="/usr/local/bin:$PATH"
        log_info "Added /usr/local/bin to PATH"
    fi
}

run_composer_command() {
    local command="$1"
    local description="$2"

    log_info "$description"

    if ! php composer "$command" --no-interaction --prefer-dist --optimize-autoloader --no-dev; then
        error_message="Composer $command command failed"
        php composer diagnose 2>/dev/null || true
        return 1
    fi

    return 0
}

run_composer_commands() {
    local composer_dir="$1"

    if [[ ! -d "$composer_dir" ]]; then
        error_message="Composer directory does not exist: $composer_dir"
        return 1
    fi

    if ! cd "$composer_dir"; then
        error_message="Failed to change directory to $composer_dir"
        return 1
    fi

    setup_composer_environment

    log_info "Starting Composer dependency management"

    run_composer_command "install" "Installing Composer dependencies..." || return 1
    run_composer_command "update" "Updating Composer dependencies..." || return 1

    log_info "Dumping Composer autoloader..."
    if ! php composer dump-autoload; then
        error_message="Composer dump-autoload command failed"
        return 1
    fi

    cd - >/dev/null || return 1
    log_info "Composer commands completed successfully"
}

# ============================================================================
# PERMISSIONS AND OWNERSHIP FUNCTIONS
# ============================================================================

detect_web_server_user() {
    local users=("www-data" "apache" "nginx" "httpd")

    for user in "${users[@]}"; do
        if id -u "$user" >/dev/null 2>&1; then
            echo "$user"
            return 0
        fi
    done

    echo "${USER:-$(whoami)}"
}

set_directory_permissions() {
    local dir="$1"
    local web_server_user="$2"
    local failed_operations=()
    local success=true

    # Disable exit on error for this function
    set +e

    local current_owner
    current_owner=$(stat -c "%U" "$dir" 2>/dev/null) || current_owner="unknown"

    # Try to change ownership if different
    if [[ "$current_owner" != "$web_server_user" ]]; then
        log_info "Changing ownership of $dir to $web_server_user"
        chown -R "$web_server_user:$web_server_user" "$dir" 2>/dev/null
        if [[ $? -eq 0 ]]; then
            log_info "Successfully changed ownership of $dir"
        else
            log_warning "Failed to change ownership of $dir (this may be normal if not running as root)"
            failed_operations+=("$dir (ownership)")
            success=false
        fi
    fi

    # Set directory permissions
    find "$dir" -type d -exec chmod 2755 {} \; 2>/dev/null
    if [[ $? -eq 0 ]]; then
        log_info "Successfully set directory permissions for $dir"
    else
        log_warning "Failed to set directory permissions for $dir"
        failed_operations+=("$dir (directory permissions)")
        success=false
    fi

    # Set file permissions
    find "$dir" -type f -exec chmod 0644 {} \; 2>/dev/null
    if [[ $? -eq 0 ]]; then
        log_info "Successfully set file permissions for $dir"
    else
        log_warning "Failed to set file permissions for $dir"
        failed_operations+=("$dir (file permissions)")
        success=false
    fi

    # Re-enable exit on error
    set -e

    # Report failures but don't fail the entire installation
    if ! $success; then
        log_warning "Some permission operations failed for $dir:"
        printf '    %s\n' "${failed_operations[@]}"
    fi

    return 0 # Always return success to avoid stopping installation
}

check_and_set_permissions() {
    local directories=("$@")
    local non_existent_directories=()
    local web_server_user

    web_server_user=$(detect_web_server_user)
    log_info "Using web server user: $web_server_user"
    log_info "Updating permissions and ownership..."

    for dir in "${directories[@]}"; do
        if [[ -d "$dir" ]]; then
            set_directory_permissions "$dir" "$web_server_user"
        else
            non_existent_directories+=("$dir")
        fi
    done

    if ((${#non_existent_directories[@]} > 0)); then
        log_warning "Following directories do not exist and will be skipped:"
        printf '  * %s\n' "${non_existent_directories[@]}"
    fi

    log_info "Permissions and ownership update completed"
    return 0 # Always succeed to continue installation
}

# ============================================================================
# PACKAGE MANAGER FUNCTIONS
# ============================================================================

detect_package_manager() {
    local managers=("apt-get" "yum" "dnf" "pacman")

    for manager in "${managers[@]}"; do
        if check_command "$manager"; then
            echo "$manager"
            return 0
        fi
    done

    return 1
}

check_package_manager() {
    log_info "Detecting package manager..."

    if PACKAGE_MANAGER=$(detect_package_manager); then
        log_info "Package manager detected: $PACKAGE_MANAGER"
        return 0
    fi

    log_warning "No supported package manager found"

    while true; do
        echo "Supported package managers: apt-get, yum, dnf, pacman"
        read -rp "Enter your package manager or full path (or 'exit' to quit): " input

        case "$input" in
        exit | quit)
            log_error "User chose to exit"
            return 1
            ;;
        apt-get | yum | dnf | pacman)
            if check_command "$input"; then
                PACKAGE_MANAGER="$input"
                log_info "Using package manager: $PACKAGE_MANAGER"
                return 0
            else
                log_error "Package manager '$input' not found"
            fi
            ;;
        *)
            if [[ -x "$input" ]]; then
                PACKAGE_MANAGER="$input"
                log_info "Using custom package manager: $PACKAGE_MANAGER"
                return 0
            else
                log_error "Invalid package manager: $input"
            fi
            ;;
        esac
    done
}

detect_php_fpm_socket() {
    possible_sockets=(
        "/run/php/php7.4-fpm.sock"
        "/run/php/php8.0-fpm.sock"
        "/run/php/php8.1-fpm.sock"
        "/run/php/php8.2-fpm.sock"
        "/var/run/php/php7.4-fpm.sock"
        "/var/run/php/php8.0-fpm.sock"
        "/var/run/php/php8.1-fpm.sock"
        "/var/run/php/php8.2-fpm.sock"
        "/run/php-fpm/www.sock"
    )

    for socket in "${possible_sockets[@]}"; do
        if [ -e "$socket" ]; then
            PHP_FPM_SOCKET="$socket"
            return
        fi
    done

    while true; do
        read -p "PHP-FPM socket not found. Please enter the full path to your PHP-FPM socket (e.g., /run/php/php7.4-fpm.sock): " user_socket
        if [ -e "$user_socket" ]; then
            PHP_FPM_SOCKET="$user_socket"
            break
        else
            echo "The specified socket path does not exist. Please check the path and try again."
        fi
    done
}

set_nginx_server_name() {

    while true; do
        echo -ne "\nEnter a unique server_name that you intend to use for Smart Report Maker, \e[1;30;43msuch as domain.com, something.domain.com, localhost, or an IP address.\e[0m The server_name tells Nginx which domain names or IP addresses should respond to requests for the Smart Report Maker application. It’s important to ensure the name is unique in your configuration to prevent conflicts and ensure requests are handled correctly. Specify domains like domain.com, subdomains like something.example.com, localhost, or IP addresses. Please note that paths (e.g., /something) are not allowed. Server_name: "

        read server_name

        if [[ -n "$server_name" ]]; then
            break
        fi
    done
}

check_xampp_apache() {
    if [ -d "/opt/lampp" ] || [ -d "/opt/xampp" ]; then
        echo "Apache with XAMPP is detected."
        chown daemon:daemon -R "/opt/lampp/htdocs/srm" 2>/dev/null
        chown daemon:daemon -R "/opt/xampp/htdocs/srm" 2>/dev/null
        return 0
    else
        echo "Apache is detected but XAMPP is not detected."
        return 1
    fi
}

is_shared_hosting() {

    if sudo -V 2>/dev/null; then
        echo "Sudo access is available."
        return 1
    fi

    if [ -w "/etc" ]; then
        echo "The /etc directory is writable."
        return 1
    fi

    if [ -d "/usr/local/cpanel" ] || [ -d "/etc/psa" ]; then
        echo "Control panel (cPanel or Plesk) detected."
        return 0
    fi

    server_api=$(php -i | grep 'Server API' | awk '{print $3}')
    if [[ "$server_api" == "CGI/FastCGI" || "$server_api" == "LiteSpeed" ]]; then
        echo "Server API: $server_api (likely shared hosting)."
        return 0
    fi

    server_software=$(php -r 'echo $_SERVER["SERVER_SOFTWARE"];')
    if [[ "$server_software" == *"LiteSpeed"* || "$server_software" == *"Apache"* ]]; then
        echo "Server Software: $server_software (could be shared hosting)."
        return 0
    fi

    return 1
}

detect_web_server_shared_hosting() {

    APACHE_RUNNING=false
    NGINX_RUNNING=false

    server_software=$(php -r 'echo $_SERVER["SERVER_SOFTWARE"];')

    if [[ "$server_software" == *"Apache"* ]]; then
        echo "Apache is being used (detected via PHP)."
        APACHE_RUNNING=true
        NGINX_RUNNING=false

    elif [[ "$server_software" == *"nginx"* ]]; then
        echo "Nginx is being used (detected via PHP)."
        NGINX_RUNNING=true
        APACHE_RUNNING=false
    fi

    if pgrep -x "httpd" >/dev/null || pgrep -x "apache2" >/dev/null; then
        echo "Apache is running (detected via process check)."
        APACHE_RUNNING=true
        NGINX_RUNNING=false

    elif pgrep -x "nginx" >/dev/null; then
        echo "Nginx is running (detected via process check)."
        NGINX_RUNNING=true
        APACHE_RUNNING=false

    fi

    headers=$(curl -I -s http://localhost)
    if echo "$headers" | grep -q "Server: Apache"; then
        APACHE_RUNNING=true
        NGINX_RUNNING=false
    elif echo "$headers" | grep -q "Server: nginx"; then
        NGINX_RUNNING=true
        APACHE_RUNNING=false
    fi

    if $APACHE_RUNNING && ! $NGINX_RUNNING; then
        echo "Apache is detected."
        APACHE_RUNNING=true
        NGINX_RUNNING=false
    elif ! $APACHE_RUNNING && $NGINX_RUNNING; then
        echo "Nginx is detected."
        NGINX_RUNNING=true
        APACHE_RUNNING=false
        configure_nginx
    elif $APACHE_RUNNING && $NGINX_RUNNING; then

        echo "Both Apache and Nginx are detected."
        while true; do
            read -p "Which web server would you like to use? (apache/nginx): " USER_CHOICE
            case "$USER_CHOICE" in
            apache)
                APACHE_RUNNING=true
                NGINX_RUNNING=false
                check_xampp_apache
                break
                ;;
            nginx)
                NGINX_RUNNING=true
                APACHE_RUNNING=false
                configure_nginx
                break
                ;;
            *)
                echo "Invalid choice. Please enter 'apache' or 'nginx'."
                ;;
            esac
        done
    else
        echo "No web server detected."
        warning_count=$((warning_count + 1))
        error_message="No web server detected"
        return 1
    fi

}

detect_web_server() {
    APACHE_RUNNING=false
    NGINX_RUNNING=false

    if check_command "httpd"; then
        APACHE_RUNNING=true
    elif check_command "apache2"; then
        APACHE_RUNNING=true
    fi

    if check_command "nginx"; then
        NGINX_RUNNING=true
    fi

    if $APACHE_RUNNING && ! $NGINX_RUNNING; then
        echo "Apache is detected."
        APACHE_RUNNING=true
        NGINX_RUNNING=false
        check_xampp_apache
        enable_mod_rewrite
    elif ! $APACHE_RUNNING && $NGINX_RUNNING; then
        echo "Nginx is detected."
        NGINX_RUNNING=true
        APACHE_RUNNING=false
        configure_nginx
    elif $APACHE_RUNNING && $NGINX_RUNNING; then

        echo "Both Apache and Nginx are detected."
        while true; do
            read -p "Which web server would you like to use? (apache/nginx): " USER_CHOICE
            case "$USER_CHOICE" in
            apache)
                APACHE_RUNNING=true
                NGINX_RUNNING=false
                check_xampp_apache
                enable_mod_rewrite
                break
                ;;
            nginx)
                NGINX_RUNNING=true
                APACHE_RUNNING=false
                configure_nginx
                break
                ;;
            *)
                echo "Invalid choice. Please enter 'apache' or 'nginx'."
                ;;
            esac
        done
    elif [ -d "/opt/lampp" ] || [ -d "/opt/xampp" ]; then
        APACHE_RUNNING=true
        check_xampp_apache
        enable_mod_rewrite
    else
        echo "No web server detected."
        warning_count=$((warning_count + 1))
        error_message="No web server detected"
        return 1
    fi

}

install_nano() {

    install_package nano || exit 1

    if [ $? -ne 0 ]; then
        echo -e "\nNo text editor found and can't install one. Please manually edit the generated config file $(pwd)/webuccino_srm.conf then try again by running \`\e[1;30;43mbash install.sh\e[0m\`"
        exit 1
    fi
}

read_webuccino_srm_conf_file() {

    if check_command "nano"; then
        EDITOR="nano"
        $EDITOR webuccino_srm.conf
    elif check_command "vim"; then
        EDITOR="vim"
        $EDITOR webuccino_srm.conf
    else
        install_nano
        if check_command "nano"; then
            EDITOR="nano"
            $EDITOR webuccino_srm.conf
        else
            echo -e "\nNo text editor found and can't install one. Please manually edit the generated config file $(pwd)/webuccino_srm.conf then try again by running $(bash install.sh)"
            exit 1
        fi
    fi
}

reload_nginx() {
    echo -e "\nReloading Nginx to apply the new configuration."

    systemctl reload nginx >/dev/null 2>&1

    if [ $? -ne 0 ]; then
        echo -e "\nSystem couldn't reload Nginx. Please manually reload Nginx by running sudo systemctl reload nginx, then run \`\e[1;30;43mbash install.sh --nginx_reloaded\e[0m\` to resume.\n".
        exit 1
    fi

}

test_nginx_configuration() {
    echo -e "\nTesting the Nginx configuration for any errors."

    if [ "$(id -u)" -ne 0 ]; then
        echo -e "Nginx configuration test failed. Please run \`sudo nginx -t\`  then run \` \e[1;30;43mbash install.sh --nginx_tested \e[0m \` to resume.."
        exit 1
    fi

    nginx -t >/dev/null 2>&1

    if [ $? -ne 0 ]; then
        echo -ne "Nginx configuration test failed. Please review and fix the errors, then run \`\e[1;30;43mbash install.sh --config_copied \e[0m\` to resume\n".
        exit 1
    fi

    echo -e "\nNginx configuration test passed successfully"

    reload_nginx
}

copy_webuccino_srm_conf_file() {
    local full_path

    full_path="$(pwd)/webuccino_srm.conf"

    local manual_message="\nPlease manually move the file $full_path to the Nginx configuration directory, typically located at \`/etc/nginx/conf.d/\`. Once completed, run \`\e[1;30;43mbash install.sh --config_copied\e[0m\`.\n"

    while true; do
        echo ""
        read -p "The system needs to copy the configuration file $full_path to the Nginx configuration directory, typically located at \`/etc/nginx/conf.d/\`. 
If you would like the script to handle the copying, press Y. 
If you prefer to copy it manually (a good option if you're using a custom configuration directory or are on shared hosting without root access), press N: " copy_config

        case $copy_config in
        [Yy]*)
            cp -r "$full_path" /etc/nginx/conf.d/
            if [ $? -ne 0 ]; then
                echo -e "$manual_message"
                exit 1
            fi

            echo ""
            read -p "Do you want to delete the copied file at $full_path? [y/N]: " confirm_delete
            case $confirm_delete in
            [Yy]*)
                rm -f "$full_path"
                echo "File deleted: $full_path"
                ;;
            *)
                echo "File not deleted. You can remove it manually if needed."
                ;;
            esac

            test_nginx_configuration
            break
            ;;
        [Nn]*)
            echo -e "$manual_message"
            exit 1
            ;;
        *)
            echo "Please answer Y or N."
            ;;
        esac
    done
}

enable_mod_rewrite() {
    if [ -f /etc/httpd/conf/httpd.conf ]; then
        HTTPD_CONF="/etc/httpd/conf/httpd.conf"
        DOC_ROOT="/var/www/html"
    elif [ -f /etc/apache2/apache2.conf ]; then
        HTTPD_CONF="/etc/apache2/apache2.conf"
        DOC_ROOT="/var/www/html"
    elif [ -f /opt/lampp/apache2/conf/httpd.conf ]; then
        HTTPD_CONF="/opt/lampp/apache2/conf/httpd.conf"
        DOC_ROOT="/opt/lampp/apache2/htdocs"
    elif [ -f /opt/xampp/apache2/conf/httpd.conf ]; then
        HTTPD_CONF="/opt/xampp/apache2/conf/httpd.conf"
        DOC_ROOT="/opt/xampp/apache2/htdocs"
    else
        while true; do
            read -p "Apache configuration file not found in default locations. Enter the full path to the Apache configuration file: " USER_CONF_PATH
            if [ -n "$USER_CONF_PATH" ] && [ -f "$USER_CONF_PATH" ]; then
                HTTPD_CONF="$USER_CONF_PATH"
                echo "Using user-provided configuration file: $HTTPD_CONF"
                break
            else
                echo "Invalid path. Please enter a valid Apache configuration file path."
            fi
        done
    fi

    echo "Configuring mod_rewrite in Apache configuration..."

    if [ ! -f "$HTTPD_CONF.bak" ]; then
        sudo cp "$HTTPD_CONF" "$HTTPD_CONF.bak" >/dev/null
        echo "Backup created at $HTTPD_CONF.bak"
    else
        echo "Backup already exists at $HTTPD_CONF.bak"
    fi

    path="$(pwd)/srm"

    cat <<EOF | sudo tee -a "$HTTPD_CONF" >/dev/null
    <Directory "$path">
            Options Indexes FollowSymLinks
            AllowOverride All
            Require all granted
    </Directory>
EOF

    if [ -f /etc/apache2/mods-available/rewrite.load ]; then
        echo "Enabling mod_rewrite module on Debian-based system..."
        sudo a2enmod rewrite >/dev/null
    fi
    restart_apache
}

restart_apache() {
    if systemctl list-unit-files | grep -q 'httpd'; then

        echo "Restarting Apache (httpd) on CentOS/RHEL/Fedora..."
        sudo systemctl restart httpd >/dev/null
    elif systemctl list-unit-files | grep -q 'apache2'; then

        echo "Restarting Apache (apache2) on Ubuntu/Debian..."
        sudo systemctl restart apache2 >/dev/null
    else
        echo "Warning: Apache service may not installed or you are using xampp. Please check your Apache installation."

        warning_count=$((warning_count + 1))
        error_message="Warning: Apache service may not installed or you are using xampp. Please check your Apache installation."
        return 0

    fi
}

configure_nginx() {
    if command -v nginx >/dev/null 2>&1; then

        project_root=$(pwd)

        detect_php_fpm_socket

        set_nginx_server_name

        nginx_conf=$(
            cat <<EOF
server {
    listen 80;
    server_name $server_name;
    root $project_root;

    include /etc/nginx/mime.types;

    index index.php;

    # Handle static assets directly (without rewriting to 'public/')
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)$ {
        try_files \$uri \$uri/ =404;
        access_log off;
        expires max;
    }

    location /srm {
        rewrite ^/srm/?$ /srm/dashboard/ redirect;
    }

    location /srm/dashboard/public/ {
        try_files \$uri \$uri/ /public/index.php\$is_args\$args;
    }

    location /srm/dashboard {
        rewrite ^(.*)$ /public/\$1 break;
        try_files \$uri \$uri/ /srm/dashboard/public/index.php\$is_args\$args;
    }

    location /srm/dashboard/public/srm_install {
        try_files \$uri \$uri/ /srm/srm_install/index.php\$is_args\$args;
    }

    location /srm/srm_install {
        try_files \$uri \$uri/ /srm/srm_install/index.php\$is_args\$args;
    }


    location ~ \.php\$ {
        # Split the path for PHP scripts
        fastcgi_split_path_info ^(.+\.php)(/.+)\$;

        # Specify the PHP-FPM socket or TCP connection
        fastcgi_pass unix:$PHP_FPM_SOCKET;

        # Specify the script filename
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;

        # Include standard FastCGI parameters
        include fastcgi_params;

        # Ensure the correct index file is used
        fastcgi_index index.php;
    }

    location ~ /\.ht {
        deny all;
    }

    # Disable directory listings
    autoindex off;

    # Disable Server signature
    server_tokens off;

    # Add mime type for .mjs files
    types {
        text/javascript mjs;
    }

    # Block access to the vendor directory
    location /vendor/ {
        deny all;
        return 403;
    }


    # Prevent direct access to files starting with a dot
    location ~ /\. {
        deny all;
    }

    # Prevent direct access to specific file extensions
    location ~* \.(xml|config|log|lock|yml)$ {
        deny all;
    }

    # Prevent access to composer.json
    location ~ composer.json {
        deny all;
    }

    # Prevent access to composer
    location ~ composer {
        deny all;
    }

    # Prevent direct access to files starting with "srm_install"
    location ~ ^/srm_install {
        deny all;
    }

    # Prevent direct access to files starting with "srm_modules"
    location ~ ^/srm_modules {
        deny all;
    }

    # Prevent direct access to files starting with "srm_storage"
    location ~ ^/srm_storage {
        deny all;
    }

    # Prevent direct access to files named "artisan" or containing the word "artisan"
    location ~* (^|/)?artisan(/|$) {
        deny all;
    }
}
EOF
        )
        echo "$nginx_conf" | tee webuccino_srm.conf >/dev/null

        if [ $? -eq 0 ]; then
            cat webuccino_srm.conf
            echo -e "\nA Nginx configuration file for the Smart Report Maker site has been generated. Please review the configuration file above to ensure it is correct and does not conflict with any existing sites on your server. This review ensures that all configurations are agreed upon before proceeding. You can either confirm it's correct or edit it if needed."

            while true; do
                echo ""
                read -p "Do you want to accept the configuration as it is? (Y to accept, N to edit): " accept_config

                case $accept_config in
                [Yy]*)
                    copy_webuccino_srm_conf_file
                    break
                    ;;
                [Nn]*)
                    read_webuccino_srm_conf_file
                    copy_webuccino_srm_conf_file
                    break
                    ;;
                *)
                    echo "Invalid input. Please enter Y to accept or N to edit."
                    ;;
                esac
            done
        else
            echo "Failed to write Nginx configuration"
            exit 1
        fi

    fi

}

update_package_cache() {
    log_info "Updating package cache..."

    case "$PACKAGE_MANAGER" in
    apt-get)
        "$PACKAGE_MANAGER" update >/dev/null 2>&1
        ;;
    yum | dnf)
        "$PACKAGE_MANAGER" check-update >/dev/null 2>&1 || true
        ;;
    pacman)
        "$PACKAGE_MANAGER" -Sy --noconfirm >/dev/null 2>&1
        ;;
    *)
        log_error "Unsupported package manager: $PACKAGE_MANAGER"
        return 1
        ;;
    esac
}

install_package() {
    local pkg="$1"

    log_info "Installing package: $pkg"
    check_package_manager || return 1

    update_package_cache || return 1

    case "$PACKAGE_MANAGER" in
    apt-get)
        "$PACKAGE_MANAGER" install -y "$pkg" >/dev/null 2>&1
        ;;
    yum | dnf)
        "$PACKAGE_MANAGER" install -y "$pkg" >/dev/null 2>&1
        ;;
    pacman)
        "$PACKAGE_MANAGER" -S --noconfirm "$pkg" >/dev/null 2>&1
        ;;
    *)
        log_error "Unsupported package manager: $PACKAGE_MANAGER"
        return 1
        ;;
    esac

    if [[ $? -eq 0 ]]; then
        log_info "Package $pkg installed successfully"
        return 0
    else
        log_error "Failed to install $pkg"
        return 1
    fi
}

# ============================================================================
# CRON JOB FUNCTIONS
# ============================================================================

extract_cron_command() {
    local job="$1"
    echo "$job" | awk '{$1=""; $2=""; $3=""; $4=""; $5=""; print $0}' | sed 's/^ *//'
}

cron_job_exists() {
    local job_command="$1"
    crontab -l 2>/dev/null | while IFS= read -r line; do
        local existing_command
        existing_command=$(extract_cron_command "$line")
        if [[ "$existing_command" == "$job_command" ]]; then
            return 0
        fi
    done
    return 1
}

add_cron_job() {
    local job="$1"
    local job_command
    job_command=$(extract_cron_command "$job")

    if cron_job_exists "$job_command"; then
        log_info "Cron job already exists: $job_command"
        return 0
    fi

    if (
        crontab -l 2>/dev/null
        echo "$job"
    ) | crontab - 2>/dev/null; then
        log_info "Added cron job: $job"
        return 0
    else
        log_warning "Failed to add cron job: $job"
        return 1
    fi
}

setup_cron_jobs() {
    local cron_jobs=("$@")

    if ((${#cron_jobs[@]} == 0)); then
        log_info "No cron jobs to install"
        return 0
    fi

    log_info "Setting up cron jobs..."

    for job in "${cron_jobs[@]}"; do
        add_cron_job "$job"
    done

    log_info "Cron job setup completed"
}

# ============================================================================
# DEPENDENCY INSTALLATION FUNCTIONS
# ============================================================================

check_and_install_dependencies() {
    log_info "Checking and installing dependencies..."

    if ! check_command git; then
        log_info "Git not found, installing..."
        if ! install_package git; then
            error_message="Failed to install Git"
            return 1
        fi
    else
        log_info "Git is already installed"
    fi

    return 0
}

# ============================================================================
# FINAL SETUP FUNCTIONS
# ============================================================================

display_final_message() {
    echo
    echo "=============================================="
    echo "          SRM Installation Complete!"
    echo "=============================================="

    if $APACHE_RUNNING; then
        echo "Access the SRM installer wizard at:"
        echo "  http://your_domain/path_to_srm/srm/dashboard/setup"
    elif $NGINX_RUNNING; then
        echo "Access the SRM installer wizard at:"
        echo "  http://your_domain/srm/dashboard/setup"
    fi

    echo
    if ((warning_count > 0)); then
        echo "Installation completed with $warning_count warning(s)"
    else
        echo "Installation completed successfully with no warnings"
    fi
    echo "=============================================="
}

# ============================================================================
# MAIN EXECUTION
# ============================================================================

main() {
    local composer_json_path="$SRM_DIR"
    local directories_to_check=(
        "$SRM_DIR"
        "$SRM_DIR/dashboard"
        "$SRM_DIR/dashboard/public"
        "$SRM_DIR/dashboard/config"
        "$SRM_DIR/dashboard/storage/logs"
        "$SRM_DIR/modules/SRM9/SRM/Reports9"
    )

    local symlinks_to_create=(
        "$SRM_DIR/dashboard/storage/app/public:$SRM_DIR/dashboard/public/srm_storage"
        "$SRM_DIR/srm_install:$SRM_DIR/dashboard/public/srm_install"
        "$SRM_DIR/modules:$SRM_DIR/dashboard/public/srm_modules"
    )

    local symlinks_to_check=(
        "$SRM_DIR/dashboard/public/srm_storage"
        "$SRM_DIR/dashboard/public/srm_install"
        "$SRM_DIR/dashboard/public/srm_modules"
    )

    local cron_jobs=(
        "* * * * * php $SRM_DIR/dashboard/scheduler.php && php $SRM_DIR/dashboard/worker.php > /dev/null 2>&1"
    )

    # Handle command line arguments for nginx resume functionality
    for arg in "$@"; do
        case $arg in
        --config_copied)
            test_nginx_configuration
            display_final_message
            exit 0
            ;;
        --nginx_tested)
            reload_nginx
            display_final_message
            exit 0
            ;;
        --nginx_reloaded)
            display_final_message
            exit 0
            ;;
        esac
    done

    # Main installation steps
    step_status="PHP availability check"
    check_php_availability || handle_error
    step_status="check php extensions"
    check_php_extensions

    step_status="package manager detection"
    check_package_manager || handle_error

    step_status="dependency installation"
    check_and_install_dependencies || handle_error

    step_status="Composer execution"
    run_composer_commands "$composer_json_path" || handle_error

    step_status="symbolic link creation"
    for symlink_pair in "${symlinks_to_create[@]}"; do
        IFS=':' read -r source destination <<<"$symlink_pair"
        create_symlink "$source" "$destination"
    done

    step_status="symbolic link verification"
    verify_symbolic_links "${symlinks_to_check[@]}" || handle_error

    step_status="directory permissions"
    check_and_set_permissions "${directories_to_check[@]}"

    step_status="SELinux permissions"
    set_selinux_permissions

    step_status="cron job setup"
    setup_cron_jobs "${cron_jobs[@]}"

    step_status="web server detection"
    if is_shared_hosting; then
        detect_web_server_shared_hosting || handle_error
    else
        detect_web_server || handle_error
    fi

    display_final_message
}

# Execute main function with all arguments
main "$@"  2>&1 | tee -a "$LOGFILE"
