#Requires -Version 5.1
<#
.SYNOPSIS
    SRM Installation Script for Windows
.DESCRIPTION
    Automated installation script for Smart Report Maker (SRM) on Windows systems.
    Handles PHP detection, dependency installation, and system configuration.
.PARAMETER ResumeAfterComposer
    Resume all steps After Composer Dependency Installation
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\install.ps1
.EXAMPLE
    powershell -ExecutionPolicy Bypass -File .\install.ps1 -ResumeAfterComposer
#>

param(
    [switch]$ResumeAfterComposer = $false
)

# ============================================================================
# GLOBAL VARIABLES AND CONSTANTS
# ============================================================================
$ErrorActionPreference = "Stop"
$global:ResumeAfterComposer = $ResumeAfterComposer

$global:ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$global:SrmDir = Join-Path $global:ScriptDir "srm"
$global:LogFile = Join-Path $global:ScriptDir "install_cli.log"
$global:RequiredExtensions = @("zip", "curl")
$global:PhpPath = ""
$global:CurrentStep = ""
$global:Warnings = @()

# ============================================================================
# LOGGING AND USER FEEDBACK FUNCTIONS
# ============================================================================
function Write-LogInfo {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [INFO] $Message"
    Write-Host $logMessage -ForegroundColor Green
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
}

function Write-LogWarning {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [WARNING] $Message"
    Write-Host $logMessage -ForegroundColor Yellow
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
    $global:Warnings += $Message
}

function Write-LogError {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [ERROR] $Message"
    Write-Host $logMessage -ForegroundColor Red
    if ($global:LogFile) {
        try {
            Add-Content -Path $global:LogFile -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Ignore logging errors
        }
    }
}

function Show-Progress {
    param(
        [int]$ProcessId,
        [int]$IntervalMs = 500
    )
    
    while (Get-Process -Id $ProcessId -ErrorAction SilentlyContinue) {
        Write-Host -NoNewline "."
        Start-Sleep -Milliseconds $IntervalMs
    }
    Write-Host ""
}

function Write-StepHeader {
    param([string]$StepName)
    $global:CurrentStep = $StepName
    Write-Host ""
    Write-Host "===================" -ForegroundColor Cyan
    Write-Host " $StepName" -ForegroundColor Cyan
    Write-Host "===================" -ForegroundColor Cyan
    Write-LogInfo "Starting step: $StepName"
}

# ============================================================================
# ERROR HANDLING FUNCTIONS
# ============================================================================
function Invoke-ErrorHandler {
    param([string]$ErrorMessage)
    Write-LogError "Installation failed at step: $global:CurrentStep"
    Write-LogError "Error: $ErrorMessage"
    
    Write-Host ""
    Write-Host "Installation Failed!" -ForegroundColor Red
    Write-Host "Step: $global:CurrentStep" -ForegroundColor Red
    Write-Host "Error: $ErrorMessage" -ForegroundColor Red
    Write-Host ""
    if ($global:LogFile) {
        Write-Host "Check the log file at: $global:LogFile" -ForegroundColor Yellow
    }
    
    Stop-Transcript -ErrorAction SilentlyContinue
    exit 1
}

function Test-AdminRights {
    $currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($currentUser)
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# ============================================================================
# PHP DETECTION AND SETUP FUNCTIONS
# ============================================================================
function Find-PhpInDirectories {
    $searchDirs = @(
        "C:\xampp\php",
        "D:\xampp\php",
        "C:\wamp64\bin\php",
        "D:\wamp64\bin\php", 
        "C:\php",
        "D:\php",
        "C:\Program Files\PHP",
        "C:\Program Files (x86)\PHP",
        "D:\Program Files\PHP",
        "D:\Program Files (x86)\PHP"
    )
    Write-LogInfo "Searching for PHP in common directories..."
    foreach ($dir in $searchDirs) {
        Write-LogInfo "Checking directory: $dir"
        
        if (-not (Test-Path $dir)) { 
            Write-LogInfo "Directory does not exist: $dir"
            continue 
        }
        
        Write-LogInfo "Directory exists, searching for php.exe..."
        $phpExe = Get-ChildItem -Path $dir -Filter "php.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
        
        if ($phpExe) {
            $phpPath = Join-Path $dir $phpExe.Name
            Write-LogInfo "Found php.exe, testing: $phpPath"
            
            Write-LogInfo "PHP found at: $phpPath"
            return $phpPath
        }
        else {
            Write-LogInfo "No php.exe found in: $dir"
            # Also check for common subdirectories
            $subDirs = @("", "bin", "cli")
            foreach ($subDir in $subDirs) {
                $subPath = if ($subDir) { Join-Path $dir $subDir } else { $dir }
                if (Test-Path $subPath) {
                    $phpInSub = Get-ChildItem -Path $subPath -Filter "php.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
                    if ($phpInSub) {
                        $phpPath = Join-Path $subPath $phpInSub.Name
                        Write-LogInfo "Found php.exe in subdirectory, testing: $phpPath"
                        Write-LogInfo "PHP found at: $phpPath"
                        return $phpPath
                    }
                }
            }
        }
    }
    
    Write-LogInfo "PHP not found in any common directories"
    return $null
}

function Get-PhpFromUser {
    while ($true) {
        Write-Host ""
        Write-Host "Please provide the full path to your PHP executable (php.exe):" -ForegroundColor Yellow
        Write-Host "Example: C:\xampp\php\php.exe" -ForegroundColor Gray
        
        $phpPath = Read-Host "PHP Path"
        
        if (-not $phpPath) {
            Write-Host "Path cannot be empty. Please try again." -ForegroundColor Red
            continue
        }
        
        if (-not (Test-Path $phpPath)) {
            Write-Host "File not found: $phpPath" -ForegroundColor Red
            continue
        }
        
        Write-LogInfo "PHP verified at user-provided path: $phpPath"
        return $phpPath
    }
}

function Initialize-PhpEnvironment {
    Write-StepHeader "PHP Environment Setup"
    
    # Check if PHP is in PATH
    try {
        $output = & "php" -v 2>$null
        if ($output -and $output.ToString().Contains("PHP")) {
            Write-LogInfo "PHP is accessible via PATH"
            $global:PhpPath = "php"
            return
        }
    }
    catch {
        Write-LogInfo "PHP not found in PATH, searching directories..."
    }
    
    # Search in common directories
    $phpPath = Find-PhpInDirectories
    
    if (-not $phpPath) {
        $phpPath = Get-PhpFromUser
    }
    
    if (-not $phpPath) {
        throw "Unable to locate PHP executable"
    }
    
    $global:PhpPath = $phpPath
    Write-LogInfo "PHP environment initialized with: $global:PhpPath"
}

function Check-PHPExtensions {
    param (
        [string[]]$RequiredExtensions
    )

    Write-StepHeader "Check PHP Extensions Step"

    Write-LogInfo "Checking required PHP extensions..." -ForegroundColor Cyan

    # Get active php.ini file path
    $phpIniOutput = & $global:PhpPath --ini 2>&1
    $phpIniPath = ($phpIniOutput | Select-String 'Loaded Configuration File') -replace '.*:\s*', ''

    if (-not $phpIniPath) {
        Write-Error "Unable to detect php.ini path. Is PHP installed and in PATH?" -ForegroundColor Red
        exit 1
    }

    Write-Host "`nActive php.ini file: $phpIniPath`n"

    while ($true) {
        $loadedExtensions = & $global:PhpPath -m 2>&1
        $missingExtensions = @()

        foreach ($extension in $RequiredExtensions) {
            if (-not ($loadedExtensions -match "^$extension$")) {
                $missingExtensions += $extension
            }
        }

        if ($missingExtensions.Count -eq 0) {
            Write-LogInfo "All required PHP extensions are available.`n" -ForegroundColor Green
            break
        }

        Write-LogError "Missing PHP extensions: $($missingExtensions -join ', ')" -ForegroundColor Red
        Write-Host ""
        Write-Host "Please do the following steps:"
        Write-Host "1. Open the php.ini file:"
        Write-Host "   $phpIniPath"
        Write-Host "2. Enable the following missing extensions by adding or uncommenting lines like:"
        foreach ($ext in $missingExtensions) {
            Write-Host "   extension=$ext"
        }
        Write-Host "3. Save the file and restart your web server (e.g., Apache, Nginx, or PHP-FPM)."
        Write-Host ""
        Read-Host "After doing that, press Enter to recheck..."
    }
}


# ============================================================================
# DEPENDENCY MANAGEMENT FUNCTIONS
# ============================================================================
function Test-GitAvailability {
    try {
        $null = & "git" --version 2>$null
        return $true
    }
    catch {
        return $false
    }
}

function Test-ZipExtension {
    try {
        $modules = & $global:PhpPath -m 2>$null
        return $modules -contains "zip"
    }
    catch {
        return $false
    }
}

function Install-DependenciesCheck {
    Write-StepHeader "Dependency Verification"
    
    if (Test-GitAvailability) {
        Write-LogInfo "Git is available"
        return
    }
    
    if (Test-ZipExtension) {
        Write-LogInfo "PHP zip extension is available (Git not required)"
        return
    }
    
    # Neither Git nor zip extension available
    $phpIni = Get-PhpIniPath
    Write-LogError "Neither Git nor PHP zip extension are available"
    Write-Host ""
    Write-Host "DEPENDENCY REQUIREMENT NOT MET" -ForegroundColor Red
    Write-Host ""
    Write-Host "Please install Git from: https://git-scm.com/downloads/win" -ForegroundColor Yellow
    Write-Host "OR enable the zip extension in your PHP configuration." -ForegroundColor Yellow
    Write-Host ""
    Write-Host "PHP ini file location: $phpIni" -ForegroundColor Gray
    Write-Host "To enable zip: Remove semicolon (;) from ;extension=zip" -ForegroundColor Gray
    Write-Host ""
    throw "Required dependencies not available"
}

function Get-PhpIniPath {
    try {
        $phpInfo = & $global:PhpPath -i 2>$null
        $iniLine = $phpInfo | Where-Object { $_ -like '*Loaded Configuration File*' }
        if ($iniLine) {
            return ($iniLine -split '=>')[1].Trim()
        }
    }
    catch {
        # Ignore errors
    }
    return "Not found"
}

# ============================================================================
# COMPOSER FUNCTIONS
# ============================================================================
# function Invoke-ComposerCommand {
#     param(
#         [string]$Command,
#         [string]$Description,
#         [string[]]$Arguments = @()
#     )
    
#     Write-LogInfo $Description
    
#     $composerPath = Join-Path $global:SrmDir "composer"
    
#     # Define command-specific arguments
#     $baseArgs = @($composerPath, $Command) + $Arguments + @("--no-interaction")
    
#     switch ($Command) {
#         "install" {
#             $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
#         }
#         "update" {
#             $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
#         }
#         "dump-autoload" {
#             # dump-autoload doesn't support --prefer-dist, --no-dev, etc.
#             $allArgs = $baseArgs + @("--optimize")
#         }
#         default {
#             $allArgs = $baseArgs
#         }
#     }
    
#     try {
#         $process = Start-Process -FilePath $global:PhpPath -ArgumentList $allArgs -NoNewWindow -PassThru -RedirectStandardError "stderr.log" -RedirectStandardOutput "stdout.log"
#         Show-Progress -ProcessId $process.Id
#         $process | Wait-Process
        
#         $stdout = Get-Content -Path "stdout.log" -Raw -ErrorAction SilentlyContinue
#         $stderr = Get-Content -Path "stderr.log" -Raw -ErrorAction SilentlyContinue
        
#         # Check for actual errors vs warnings/info
#         if ($stderr -and $stderr.Trim()) {
#             Write-LogInfo "Composer output: $($stderr.Trim())"
#         }
#         if ($stdout -and $stdout.Trim()) {
#             Write-LogInfo "Composer output: $($stdout.Trim())"
#         }
#     }
#     finally {
#         Remove-Item -Path "stderr.log", "stdout.log" -ErrorAction SilentlyContinue
#     }
# }

function Install-ComposerDependencies {
    Write-StepHeader "Composer Dependency Installation"

    $srm = $global:SrmDir
    $originalLocation = Get-Location
    try {
        Set-Location $srm
        
        # Check if composer.json exists
        if (-not (Test-Path "composer.json")) {
            Write-LogWarning "composer.json not found in $srm - skipping Composer operations"
            return
        }
        
        # Function to execute composer command and check for errors
        function Invoke-ComposerCommandSafe {
            param(
                [string]$Command,
                [string]$Description,
                [string[]]$Arguments = @()
            )
            
            Write-LogInfo $Description
            
            $composerPath = Join-Path $global:SrmDir "composer"
            
            # Define command-specific arguments
            $baseArgs = @($composerPath, $Command) + $Arguments + @("--no-interaction")
            
            switch ($Command) {
                "install" {
                    $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
                }
                "update" {
                    $allArgs = $baseArgs + @("--prefer-dist", "--optimize-autoloader", "--no-dev")
                }
                "dump-autoload" {
                    # dump-autoload doesn't support --prefer-dist, --no-dev, etc.
                    $allArgs = $baseArgs + @("--optimize")
                }
                default {
                    $allArgs = $baseArgs
                }
            }
            
            try {
                $process = Start-Process -FilePath $global:PhpPath -ArgumentList $allArgs -NoNewWindow -PassThru -RedirectStandardError "stderr.log" -RedirectStandardOutput "stdout.log"
                Show-Progress -ProcessId $process.Id
                $process | Wait-Process
                
                $stdout = Get-Content -Path "stdout.log" -Raw -ErrorAction SilentlyContinue
                $stderr = Get-Content -Path "stderr.log" -Raw -ErrorAction SilentlyContinue
                
                # Combine stdout and stderr for error checking
                $combinedOutput = "$stdout`n$stderr"
                # Check for common error indicators in the output
                $errorKeywords = @(
                    "(?i)error\s",
                    "(?i)failed\s",
                    "(?i)exception\s",
                    "(?i)fatal\s",
                    "(?i)could not\s",
                    "(?i)cannot\s",
                    "(?i)unable to\s",
                    "(?i)connection timed out\s",
                    "(?i)network error\s",
                    "(?i)curl error\s",
                    "(?i)ssl error\s",
                    "(?i)certificate\s",
                    "(?i)permission denied\s",
                    "(?i)access denied\s",
                    "(?i)file not found\s",
                    "(?i)directory not found\s",
                    "(?i)out of memory\s",
                    "(?i)killed\s",
                    "(?i)aborted\s",
                    "(?i)interrupted\s"
                )

                
                $hasError = $false
                foreach ($keyword in $errorKeywords) {
                    if ($combinedOutput -match $keyword) {
                        $hasError = $true
                        break
                    }
                }

                
                if ($hasError) {
                    Write-LogError "Composer command failed: $Command"
                    if ($stdout -and $stdout.Trim()) {
                        Write-LogError "Stdout: $($stdout.Trim())"
                    }
                    if ($stderr -and $stderr.Trim()) {
                        Write-LogError "Stderr: $($stderr.Trim())"
                    }
                    throw "Composer has stopped unexpectedly. This may be due to your antivirus or firewall interfering, or an unstable internet connection. Please try temporarily disabling your antivirus/firewall or checking your internet stability. You can also run the installer script again by repeating the previous command"
                }
                
                # Log output for successful operations
                if ($stderr -and $stderr.Trim()) {
                    Write-LogInfo "Composer output: $($stderr.Trim())"
                }
                if ($stdout -and $stdout.Trim()) {
                    Write-LogInfo "Composer output: $($stdout.Trim())"
                }
                
                Write-LogInfo "Command completed successfully"
                return $combinedOutput
            }
            finally {
                Remove-Item -Path "stderr.log", "stdout.log" -ErrorAction SilentlyContinue
            }
        }
        
        # Install dependencies first
        Invoke-ComposerCommandSafe -Command "install" -Description "Installing Composer dependencies..."
        
        # Only update if install was successful and update is needed
        try {
            Invoke-ComposerCommandSafe -Command "update" -Description "Updating Composer dependencies..."
        }
        catch {
            Write-LogWarning "Composer update failed, but install was successful. Continuing..."
            # Check if this is the specific error we want to handle
            if ($_.Exception.Message -match "Composer has stopped unexpectedly") {
                throw $_.Exception.Message
            }
        }
        
        # Generate optimized autoloader
        Invoke-ComposerCommandSafe -Command "dump-autoload --optimize" -Description "Generating optimized autoloader..."
        
        Write-LogInfo "Composer operations completed successfully"
    }
    catch {
        if ($_.Exception.Message -match "Composer has stopped unexpectedly") {
            Write-LogError $_.Exception.Message
            exit 1
        } else {
            Write-LogError "Composer operations failed: $($_.Exception.Message)"
            # Don't throw here - allow the update to continue even if Composer fails
            Write-LogWarning "Continuing update process despite Composer issues..."
        }
    }
    finally {
        Set-Location $originalLocation
    }
}

# ============================================================================
# FILE SYSTEM FUNCTIONS
# ============================================================================
function New-SafeJunction {
    param(
        [string]$Source,
        [string]$Destination
    )
    
    if (-not (Test-Path $Source)) {
        Write-LogWarning "Source path does not exist: $Source"
        return $false
    }
    
    # Ensure source is a directory (junctions only work with directories)
    $sourceItem = Get-Item $Source -Force
    if (-not $sourceItem.PSIsContainer) {
        Write-LogWarning "Source must be a directory for junction creation: $Source"
        return $false
    }
    
    # Remove existing link/directory if it exists
    if (Test-Path $Destination) {
        $item = Get-Item $Destination -Force
        if ($item.Attributes -band [System.IO.FileAttributes]::ReparsePoint) {
            Write-LogInfo "Removing existing junction/symbolic link: $Destination"
            Remove-Item -Path $Destination -Force
        }
        else {
            Write-LogWarning "Removing existing directory/file: $Destination"
            Remove-Item -Path $Destination -Recurse -Force
        }
    }
    
    try {
        New-Item -ItemType Junction -Path $Destination -Target $Source -Force | Out-Null
        Write-LogInfo "Created junction: $Destination -> $Source"
        return $true
    }
    catch {
        Write-LogWarning "Failed to create junction: $Destination -> $Source. Error: $($_.Exception.Message)"
        return $false
    }
}


function Test-Junctions {
    param([string[]]$Links)
    
    Write-LogInfo "Verifying junctions..."
    $brokenLinks = @()
    
    foreach ($link in $Links) {
        if (-not (Test-Path $link)) {
            Write-LogError "Missing junction: $link"
            $brokenLinks += $link
        }
        else {
            $item = Get-Item $link -Force
            if (-not ($item.Attributes -band [System.IO.FileAttributes]::ReparsePoint)) {
                Write-LogError "Path is not a junction: $link"
                $brokenLinks += $link
            }
            else {
                # Additional check to ensure it's specifically a junction (not a symbolic link)
                try {
                    $reparsePoint = $item.LinkType
                    if ($reparsePoint -ne "Junction") {
                        Write-LogError "Path is a reparse point but not a junction: $link (Type: $reparsePoint)"
                        $brokenLinks += $link
                    }
                }
                catch {
                    # If LinkType property is not available, assume it's valid if it's a ReparsePoint
                    Write-LogInfo "Could not determine exact reparse point type for: $link (assuming valid junction)"
                }
            }
        }
    }
    
    if ($brokenLinks.Count -gt 0) {
        Write-LogError "One or more junctions are broken"
        return $false
    }
    
    Write-LogInfo "All junctions are valid"
    return $true
}

function Initialize-Junctions {
    Write-StepHeader "Junction Creation"
    
    $links = @(
        @{
            Source = Join-Path $global:SrmDir "dashboard\storage\app\public"
            Destination = Join-Path $global:SrmDir "dashboard\public\srm_storage"
        },
        @{
            Source = Join-Path $global:SrmDir "srm_install"
            Destination = Join-Path $global:SrmDir "dashboard\public\srm_install"
        },
        @{
            Source = Join-Path $global:SrmDir "modules"
            Destination = Join-Path $global:SrmDir "dashboard\public\srm_modules"
        }
    )
    
    foreach ($link in $links) {
        New-SafeJunction -Source $link.Source -Destination $link.Destination
    }
    
    $linkPaths = $links | ForEach-Object { $_.Destination }
    if (-not (Test-Junctions -Links $linkPaths)) {
        throw "Junction verification failed"
    }
}
# ============================================================================
# PERMISSIONS FUNCTIONS
# ============================================================================
function Set-DirectoryPermissions {
    param([string[]]$Directories)
    
    Write-StepHeader "Permission Configuration"
    
    if (-not (Test-AdminRights)) {
        Write-LogWarning "Not running as administrator - some permission operations may fail"
    }
    
    foreach ($dir in $Directories) {
        if (-not (Test-Path $dir)) {
            Write-LogWarning "Directory does not exist: $dir"
            continue
        }
        
        Write-LogInfo "Setting permissions for: $dir"
        try {
            # Grant full control to Everyone (modify permissions)
            & icacls $dir /grant "Everyone:(OI)(CI)M" /T /C | Out-Null
            Write-LogInfo "Successfully set permissions for: $dir"
        }
        catch {
            Write-LogWarning "Failed to set permissions for: $dir. Error: $($_.Exception.Message)"
        }
    }
}

# ============================================================================
# MAIN INSTALLATION FUNCTIONS
# ============================================================================
function Initialize-Installation {
    Write-StepHeader "Installation Initialization"
    
    # Initialize logging with safe file handling
    Initialize-Logging
    
    Write-LogInfo "SRM Installation Started"
    Write-LogInfo "Script directory: $global:ScriptDir"
    Write-LogInfo "SRM directory: $global:SrmDir"
    Write-LogInfo "Administrator rights: $(Test-AdminRights)"
    
    if (-not (Test-Path $global:SrmDir)) {
        throw "SRM directory not found: $global:SrmDir"
    }
}

function Initialize-Logging {
    # Stop any existing transcript
    try {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
    catch {
        # Ignore errors from stopping transcript
    }
    
    # Find an available log file name
    $logCounter = 0
    $baseLogFile = $global:LogFile
    $transcriptEnabled = $false
    
    while ($logCounter -le 10) {
        try {
            # Test if we can write to this log file
            $testContent = "# Log file test - $(Get-Date)"
            Add-Content -Path $global:LogFile -Value $testContent -ErrorAction Stop
            
            # If we can write, try to start transcript
            try {
                Start-Transcript -Path $global:LogFile -Append | Out-Null
                $transcriptEnabled = $true
                Write-Host "Logging initialized with transcript: $global:LogFile" -ForegroundColor Green
                break
            }
            catch {
                # Transcript failed but we can write to file, continue with basic logging
                Write-Host "Logging initialized (basic mode): $global:LogFile" -ForegroundColor Yellow
                break
            }
        }
        catch {
            # Cannot write to this file, try a different name
            $logCounter++
            if ($logCounter -le 10) {
                $fileInfo = [System.IO.FileInfo]$baseLogFile
                $global:LogFile = Join-Path $fileInfo.Directory.FullName "$($fileInfo.BaseName)_$logCounter$($fileInfo.Extension)"
            }
        }
    }
    
    if ($logCounter -gt 10) {
        # Generate a unique name with timestamp
        $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
        $fileInfo = [System.IO.FileInfo]$baseLogFile
        $global:LogFile = Join-Path $fileInfo.Directory.FullName "$($fileInfo.BaseName)_$timestamp$($fileInfo.Extension)"
        
        try {
            $testContent = "# Log file created - $(Get-Date)"
            Add-Content -Path $global:LogFile -Value $testContent -ErrorAction Stop
            Write-Host "Logging initialized with timestamp: $global:LogFile" -ForegroundColor Yellow
        }
        catch {
            Write-Warning "Could not initialize any log file. Proceeding without file logging."
            $global:LogFile = $null
        }
    }
}

function Show-InstallationSummary {
    Write-Host ""
    Write-Host "===================" -ForegroundColor Green
    Write-Host " INSTALLATION COMPLETE!" -ForegroundColor Green
    Write-Host "===================" -ForegroundColor Green
    Write-Host ""
    Write-Host "Next Steps:" -ForegroundColor Cyan
    Write-Host "1. Open your web browser" -ForegroundColor White
    Write-Host "2. Navigate to: http://your-domain/path-to-srm/srm/dashboard/setup" -ForegroundColor White
    Write-Host "3. Follow the setup wizard to complete installation" -ForegroundColor White
    Write-Host ""
    
    if ($global:Warnings.Count -gt 0) {
        Write-Host "Installation completed with $($global:Warnings.Count) warning(s)" -ForegroundColor Yellow
        if ($global:LogFile) {
            Write-Host "Check the log file for details: $global:LogFile" -ForegroundColor Yellow
        }
    }
    else {
        Write-Host "Installation completed successfully with no warnings!" -ForegroundColor Green
    }
    Write-Host ""
}

function Start-Installation {
    try {
        Initialize-Installation
        Initialize-PhpEnvironment
        Check-PHPExtensions -RequiredExtensions $global:RequiredExtensions
        Install-DependenciesCheck
        Install-ComposerDependencies
        Initialize-Junctions
        
        $directories = @(
            $global:SrmDir,
            (Join-Path $global:SrmDir "dashboard"),
            (Join-Path $global:SrmDir "dashboard\public"),
            (Join-Path $global:SrmDir "dashboard\config"),
            (Join-Path $global:SrmDir "dashboard\storage\logs"),
            (Join-Path $global:SrmDir "modules\SRM9\SRM\Reports9")
        )
        
        Set-DirectoryPermissions -Directories $directories
        Show-InstallationSummary
    }
    catch {
        Invoke-ErrorHandler -ErrorMessage $_.Exception.Message
    }
    finally {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
}

function Resume-AfterComposer {
    try {
        Initialize-Installation
        Initialize-PhpEnvironment
        Write-LogInfo "Resume all steps After Composer Dependency Installation"
                
        # Skip directly to symbolic links
        Initialize-Junctions
        
        $directories = @(
            $global:SrmDir,
            (Join-Path $global:SrmDir "dashboard"),
            (Join-Path $global:SrmDir "dashboard\public"),
            (Join-Path $global:SrmDir "dashboard\config"),
            (Join-Path $global:SrmDir "dashboard\storage\logs"),
            (Join-Path $global:SrmDir "modules\SRM9\SRM\Reports9")
        )
        
        Set-DirectoryPermissions -Directories $directories
        Show-InstallationSummary
    }
    catch {
        Invoke-ErrorHandler -ErrorMessage $_.Exception.Message
    }
    finally {
        Stop-Transcript -ErrorAction SilentlyContinue
    }
}


# ============================================================================
# SCRIPT ENTRY POINT
# ============================================================================
# Validate PowerShell version
if ($PSVersionTable.PSVersion.Major -lt 5) {
    Write-Error "This script requires PowerShell 5.1 or later. Current version: $($PSVersionTable.PSVersion)"
    exit 1
}

# Display header
Clear-Host
Write-Host "Smart Report Maker (SRM) - Windows Installation Script" -ForegroundColor Cyan
Write-Host "Version: 1.0" -ForegroundColor Gray
if ($ResumeAfterComposer) {
    Write-Host "Mode: Skip to Symbolic Links" -ForegroundColor Yellow
}
Write-Host ""

# Start installation based on flag
if ($ResumeAfterComposer) {
    Resume-AfterComposer
} else {
    Start-Installation
}