<?php
/**
 * Plugin Name: SRM Embed (Smart Report Maker)
 * Description: Embed Smart Report Maker resources with secure token management and shortcuts
 * Version: 1.9.0
 * Author: Webuccino
 * License: GPL v2 or later
 * Text Domain: srm-embed-smart-report-maker
 */

// ============================================================================
// SECURITY: Prevent direct access to this file
// ============================================================================
if (!defined('ABSPATH')) {
    exit;
}

// ============================================================================
// PLUGIN CONSTANTS AND CONFIGURATION
// ============================================================================
define('SRM_EMBED_VERSION', '1.9.0');
define('SRM_EMBED_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SRM_EMBED_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('SRM_EMBED_OPTION_KEY', 'srm_embed_options');
define('SRM_EMBED_SHORTCUTS_KEY', 'srm_embed_shortcuts');

// ============================================================================
// ENCRYPTION HELPER FUNCTIONS
// Derive a stable 32-byte binary secret from WP salts (no hardcoding)
// ============================================================================
function srm_embed_secret_key() {
    return hash('sha256', AUTH_KEY . SECURE_AUTH_SALT, true);
}

// Encrypt plaintext -> base64(JSON{m, n, c, t?})
// Preferred: sodium_crypto_secretbox() with random 24-byte nonce
// Fallback: openssl AES-256-GCM with 12-byte IV and 16-byte tag
function srm_embed_encrypt($plaintext) {
    if (function_exists('sodium_crypto_secretbox')) {
        $nonce = random_bytes(24);
        $ciphertext = sodium_crypto_secretbox($plaintext, $nonce, srm_embed_secret_key());
        $payload = [
            'm' => 'sodium',
            'n' => base64_encode($nonce),
            'c' => base64_encode($ciphertext),
            't' => ''
        ];
        return base64_encode(json_encode($payload));
    } elseif (function_exists('openssl_encrypt')) {
        $iv = random_bytes(12);
        $tag = '';
        $ciphertext = openssl_encrypt(
            $plaintext,
            'aes-256-gcm',
            srm_embed_secret_key(),
            OPENSSL_RAW_DATA,
            $iv,
            $tag
        );
        $payload = [
            'm' => 'aes-256-gcm',
            'n' => base64_encode($iv),
            'c' => base64_encode($ciphertext),
            't' => base64_encode($tag)
        ];
        return base64_encode(json_encode($payload));
    }
    return false;
}

// Decrypt base64(JSON{m, n, c, t?}) -> plaintext
function srm_embed_decrypt($encoded) {
    try {
        $payload = json_decode(base64_decode($encoded), true);
        if (!$payload || !isset($payload['m'], $payload['n'], $payload['c'])) {
            return false;
        }

        if ($payload['m'] === 'sodium' && function_exists('sodium_crypto_secretbox_open')) {
            $nonce = base64_decode($payload['n']);
            $ciphertext = base64_decode($payload['c']);
            return sodium_crypto_secretbox_open($ciphertext, $nonce, srm_embed_secret_key());
        } elseif ($payload['m'] === 'aes-256-gcm' && function_exists('openssl_decrypt')) {
            $iv = base64_decode($payload['n']);
            $ciphertext = base64_decode($payload['c']);
            $tag = base64_decode($payload['t']);
            return openssl_decrypt(
                $ciphertext,
                'aes-256-gcm',
                srm_embed_secret_key(),
                OPENSSL_RAW_DATA,
                $iv,
                $tag
            );
        }
    } catch (Exception $e) {
        return false;
    }
    return false;
}

// ============================================================================
// PLUGIN ACTIVATION AND DEACTIVATION HOOKS
// ============================================================================
register_activation_hook(__FILE__, 'srm_embed_activate');
register_uninstall_hook(__FILE__, 'srm_embed_uninstall');

function srm_embed_activate() {
    $default_options = [
        'srm_token_endpoint' => '',
        'srm_access_key' => ''
    ];
    
    if (!get_option(SRM_EMBED_OPTION_KEY)) {
        add_option(SRM_EMBED_OPTION_KEY, $default_options);
    }
    
    if (!get_option(SRM_EMBED_SHORTCUTS_KEY)) {
        add_option(SRM_EMBED_SHORTCUTS_KEY, []);
    }
}

function srm_embed_uninstall() {
    delete_option(SRM_EMBED_OPTION_KEY);
    delete_option(SRM_EMBED_SHORTCUTS_KEY);
}

// ============================================================================
// ADMIN MENU SETUP
// ============================================================================
add_action('admin_menu', 'srm_embed_admin_menu');

function srm_embed_admin_menu() {
    add_menu_page(
        'SRM Embed',
        'SRM Embed',
        'manage_options',
        'srm-embed',
        'srm_embed_main_page',
        'dashicons-embed-generic',
        30
    );
    
    add_submenu_page(
        'srm-embed',
        'General Settings',
        'General Settings',
        'manage_options',
        'srm-embed-settings',
        'srm_embed_settings_page'
    );
    
    add_submenu_page(
        'srm-embed',
        'Embed Shortcodes',
        'Embed Shortcodes',
        'manage_options',
        'srm-embed-shortcuts',
        'srm_embed_shortcuts_page'
    );
}

// ============================================================================
// MAIN ADMIN PAGE
// ============================================================================
function srm_embed_main_page() {
    ?>
    <div class="wrap">
        <h1>SRM Embed (Smart Report Maker)</h1>
        <div class="card">
			<h2>Welcome to SRM Embed.</h2>
			<p>This plugin allows you to embed Smart Report Maker reports, charts, dashboards, pivot tables, and KPIs into your WordPress site with token-based management.</p>
			<p>You can learn more about Smart Report Maker at: <a href="https://mysqlreports.com/srm-embed-smart-report-maker/" target="_blank">https://mysqlreports.com/srm-embed-smart-report-maker/</a></p>
            <p><strong>Quick Navigation:</strong></p>
            <ul>
                <li><a href="<?php echo esc_url(admin_url('admin.php?page=srm-embed-settings')); ?>">General Settings</a> - Configure your SRM API endpoint and access key</li>
                <li><a href="<?php echo esc_url(admin_url('admin.php?page=srm-embed-shortcuts')); ?>">Embed Shortcodes</a> - Create and manage embed shortcodes</li>
            </ul>
        </div>
    </div>
    <?php
}

// ============================================================================
// GENERAL SETTINGS PAGE
// ============================================================================
function srm_embed_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'srm-embed-smart-report-maker'));
    }

    // Check if crypto libraries are available
    $crypto_available = function_exists('sodium_crypto_secretbox') || function_exists('openssl_encrypt');
    
    if (isset($_POST['submit']) && check_admin_referer('srm_embed_settings_nonce')) {
        $options = get_option(SRM_EMBED_OPTION_KEY, []);
        
        if (isset($_POST['srm_token_endpoint'])) {
            $options['srm_token_endpoint'] = esc_url_raw(wp_unslash($_POST['srm_token_endpoint']));
        }
        
        if ($crypto_available && isset($_POST['srm_access_key']) && !empty($_POST['srm_access_key'])) {
            $access_key = sanitize_text_field(wp_unslash($_POST['srm_access_key']));
            $encrypted_key = srm_embed_encrypt($access_key);
            if ($encrypted_key) {
                $options['srm_access_key'] = $encrypted_key;
            }
        }
        
        update_option(SRM_EMBED_OPTION_KEY, $options);
        echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
    }
    
    $options = get_option(SRM_EMBED_OPTION_KEY, []);
    $current_key = $options['srm_access_key'] ?? '';
    $key_display = $current_key ? '••••••••••••••••' : '';
    
    ?>
    <div class="wrap">
        <h1>SRM Embed - General Settings</h1>
        
        <div class="card">
            <h2>Configuration Help</h2>
            <p><strong>Where to get these settings:</strong> You need to configure these settings in your Smart Report Maker (SRM) system's Embed Manager. The Embed Manager provides the API endpoint URL and access key needed to embed private resources.</p>
            <p><strong>Smart Report Maker:</strong> <a href="https://mysqlreports.com/srm-embed-smart-report-maker/" target="_blank">Learn more about SRM</a> - an advanced PHP-based report builder for MySQL with features like dynamic dashboards, pivot tables, charts, and KPI monitoring.</p>
        </div>
        
        <?php if (!$crypto_available): ?>
        <div class="notice notice-error">
            <p><strong>Warning:</strong> Required encryption libraries are not available. The access key cannot be saved securely.</p>
            <p>Please ensure either libsodium or OpenSSL is available on your server.</p>
        </div>
        <?php endif; ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('srm_embed_settings_nonce'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="srm_token_endpoint">Embed Token API URL</label>
                    </th>
                    <td>
                        <input type="url" id="srm_token_endpoint" name="srm_token_endpoint" 
                               value="<?php echo esc_attr($options['srm_token_endpoint'] ?? ''); ?>" 
                               class="regular-text" required />
                        <p class="description">The full URL to your SRM embed token API endpoint</p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="srm_access_key">Access Key</label>
                    </th>
                    <td>
                        <?php if ($current_key): ?>
                            <input type="text" id="srm_access_key" name="srm_access_key" 
                                   value="<?php echo esc_attr($key_display); ?>" 
                                   class="regular-text" placeholder="Enter new key to replace existing" />
                            <p class="description">Current key is encrypted and stored. Enter a new key above to replace it.</p>
                        <?php else: ?>
                            <input type="text" id="srm_access_key" name="srm_access_key" 
                                   class="regular-text" required />
                            <p class="description">Your SRM access key for private resources</p>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <input type="submit" name="submit" id="submit" class="button button-primary" 
                       value="Save Settings" <?php echo $crypto_available ? '' : 'disabled'; ?> />
            </p>
        </form>
    </div>
    <?php
}

// ============================================================================
// SHORTCUTS MANAGEMENT PAGE
// ============================================================================
function srm_embed_shortcuts_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'srm-embed-smart-report-maker'));
    }

    $shortcuts = get_option(SRM_EMBED_SHORTCUTS_KEY, []);
    
    // Handle form submissions
    if (isset($_POST['action'])) {
        $action = sanitize_text_field(wp_unslash($_POST['action']));
        
        if ($action === 'add_shortcut' && check_admin_referer('srm_embed_shortcut_nonce')) {
            if (!isset($_POST['resource_url'])) {
                echo '<div class="notice notice-error"><p><strong>Error:</strong> Resource URL is required.</p></div>';
            } else {
                $resource_url = esc_url_raw(wp_unslash($_POST['resource_url']));
                
                // Validate SRM resource URL contains at least one required term
                $requiredTerms = ['srm_modules', 'dashboards', 'pivot-tables', 'charts', 'metrics'];
                $hasRequiredTerm = false;
                foreach ($requiredTerms as $term) {
                    if (stripos($resource_url, $term) !== false) {
                        $hasRequiredTerm = true;
                        break;
                    }
                }
                
                // Validate that the URL includes mode=embed parameter
                $hasModeEmbed = stripos($resource_url, 'mode=embed') !== false;
                
                if (!$hasRequiredTerm) {
                    echo '<div class="notice notice-error"><p><strong>Error:</strong> The provided SRM resource URL is not valid. Please verify the link and try again.</p></div>';
                } elseif (!$hasModeEmbed) {
                    echo '<div class="notice notice-error"><p><strong>Error:</strong> Be sure to include a valid resource URL from the \'SRM WP Plugin\' section found under Embed Manager in your SRM account.</p></div>';
                } else {
                    $shortcut_name = isset($_POST['shortcut_name']) ? sanitize_text_field(wp_unslash($_POST['shortcut_name'])) : '';
                    $resource_type = isset($_POST['resource_type']) ? sanitize_text_field(wp_unslash($_POST['resource_type'])) : '';
                    $access_control = isset($_POST['access_control']) ? sanitize_text_field(wp_unslash($_POST['access_control'])) : '';
                    $iframe_width = isset($_POST['iframe_width']) ? sanitize_text_field(wp_unslash($_POST['iframe_width'])) : '100%';
                    $iframe_height = isset($_POST['iframe_height']) ? sanitize_text_field(wp_unslash($_POST['iframe_height'])) : '800px';
                    
                    $new_shortcut = [
                        'id' => uniqid('shortcut_'),
                        'name' => $shortcut_name,
                        'resource_url' => $resource_url,
                        'resource_type' => $resource_type,
                        'access_control' => $access_control,
                        'iframe_width' => $iframe_width,
                        'iframe_height' => $iframe_height,
                        'created_at' => current_time('mysql')
                    ];
                
                $shortcuts[] = $new_shortcut;
                update_option(SRM_EMBED_SHORTCUTS_KEY, $shortcuts);
                echo '<div class="notice notice-success"><p>Shortcode created successfully!</p></div>';
                echo '<div class="notice notice-info"><p>Copy and paste this shortcode into the WordPress page where you want to display the resource (report, chart, dashboard, or KPI).</p><p><span style="color:#d63638;">If the resource should only be available to specific members, make sure to place it on a private or a protected page that is accessible exclusively to those members.</span></p></div>';

                // Show the newly created shortcode with a copy button
                $sc_resource_url = esc_attr($new_shortcut['resource_url']);
                $sc_access       = esc_attr($new_shortcut['access_control']);
                $sc_width        = esc_attr($new_shortcut['iframe_width']);
                $sc_height       = esc_attr($new_shortcut['iframe_height']);
                $sc_type         = esc_attr($new_shortcut['resource_type']);
                $sc_text         = "[srm_embed resource_url=\"{$sc_resource_url}\" access_rule=\"{$sc_access}\" iframe_width=\"{$sc_width}\" iframe_height=\"{$sc_height}\" resource_type=\"{$sc_type}\"]";

                echo '<div class="notice notice-info"><p style="margin-bottom:8px;"><strong>New Shortcode:</strong></p>';
                echo '<div style="display:flex; align-items:center; gap:10px;">';
                echo '<code style="flex:1; word-break:break-all;">' . esc_html($sc_text) . '</code>';
                echo '<button type="button" class="button button-small" onclick="copyShortcode(this)" data-shortcode="' . esc_attr($sc_text) . '">Copy</button>';
                echo '</div></div>';
                }
            }
            
        } elseif ($action === 'delete_shortcut' && check_admin_referer('srm_embed_shortcut_nonce')) {
            if (!isset($_POST['shortcut_id'])) {
                echo '<div class="notice notice-error"><p><strong>Error:</strong> Shortcut ID is required.</p></div>';
            } else {
                $shortcut_id = sanitize_text_field(wp_unslash($_POST['shortcut_id']));
                $shortcuts = array_filter($shortcuts, function($s) use ($shortcut_id) {
                    return $s['id'] !== $shortcut_id;
                });
                update_option(SRM_EMBED_SHORTCUTS_KEY, array_values($shortcuts));
                echo '<div class="notice notice-success"><p>Shortcode deleted successfully!</p></div>';
            }
        }
        
        $shortcuts = get_option(SRM_EMBED_SHORTCUTS_KEY, []);
    }
    
    ?>
    <div class="wrap">
        <style>
        /* Increase the width of cards on the shortcodes page (WP default ~520px) */
        .wrap .card {
            max-width: 1560px;
            width: 100%;
        }
        .wrap .card .form-table,
        .wrap .card .wp-list-table {
            width: 100%;
        }
        .wrap .card code {
            white-space: pre-wrap;
        }
        </style>
        <h1>SRM ShortCodes</h1>
        
        <div class="card">
            <h2>Create New Shortcode</h2>
            <form method="post" action="">
                <?php wp_nonce_field('srm_embed_shortcut_nonce'); ?>
                <input type="hidden" name="action" value="add_shortcut" />
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="shortcut_name">Shortcode Name</label>
                        </th>
                        <td>
                            <input type="text" id="shortcut_name" name="shortcut_name" 
                                   class="regular-text" required />
                            <p class="description">A descriptive name for this shortcode</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="resource_url">Resource URL</label>
                        </th>
                        <td>
                            <input type="url" id="resource_url" name="resource_url" 
                                   class="regular-text" required />
                            <p class="description">The full URL to your SRM resource</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="resource_type">Resource Type</label>
                        </th>
                        <td>
                            <select id="resource_type" name="resource_type" required>
                                <option value="">Select Resource Type</option>
                                <option value="dashboard">Dashboard</option>
                                <option value="pivot-table">Pivot Table</option>
                                <option value="report">Report</option>
                                <option value="chart">Chart</option>
                                <option value="kpi-metrics">KPIs / Metrics</option>
                            </select>
                            <p class="description">The type of SRM resource</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="access_control">Resource Access Control</label>
                        </th>
                        <td>
                            <select id="access_control" name="access_control" required>
                                <option value="private">Private</option>
                                <option value="public">Public</option>
                            </select>
                            <p class="description">Access control for this resource</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="iframe_width">Iframe Width</label>
                        </th>
                        <td>
                            <input type="text" id="iframe_width" name="iframe_width" 
                                   value="100%" class="regular-text" required />
                            <p class="description">Width of the iframe (e.g., 100%, 800px, 50vw)</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="iframe_height">Iframe Height</label>
                        </th>
                        <td>
                            <input type="text" id="iframe_height" name="iframe_height" 
                                   value="800px" class="regular-text" required />
                            <p class="description">Height of the iframe (e.g., 800px, 100vh, 600px)</p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary" value="Create Shortcode" />
                </p>
            </form>
        </div>
        
        <div class="card">
            <h2>Existing Shortcodes</h2>
			<p><strong>How to use:</strong> Copy and paste this shortcode into the WordPress page where you want to display the resource (report, chart, dashboard, or KPI). <span style="color:#d63638;">If the resource should only be visible to specific members, be sure to place it on private pages that are accessible exclusively to those members.</span> </p>
            
            <?php if (empty($shortcuts)): ?>
                <p>No shortcodes created yet.</p>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Resource Type</th>
                            <th>Access Control</th>
                            <th>Dimensions</th>
                            <th>Shortcode</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_reverse($shortcuts) as $shortcut): ?>
                            <tr>
                                <td><?php echo esc_html($shortcut['name']); ?></td>
                                <td><?php echo esc_html($shortcut['resource_type']); ?></td>
                                <td><?php echo esc_html($shortcut['access_control']); ?></td>
                                <td><?php echo esc_html($shortcut['iframe_width']) . ' × ' . esc_html($shortcut['iframe_height']); ?></td>
                                <td>
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <code style="flex: 1; word-break: break-all;">[srm_embed resource_url="<?php echo esc_attr($shortcut['resource_url']); ?>" access_rule="<?php echo esc_attr($shortcut['access_control']); ?>" iframe_width="<?php echo esc_attr($shortcut['iframe_width']); ?>" iframe_height="<?php echo esc_attr($shortcut['iframe_height']); ?>" resource_type="<?php echo esc_attr($shortcut['resource_type']); ?>"]</code>
                                        <button type="button" class="button button-small" onclick="copyShortcode(this)" data-shortcode="[srm_embed resource_url=&quot;<?php echo esc_attr($shortcut['resource_url']); ?>&quot; access_rule=&quot;<?php echo esc_attr($shortcut['access_control']); ?>&quot; iframe_width=&quot;<?php echo esc_attr($shortcut['iframe_width']); ?>&quot; iframe_height=&quot;<?php echo esc_attr($shortcut['iframe_height']); ?>&quot; resource_type=&quot;<?php echo esc_attr($shortcut['resource_type']); ?>&quot;]">
                                            Copy
                                        </button>
                                    </div>
                                </td>
                                <td>
                                    <form method="post" action="" style="display:inline;">
                                        <?php wp_nonce_field('srm_embed_shortcut_nonce'); ?>
                                        <input type="hidden" name="action" value="delete_shortcut" />
                                        <input type="hidden" name="shortcut_id" value="<?php echo esc_attr($shortcut['id']); ?>" />
                                        <button type="submit" class="button button-small button-link-delete" 
                                                onclick="return confirm('Are you sure you want to delete this shortcode?')">
                                            Delete
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <script>
                function copyShortcode(button) {
                    const shortcode = button.getAttribute('data-shortcode');
                    navigator.clipboard.writeText(shortcode).then(function() {
                        const originalText = button.textContent;
                        button.textContent = 'Copied!';
                        button.style.backgroundColor = '#46b450';
                        button.style.color = 'white';
                        setTimeout(function() {
                            button.textContent = originalText;
                            button.style.backgroundColor = '';
                            button.style.color = '';
                        }, 2000);
                    }).catch(function(err) {
                        console.error('Could not copy text: ', err);
                        alert('Failed to copy shortcode. Please copy manually.');
                    });
                }
                </script>
            <?php endif; ?>
        </div>
    </div>
    <?php
}

// ============================================================================
// SHORTCODE REGISTRATION AND PROCESSING
// ============================================================================
add_shortcode('srm_embed', 'srm_embed_shortcode');

function srm_embed_shortcode($atts) {
    // ============================================================================
    // CONFIGURATION SECTION 
    // Get requested resource URL, type and access key from shortcode attributes
    // ============================================================================
    
    $atts = shortcode_atts([
        'resource_url' => '',
        'access_rule' => 'private',
        'iframe_width' => '100%',
        'iframe_height' => '800px',
        'resource_type' => ''
    ], $atts, 'srm_embed');
    
    $config = [
        'srm_resource_url' => $atts['resource_url'],
        'srm_access_type' => $atts['access_rule'],
        'srm_iframe_width' => $atts['iframe_width'],
        'srm_iframe_height' => $atts['iframe_height'],
        'srm_resource_type' => $atts['resource_type']
    ];
    
    // ============================================================================
    // CONFIGURATION VALIDATION
    // Validate the configuration such as resource URL, access type, iframe width and height
    // validate the needed extensions are enabled
    // ============================================================================
    
    if (!isset($config['srm_resource_url']) || trim((string)$config['srm_resource_url']) === '') {
        return "<div class='alert alert-danger' style='margin:1rem;'>
            <strong>Error:</strong> Resource <code>URL</code> cannot be empty.<br>
            Please set the <code>resource_url</code> attribute to your SRM resource (e.g., <code>https://reports.example.com/r/123</code>) and try again.
        </div>";
    }
    
    // Validate SRM resource URL contains at least one required term (generic error if not)
    $url = trim((string)$config['srm_resource_url']);
    $requiredTerms = ['srm_modules', 'dashboards', 'pivot-tables', 'charts', 'metrics'];
    
    $hasRequiredTerm = false;
    foreach ($requiredTerms as $term) {
        if (stripos($url, $term) !== false) {
            $hasRequiredTerm = true;
            break;
        }
    }
    
    if (!$hasRequiredTerm) {
        return "<div class='alert alert-danger' style='margin:1rem;'>
            <strong>Error:</strong> The provided SRM resource <code>URL</code> is not valid. Please verify the link and try again.
        </div>";
    }
    
    // Validate Access Key when access type is private
    $accessType = strtolower(trim((string)($config['srm_access_type'] ?? '')));
    $allowedAccessTypes = ['public', 'private'];
    
    if ($accessType === '' || !in_array($accessType, $allowedAccessTypes, true)) {
        return "<div class='alert alert-danger' style='margin:1rem;'>
            <strong>Error:</strong> Unrecognized access rule.<br>
            Please set <code>access_rule</code> to <code>public</code> or <code>private</code> and try again.
        </div>";
    }
    
    // Check if JSON extension is enabled
    // WordPress HTTP API (wp_remote_post) handles HTTP transport automatically
    // JSON is required for API communication
    $missing = [];
    
    // JSON check (extension + key function)
    if (!extension_loaded('json') || !function_exists('json_encode')) {
        $missing[] = ['name' => 'JSON', 'ini' => 'extension=json'];
    }
    
    if (!empty($missing)) {
        // Fail hard so execution doesn't continue
        // Build guidance list
        $items = '';
        foreach ($missing as $ext) {
            $items .= '<li><code>' . esc_html($ext['name']) . '</code> — enable in <code>php.ini</code> (e.g. add/uncomment <code>' . esc_html($ext['ini']) . '</code>)</li>';
        }
        
        return "<div class='alert alert-danger' style='margin:1rem;'>
            <strong>Error:</strong> Required PHP extension(s) are not available:
            <ul>{$items}</ul>
            After enabling, <strong>restart</strong> your web server (Apache/Nginx) or PHP-FPM.
        </div>";
    }
    
    // ============================================================================
    // PRIVATE RESOURCE: FETCH EMBED URL
    // Expect API to return a full URL to embed (including token if needed)
    // ============================================================================
    
    if ($config['srm_access_type'] == "private") {
        $options = get_option(SRM_EMBED_OPTION_KEY, []);
        
        if (empty($options['srm_token_endpoint']) || empty($options['srm_access_key'])) {
            return "<div class='alert alert-danger' style='margin:1rem;'>
                <strong>Error:</strong> SRM configuration is incomplete.<br>
                Please configure the Embed Token API URL and Access Key in the SRM Embed settings.
            </div>";
        }
        
        $config['srm_token_endpoint'] = $options['srm_token_endpoint'];
        $config['srm_access_key'] = srm_embed_decrypt($options['srm_access_key']);
        
        if ($config['srm_access_key'] === false) {
            return "<div class='alert alert-danger' style='margin:1rem;'>
                <strong>Error:</strong> Failed to decrypt access key.<br>
                Please check your SRM Embed configuration.
            </div>";
        }
        
        $post_data = json_encode([
            'resource_url' => $config['srm_resource_url']
        ]);
        
        // Call the embed API of SRM to obtain a full embed URL
        $response = wp_remote_post($config['srm_token_endpoint'], [
            'body' => $post_data,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $config['srm_access_key']
            ],
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            return "<div class='alert alert-danger'><strong>Embed Token Error:</strong><br>" . esc_html($response->get_error_message()) . "</div>";
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Expect a URL in the API response under 'token' key; validate it and use directly
        if ($http_code === 200) {
            $returned_url = null;
            $decoded = json_decode($response_body, true);

            if (is_array($decoded) && isset($decoded['token'])) {
                $returned_url = trim((string)$decoded['token']);
            } elseif (is_string($decoded)) {
                $returned_url = trim($decoded);
            } else {
                $returned_url = trim((string)$response_body);
            }

            if ($returned_url !== '' && filter_var($returned_url, FILTER_VALIDATE_URL)) {
                $config['srm_resource_url'] = $returned_url;
            } else {
                return "<div class='alert alert-danger'><strong>Embed Token Error:</strong><br>" . esc_html($response_body) . "</div>";
            }
        } else {
            return "<div class='alert alert-danger'><strong>Embed Token Error:</strong><br>" . esc_html($response_body) . "</div>";
        }
    }
    
    // ============================================================================
    // RENDER IFRAME
    // Display iframe with dimensions from shortcode attributes
    // ============================================================================
    
    $final_url = esc_url($config['srm_resource_url']);
    $w = esc_attr($config['srm_iframe_width']);
    $h = esc_attr($config['srm_iframe_height']);
    
    return "<iframe
        src=\"{$final_url}\"
        style=\"width: {$w}; height: {$h}; border: none;\"
        title=\"Smart Report Maker Resource\"
        allowfullscreen
        loading=\"eager\"
        referrerpolicy=\"origin\"></iframe>";
}

// ============================================================================
// ADMIN NOTICES FOR MISSING CRYPTO LIBRARIES
// ============================================================================
add_action('current_screen', 'srm_embed_check_crypto_libraries');

function srm_embed_check_crypto_libraries($screen) {
    // Only show notice on SRM Embed admin pages
    $srm_screens = ['toplevel_page_srm-embed', 'srm-embed_page_srm-embed-settings', 'srm-embed_page_srm-embed-shortcuts'];
    if (!in_array($screen->id, $srm_screens, true)) {
        return;
    }
    
    $crypto_available = function_exists('sodium_crypto_secretbox') || function_exists('openssl_encrypt');
    
    if (!$crypto_available) {
        add_action('admin_notices', 'srm_embed_admin_notices');
    }
}

function srm_embed_admin_notices() {
    echo '<div class="notice notice-warning">
        <p><strong>SRM Embed:</strong> Required encryption libraries are not available. Please ensure either libsodium or OpenSSL is available on your server for secure access key storage.</p>
    </div>';
}
